package com.digiwin.athena.atdm.datasource.datasource;

import com.digiwin.athena.appcore.exception.BusinessException;
import com.digiwin.athena.atdm.UiBotConstants;
import com.digiwin.athena.atdm.constant.ErrorCodeEnum;
import com.digiwin.athena.atdm.datasource.domain.ApiMetadataCollection;
import com.digiwin.athena.atdm.datasource.domain.ExecuteContext;
import com.digiwin.athena.atdm.datasource.domain.QueryAction;
import com.digiwin.athena.atdm.datasource.domain.QueryResult;
import com.digiwin.athena.atdm.datasource.dto.PageInfo;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import lombok.Data;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.BooleanUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeoutException;
import java.util.stream.Collectors;

/**
 * 多个数据源的union合并，多个数据源单独查询，返回的结构要一致，会把所有的查询结果累加到一个集合
 */
@Data
public class UnionDataSource extends DataSourceBase {

    private static Logger logger = LoggerFactory.getLogger(UnionDataSource.class);
    /**
     * 指定可以对union类型的数据做合并查询
     */
    private Boolean mergeQuery;

    /**
     * 查询的数据源
     */
    private List<DataSourceBase> unionItems;

    @Override
    public QueryAction getAction() {
        if (CollectionUtils.isEmpty(unionItems)) {
            return null;
        } else {
            return unionItems.get(0).getAction();
        }
    }

    @Override
    public String getActionId() {
        if (getAction() != null) {
            return getAction().getActionId();
        }
        return "";
    }


    public UnionDataSource() {
        super();
        this.setType(DataSourceConstants.ACTION_CATEGORY_MIX_UNION);
    }

    /**
     * 查询数据
     *
     * @param executeContext 执行上下文
     * @param parameter      参数
     * @return
     */
    @Override
    protected QueryResult queryCore(ExecuteContext executeContext, Map<String, Object> parameter, PageInfo pageInfo, List<Map> sortInfo, List<Map> searchInfo) {

        if (CollectionUtils.isEmpty(unionItems)) {
            return QueryResult.empty(this.getName());
        }
        if (this.mergeQuery != null && this.mergeQuery) {
            mergeQuery();
        }
        QueryResult result = unionItems.get(0).query(executeContext, parameter, pageInfo, sortInfo, searchInfo);
        result.setDataSourceName(this.getName());
        if (unionItems.size() >= 2) {
            if (unionItems.size() == 2) {
                QueryResult nextResult = unionItems.get(1).query(executeContext, parameter, pageInfo, sortInfo, searchInfo);
                if (CollectionUtils.isNotEmpty(nextResult.getData())) {
                    result.appendData(nextResult);
                }
            } else {
                List<QueryResult> results = null;
                try {
                    results = ParallelQueryTaskUtils.query(executeContext, parameter, unionItems.subList(1, unionItems.size()), pageInfo, sortInfo, searchInfo);
                } catch (ExecutionException | TimeoutException | InterruptedException e) {//NOSONAR
                    if (e.getCause() instanceof BusinessException) {
                        throw (BusinessException) e.getCause();
                    }
                    logger.error("多线程处理出问题：" + e.getMessage());
                    throw BusinessException.create(ErrorCodeEnum.NUM_500_0045.getErrCode(),"多线程处理出问题", e);
                }
                for (QueryResult queryResult : results) {
                    if (CollectionUtils.isNotEmpty(queryResult.getData())) {
                        result.appendData(queryResult);
                    }
                }
            }
        }

        return result;
    }

    /**
     * 查询总行数
     *
     * @param executeContext 执行上下文
     * @param parameter      参数
     * @return
     */
    @Override
    @Deprecated
    public int size(ExecuteContext executeContext, Map<String, Object> parameter, String sizeType) {
        //直接做合并，减少api调用次数
        if (this.unionItems != null && this.unionItems.size() > 0) {
            if (UiBotConstants.ACTION_CATEGORY_ESP.equals(this.unionItems.get(0).getType())) {
                mergeQuery();
            }
        }
        return super.size(executeContext, parameter, sizeType);
    }

    @Override
    public int size(DataQuery dataQuery) {
        if (CollectionUtils.isNotEmpty(this.unionItems)
                && UiBotConstants.ACTION_CATEGORY_ESP.equals(this.unionItems.get(0).getType())) {
                mergeQuery();
        }
        return super.size(dataQuery);
    }

    private void mergeQuery() {
        if (this.unionItems != null && this.unionItems.size() > 1) {
            //先按照acitonid做分组
            Map<String, List<DataSourceBase>> groupDs = this.unionItems
                    .stream()
                    // BUG#159786 合并查询去除queryData为false的
                    .filter(e -> {
                            if (Objects.isNull(e.getAction())) {
                                return true;
                            }
                            return BooleanUtils.isNotFalse(e.getAction().getQueryData());
                    })
                    .collect(Collectors.groupingBy(e -> StringUtils.isNotBlank(e.getActionId()) ? e.getActionId() : "UIBOT_NULL_ACTION_ID"));
            this.unionItems.clear();
            for (List<DataSourceBase> dataSourceBaseList : groupDs.values()) {

                if (dataSourceBaseList.size() == 1) {
                    this.unionItems.add(dataSourceBaseList.get(0));
                    continue;
                }

                // 有数据源的action.businessUnit不相同，则不能合并
                if (!businessUnitIsSame(dataSourceBaseList)) {
                    this.unionItems.addAll(dataSourceBaseList);
                    continue;
                }

                //把所有的参数都合并到第一个
                DataSourceBase first = dataSourceBaseList.get(0);
                if (first.getAction() != null) {
                    if (first instanceof RawDataSource){
                        merge(dataSourceBaseList);

                    }
                    Map firstParas = first.getAction().getParas();
                    //如果没有参数，则认为是全部查询，不合并
                    if (firstParas == null) {
                        this.unionItems.add(first);
                        continue;
                    }
                    for (Object key : firstParas.keySet()) {
                        Object firstParaValue = firstParas.get(key);
                        if (firstParaValue instanceof Collection) {
                            Collection firstParaValueCollection = (Collection) firstParaValue;
                            for (int i = 1; i < dataSourceBaseList.size(); i++) {
                                DataSourceBase dataSourceBase = dataSourceBaseList.get(i);
                                if (dataSourceBase.getAction() != null && MapUtils.isNotEmpty(dataSourceBase.getAction().getParas())) {
                                    Map paras = dataSourceBase.getAction().getParas();
                                    if (paras.containsKey(key)) {
                                        firstParaValueCollection.addAll((Collection) paras.get(key));
                                    }
                                    if (dataSourceBase.getLimit() != null) {
                                        if (first.getLimit() == null) first.setLimit(0);
                                        first.setLimit(first.getLimit() + dataSourceBase.getLimit());
                                    }
                                }
                            }
                        }
                    }
                    this.unionItems.add(first);
                }
            }

        }
    }

    private void merge(List<DataSourceBase> dataSourceBaseList) {
        DataSourceBase first = dataSourceBaseList.get(0);
        int hasRawDataIndex = 0;
        HashMap<String, Object> rawData = null;
        for (int i = 0; i < dataSourceBaseList.size(); i++) {
            if (dataSourceBaseList.get(i) instanceof RawDataSource) {
                HashMap<String, Object> hasData = ((RawDataSource) dataSourceBaseList.get(i)).getRawData();
                if (hasData != null) {
                    hasRawDataIndex = i;
                    rawData = hasData;
                    break;
                }
            }
        }
        if (rawData == null){
            return;
        }
        ((RawDataSource) first).setRawData(rawData);
        for (Object key : rawData.keySet()) {
            Object firstParaValue = rawData.get(key);
            if (firstParaValue instanceof Collection) {
                Collection firstParaValueCollection = (Collection) firstParaValue;
                for (int i = 0; i < dataSourceBaseList.size(); i++) {
                    if (i == 0 || i == hasRawDataIndex) {
                        continue;
                    }
                    DataSourceBase dataSourceBase = dataSourceBaseList.get(i);
                    if (dataSourceBase instanceof RawDataSource) {
                        if (MapUtils.isNotEmpty(((RawDataSource) dataSourceBase).getRawData())) {
                            Map paras = ((RawDataSource) dataSourceBase).getRawData();
                            if (paras.containsKey(key)) {
                                firstParaValueCollection.addAll((Collection) paras.get(key));
                            }
                            if (dataSourceBase.getLimit() != null) {
                                if (first.getLimit() == null) {
                                    first.setLimit(0);
                                }
                                first.setLimit(first.getLimit() + dataSourceBase.getLimit());
                            }
                        }
                    }

                }
            }
        }
    }

    private boolean businessUnitIsSame(List<DataSourceBase> dataSourceBaseList) {
        if (CollectionUtils.isEmpty(dataSourceBaseList) || dataSourceBaseList.size() == 1) {
            return true;
        }

        Map<String, Object> businessUnit = null;
        for (DataSourceBase dataSource : dataSourceBaseList) {
            if (null == dataSource || null == dataSource.getAction() || null == dataSource.getAction().getBusinessUnit()) {
                continue;
            }

            if (null == businessUnit) {
                businessUnit = dataSource.getAction().getBusinessUnit();
                continue;
            }

            Map<String, Object> tmpBusinessUnit = dataSource.getAction().getBusinessUnit();
            if (businessUnit.size() != tmpBusinessUnit.size()) {
                return false;
            }

            // 比较businessUnit、tmpBusinessUnit是否相同
            if(!compare(businessUnit, tmpBusinessUnit)){
                return false;
            }
//            Set<Map.Entry<String, String>> entries = businessUnit.entrySet();
//            for (Map.Entry<String, String> entry : entries) {
//                /**
//                 * StringUtils.equals(null, null)   = true
//                 * StringUtils.equals(null, "abc")  = false
//                 * StringUtils.equals("abc", null)  = false
//                 * StringUtils.equals("abc", "abc") = true
//                 * StringUtils.equals("abc", "ABC") = false
//                 */
//                if (!StringUtils.equals(entry.getValue(), tmpBusinessUnit.get(entry.getKey()))) {
//                    return false;
//                }
//            }
        }
        return true;
    }


    /**
     * 附加数据的元数据
     *
     * @param executeContext 执行上下文
     * @param parameter      参数
     * @param queryResult    查询结果
     * @return
     */
    @Override
    protected ApiMetadataCollection queryMetaDataCore(ExecuteContext executeContext,
                                                      Map<String, Object> parameter,
                                                      QueryResult queryResult) {
        if (queryResult == null || CollectionUtils.isEmpty(unionItems)) {
            return null;
        }
        return unionItems.get(0).queryMetaData(executeContext, parameter, queryResult);
    }

    @Override
    protected DataSourceBase copyWithoutProcessorCore() {
        UnionDataSource unionDataSource = new UnionDataSource();
        unionDataSource.setName(this.getName());
        unionDataSource.setAction(this.getAction());
        unionDataSource.setType(this.getType());
        unionDataSource.setDataKeys(this.getDataKeys());
        unionDataSource.setActionId(this.getActionId());
        if (CollectionUtils.isNotEmpty(unionItems)) {
            List<DataSourceBase> dataSourceBases = new ArrayList<>();
            for (DataSourceBase dataSourceBase : this.unionItems) {
                dataSourceBases.add(dataSourceBase.copyWithoutProcessorCore());
            }
            unionDataSource.setUnionItems(dataSourceBases);
        }
        return unionDataSource;
    }

    public static boolean compare(Map<String, Object> map1, Map<String, Object> map2) {
        ObjectMapper mapper = new ObjectMapper();
        // 保证 key 顺序一致
        mapper.configure(SerializationFeature.ORDER_MAP_ENTRIES_BY_KEYS, true);
        // 转成 JSON 树节点进行结构比较
        JsonNode node1 = mapper.valueToTree(map1);
        JsonNode node2 = mapper.valueToTree(map2);
        return node1.equals(node2);
    }
}
