package com.digiwin.athena.atdm.datasource.datasource;

import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.appcore.util.SpringUtil;
import com.digiwin.athena.atdm.UiBotConstants;
import com.digiwin.athena.atdm.thememap.CommonMetadataService;
import com.digiwin.athena.atdm.retrieveData.CommonRetrieveDataService;
import com.digiwin.athena.atdm.terminateData.CommonTerminateDataService;
import com.digiwin.athena.atdm.datasource.domain.ApiMetadata;
import com.digiwin.athena.atdm.datasource.domain.ApiMetadataCollection;
import com.digiwin.athena.atdm.datasource.domain.ExecuteContext;
import com.digiwin.athena.atdm.datasource.domain.MetadataField;
import com.digiwin.athena.atdm.datasource.domain.QueryResult;
import com.digiwin.athena.atdm.datasource.domain.TagConstant;
import com.digiwin.athena.atdm.datasource.domain.TagDefinition;
import com.digiwin.athena.atdm.datasource.domain.ThemeMapTag;
import com.digiwin.athena.atdm.datasource.dto.PageInfo;
import com.digiwin.athena.atdm.datasource.datasource.process.CustomizeTagUtils;
import lombok.Data;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Optional;

/**
 * “已搁置”页签数据源（查询终止、撤回的数据）
 *
 * @author majianfu
 * @date 2022/12/13
 * @since
 */
@Data
public class ShelvedDataSource extends DataSourceBase {
    public ShelvedDataSource() {
        super();
        this.setType(UiBotConstants.ACTION_CATEGORY_SHELVED_DATA);
    }

    @Override
    protected QueryResult queryCore(ExecuteContext executeContext, Map<String, Object> parameter, PageInfo pageInfo, List<Map> sortInfo, List<Map> searchInfo) {
        List<String> workItemIds = new ArrayList<>();
        String loginUserId = null;
        if (null != this.getAction()) {
            workItemIds = Optional.ofNullable(this.getAction().getParas())
                    .map(paras -> (List<String>) paras.get("allWorkItemId"))
                    .orElse(Collections.emptyList());

            loginUserId = Optional.ofNullable(this.getAction().getParas())
                    .map(paras -> (String) paras.get("uibot_login_userId"))
                    .orElse("");
        } else {
            workItemIds = Optional.ofNullable(this.getExtendedData())
                    .map(extendedData -> (List<String>) extendedData.get("allWorkItemId"))
                    .orElse(Collections.emptyList());

            loginUserId = Optional.ofNullable(this.getExtendedData())
                    .map(extendedData -> (String) extendedData.get("uibot_login_userId"))
                    .orElse("");
        }

        if (CollectionUtils.isEmpty(workItemIds)) {
            return QueryResult.empty(this.getName());
        }

        AuthoredUser authoredUser = null != AppAuthContextHolder.getContext().getProxyAuthoredUser() ?
                AppAuthContextHolder.getContext().getProxyAuthoredUser() : AppAuthContextHolder.getContext().getAuthoredUser();
        String userId = StringUtils.isNotBlank(loginUserId) ? loginUserId : authoredUser.getUserId();
        List dataList = queryShelvedDataList(authoredUser, userId, executeContext.getTmActivityId(), workItemIds);
        return CollectionUtils.isNotEmpty(dataList) ? QueryResult.withData(this.getName(), dataList) : QueryResult.empty(this.getName());
    }

    private List queryShelvedDataList(AuthoredUser authoredUser, String userId, String tmActivityId, List<String> workItemIdList) {
        // 获取已终止数据
        List terminateDataList = SpringUtil.getBean(CommonTerminateDataService.class).queryTerminateDataList(authoredUser.getToken(), authoredUser.getTenantId(), userId, tmActivityId, workItemIdList);
        // 获取已撤回数据
        List withdrawDataList = SpringUtil.getBean(CommonRetrieveDataService.class).queryRetrieveDataList(authoredUser, tmActivityId, workItemIdList);

        List resultList = new ArrayList();
        if (CollectionUtils.isNotEmpty(terminateDataList)) {
            resultList.addAll(terminateDataList);
        }

        if (CollectionUtils.isNotEmpty(withdrawDataList)) {
            String dateFieldName = UiBotConstants.FieldName.UIBOT_DATETIME;
            withdrawDataList.sort((data1, data2) -> {
                Map<String, Object> map1 = (Map<String, Object>) data1;
                Map<String, Object> map2 = (Map<String, Object>) data1;

                if (!map1.containsKey(dateFieldName) || null == map1.get(dateFieldName)) {
                    return -1;
                }

                if (!map2.containsKey(dateFieldName) || null == map2.get(dateFieldName)) {
                    return 1;
                }

                LocalDateTime dateTime1 = (LocalDateTime) map1.get(dateFieldName);
                LocalDateTime dateTime2 = (LocalDateTime) map2.get(dateFieldName);
                return Boolean.compare(dateTime1.isBefore(dateTime2), true);
            });

            resultList.addAll(withdrawDataList);
        }
        return resultList;
    }

    @Override
    protected ApiMetadataCollection queryMetaDataCore(ExecuteContext executeContext, Map<String, Object> parameter, QueryResult queryResult) {
        ApiMetadataCollection apiMetadataCollection = new ApiMetadataCollection();

        // 从ESP查询API元数据
        String actionId = null != this.getAction() ? this.getAction().getActionId() : this.getActionId();
        ApiMetadata apiMetadata = SpringUtil.getBean(CommonMetadataService.class).getMetadata(executeContext.getLocale(), actionId);
        this.appendCustomFields(apiMetadata);
        apiMetadataCollection.setMasterApiMetadata(apiMetadata);

        return apiMetadataCollection;
    }

    private void appendCustomFields(ApiMetadata apiMetadata) {
        if (null == apiMetadata || CollectionUtils.isEmpty(apiMetadata.getResponseFields())) {
            return;
        }

        MetadataField topField = apiMetadata.getResponseFields().get(0);
        // 判断下，若topField.getSubFields()包含了 uibot__operationType 字段，则无需再重复添加
        if (skipAppendCustomField(topField)) {
            return;
        }

        MessageUtils messageUtils = SpringUtil.getBean(MessageUtils.class);

        List<MetadataField> allRespFields = new LinkedList<>();
        // 将操作类型，操作人，操作日期，任务名称 四列放在最前面
        allRespFields.add(createCustomField("uibot__operationType", messageUtils.getMessage("shelved_uibot__operation_type")));
        allRespFields.add(createCustomField("uibot__userName", messageUtils.getMessage("shelved_uibot__operator")));
        allRespFields.add(createCustomField("uibot__dateTime", messageUtils.getMessage("shelved_uibot__operation_dateTime")));
        //添加操作原因列
        allRespFields.add(createCustomField("uibot__operationReason", messageUtils.getMessage("shelved_uibot__operation_reason")));
        allRespFields.add(createTaskNameField());

        allRespFields.addAll(topField.getSubFields());
        topField.setSubFields(allRespFields);
    }

    private boolean skipAppendCustomField(MetadataField topField) {
        if (CollectionUtils.isEmpty(topField.getSubFields())) {
            return true;
        }

        Optional<MetadataField> optional = topField.getSubFields().stream()
                .filter(subField -> StringUtils.equals("uibot__operationType", subField.getName()))
                .findFirst();
        return optional.isPresent();
    }

    private MetadataField createTaskNameField() {
        MetadataField field = new MetadataField();
        field.setName("uibot__initiateActivityName");

        MessageUtils messageUtils = SpringUtil.getBean(MessageUtils.class);
        field.setDescription(messageUtils.getMessage("shelved_uibot__taskName"));
        field.setDataKey("false");
        field.setDataType("string");

        List<TagDefinition> tagDefs = new ArrayList<>();
        field.setTagDefinitions(tagDefs);

        TagDefinition tagDefinition = new TagDefinition();
        tagDefinition.setName("数据组件");
        tagDefinition.setDescription("数据组件");
        tagDefinition.setCategory(TagConstant.CATEGORY_DATATYPE);
        tagDefinition.setCustomize(false);
        tagDefinition.setTmTagCode(TagConstant.TYPE_STRING_TAG);
        tagDefinition.setCode(TagConstant.TYPE_STRING_TAG);
        tagDefinition.setInterpreterServiceName("navigateToTaskInterpreter");
        tagDefs.add(tagDefinition);

        ThemeMapTag themeMapTag = new ThemeMapTag();
        themeMapTag.setName("xxx");
        themeMapTag.setCategory(TagConstant.ORDER_TAG);
        themeMapTag.setValue("1");
        themeMapTag.setCode(TagConstant.ORDER_PREFIX + "1");
        themeMapTag.setUiBotCode(TagConstant.ORDER_TAG);

        tagDefinition.setThemeMapTag(themeMapTag);

        TagDefinition orderTagDef = CustomizeTagUtils.createOrderTagDefinitions("1").get(0);
        orderTagDef.setThemeMapTag(themeMapTag);
        tagDefs.add(orderTagDef);

        return field;
    }

    private MetadataField createCustomField(String name, String description) {
        MetadataField field = new MetadataField();
        field.setName(name);
        field.setDescription(description);
        field.setDataKey("false");
        field.setDataType("string");

        field.setTagDefinitions(createTagDefs(field));

        return field;
    }

    private List<TagDefinition> createTagDefs(MetadataField field) {
        List<TagDefinition> tagDefs = new ArrayList<>();
        TagDefinition typeStringTagDef = CustomizeTagUtils.createDataTypeDefinition(field);
        tagDefs.add(typeStringTagDef);
        ThemeMapTag themeMapTag = new ThemeMapTag();
        themeMapTag.setName("xxx");
        themeMapTag.setCategory(TagConstant.ORDER_TAG);
        themeMapTag.setValue("1");
        themeMapTag.setCode(TagConstant.ORDER_PREFIX + "1");
        themeMapTag.setUiBotCode(TagConstant.ORDER_TAG);
        typeStringTagDef.setThemeMapTag(themeMapTag);


        TagDefinition orderTagDef = CustomizeTagUtils.createOrderTagDefinitions("1").get(0);
        tagDefs.add(orderTagDef);
        orderTagDef.setThemeMapTag(themeMapTag);
        return tagDefs;
    }

    @Override
    protected DataSourceBase copyWithoutProcessorCore() {
        ShelvedDataSource dataSource = new ShelvedDataSource();
        dataSource.setName(this.getName());
        dataSource.setAction(this.getAction());
        dataSource.setType(this.getType());
        dataSource.setDataKeys(this.getDataKeys());
        dataSource.setActionId(this.getActionId());
        return dataSource;
    }
}
