package com.digiwin.athena.atdm.datasource.datasource;

import com.digiwin.athena.atdm.UiBotConstants;
import com.digiwin.athena.atdm.datasource.domain.*;
import com.digiwin.athena.atdm.datasource.dto.PageInfo;
import lombok.Data;
import org.apache.commons.collections4.CollectionUtils;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 先查询左边的数据源，然后join右边的数据源，最后把结果合并到左边的数据源
 * 右边的数据源是根据左边的数据源查询结果提取出来的key作为条件去查询。
 */
@Data
public class LeftJoinDataSource extends  DataSourceBase {

    private DataSourceBase left;

    private Boolean override = false;

    /**
     * 查询的数据源
     */
    private List<DataSourceBase> rightList;


    @Override
    public QueryAction getAction() {
        if (left ==null){
            return null;
        }
        return left.getAction();
    }

    @Override
    public String getActionId() {
        if (getAction()!= null){
            return getAction().getActionId();
        }
        return "";
    }

    public LeftJoinDataSource(){
        super();
        this.setType(DataSourceConstants.ACTION_CATEGORY_MIX_LEFT_JOIN);
    }


    /**
     * 查询数据
     *
     * @param executeContext 执行上下文
     * @param parameter      参数
     * @return
     */
    @Override
    protected QueryResult  queryCore(ExecuteContext executeContext, Map<String, Object> parameter, PageInfo pageInfo,List<Map> sortInfo,List<Map> searchInfo) {
        if (left == null) {
            return QueryResult.empty(this.getName());
        }
        QueryResult mainResult = left.query(executeContext, parameter,pageInfo,sortInfo,searchInfo);
        mainResult.setDataSourceName(this.getName());
        if (mainResult.getData().size() == 0) {
            return mainResult;
        }
        if (CollectionUtils.isEmpty(rightList)) {
            return mainResult;
        }
        if (CollectionUtils.isEmpty(left.getDataKeys())) {
            return mainResult;
        }

        for (DataSourceBase dataSourceBase : rightList) {
            QueryResult result = dataSourceBase.query(executeContext, parameter,pageInfo,sortInfo,searchInfo);
            if (result.size() == 0) {
                continue;
            }
            //左关联是比较左边和右边的key，如果相等，则右边的数据出现在结果中，并且如果有多行命中，结果也会是多行；如果没有相等的，则只显示left的列
            //1、先对right根据left的key做分组，复杂度为O(n)
            HashMap<String, List<Map<String, Object>>> rightGroup = new HashMap<>();
            StringBuilder keyBuilder = new StringBuilder();
            for (Map<String, Object> datum : result.getData()) {
                keyBuilder.setLength(0);
                for (String key : left.getDataKeys()) {
                    if (datum.containsKey(key)) {
                        keyBuilder.append(datum.get(key) == null ? "NAN" : datum.get(key).toString()).append(";");
                    }
                }
                if (keyBuilder.length() == 0) {
                    continue;
                }
                String key = keyBuilder.toString();
                if (!rightGroup.containsKey(key)) {
                    List<Map<String, Object>> dataRows = new ArrayList<>();
                    dataRows.add(datum);
                    rightGroup.put(key, dataRows);
                } else {
                    rightGroup.get(key).add(datum);
                }
            }
            //2、遍历left的数据key去分组结果中获取匹配的集合，复杂度O(m*z)
            List<Map<String, Object>> newResult = new ArrayList<>();
            for (Map<String, Object> datum : mainResult.getData()) {
                String mainKey = datum.getOrDefault(UiBotConstants.DATA_SOURCE_DATA_KEY, "").toString();
                if (mainKey.length() > 0 && rightGroup.containsKey(mainKey)) {
                    for (Map<String, Object> stringObjectMap : rightGroup.get(mainKey)) {
                        //合并数据，如果override=false以主数据的值为基准，否则如果存在相同的字段，以右边的为准
                        for (Map.Entry<String, Object> stringObjectEntry : datum.entrySet()) {
                            if(override) {
                                if (!stringObjectMap.containsKey(stringObjectEntry.getKey())) {
                                    stringObjectMap.put(stringObjectEntry.getKey(), stringObjectEntry.getValue());
                                }
                            }else {
                                stringObjectMap.put(stringObjectEntry.getKey(), stringObjectEntry.getValue());
                            }
                        }
                        newResult.add(stringObjectMap);
                    }
                } else {
                    newResult.add(datum);
                }
            }
            mainResult.resetData(newResult);
        }
        mainResult.setDataKeys(left.getDataKeys());
        mainResult.initializeDataKey();
        return mainResult;
    }

    /**
     * 附加数据的元数据
     *
     * @param executeContext 执行上下文
     * @param parameter      参数
     * @param queryResult    查询结果
     * @return
     */
    @Override
    protected ApiMetadataCollection queryMetaDataCore(ExecuteContext executeContext,  Map<String, Object> parameter, QueryResult queryResult) {
        if (queryResult.getData().size() ==0 ){
            return null;
        }
        ApiMetadataCollection apiMetadataCollection =  left.queryMetaData(executeContext,  parameter,queryResult);
        if (CollectionUtils.isEmpty(rightList)){
            return apiMetadataCollection;
        }
        for (DataSourceBase dataSourceBase : rightList) {
            ApiMetadataCollection metadataCollection =  dataSourceBase.queryMetaData(executeContext, parameter,queryResult);
            if (metadataCollection!=null && metadataCollection.getMasterApiMetadata() != null && metadataCollection.getMasterApiMetadata().getResponseFields().size() >0) {
                apiMetadataCollection.getMasterApiMetadata().addResponseFields(metadataCollection.getMasterApiMetadata().getResponseFields().get(0).getSubFields());
            }
        }
        return apiMetadataCollection;
    }

    @Override
    protected DataSourceBase copyWithoutProcessorCore() {
        LeftJoinDataSource mergeDataSource = new LeftJoinDataSource();
        mergeDataSource.setName(this.getName());
        mergeDataSource.setAction(this.getAction());
        mergeDataSource.setType(this.getType());
        mergeDataSource.setDataKeys(this.getDataKeys());
        mergeDataSource.setActionId(this.getActionId());
        if (this.left != null) {
            mergeDataSource.setLeft(this.left.copyWithoutProcessor());
        }
        if (CollectionUtils.isNotEmpty(rightList)) {
            List<DataSourceBase> dataSourceBases = new ArrayList<>();
            for (DataSourceBase dataSourceBase : this.rightList) {
                dataSourceBases.add( dataSourceBase.copyWithoutProcessorCore());
            }
            mergeDataSource.setRightList(dataSourceBases);
        }

        return mergeDataSource;
    }
}
