package com.digiwin.athena.atdm.datasource.datasource;

import com.digiwin.athena.appcore.exception.BusinessException;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.appcore.util.SpringUtil;
import com.digiwin.athena.atdm.UiBotConstants;
import com.digiwin.athena.atdm.constant.ErrorCodeEnum;
import com.digiwin.athena.atdm.datasource.domain.*;
import com.digiwin.athena.atdm.datasource.dto.PageInfo;
import lombok.Data;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.util.StringUtils;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * 多个数据源的合并，多个数据源单独查询，且返回值必须含有相同的dataKey，当一行的dataKey字段对应的值相等时，合并到一行上。
 * 合并的数据需要能独立查询，不存在有依赖关系
 */
@Data
public class MergeDataSource extends  DataSourceBase {

    private Boolean override;


    private DataSourceBase left;

    /**
     * 查询的数据源
     */
    private List<DataSourceBase> rightList;


    @Override
    public QueryAction getAction() {
        if (left ==null){
            return null;
        }
        return left.getAction();
    }

    @Override
    public String getActionId() {
        if (getAction()!= null){
            return getAction().getActionId();
        }
        return "";
    }

    public MergeDataSource(){
        super();
        this.setType(DataSourceConstants.ACTION_CATEGORY_MIX_MERGE);
    }


    /**
     * 查询数据
     *
     * @param executeContext 执行上下文
     * @param parameter      参数
     * @return
     */
    @Override
    protected QueryResult queryCore(ExecuteContext executeContext, Map<String, Object> parameter, PageInfo pageInfo,List<Map> sortInfo,List<Map> searchInfo) {
        if (left == null) {
            return QueryResult.empty(this.getName());
        }
        QueryResult mainResult = left.query(executeContext, parameter,pageInfo,sortInfo,searchInfo);
        mainResult.setDataSourceName(this.getName());
        if (mainResult.getData().size() == 0) {
            return mainResult;
        }
        if (CollectionUtils.isEmpty(rightList)) {
            return mainResult;
        }
        for (DataSourceBase dataSourceBase : rightList) {
            QueryResult result = dataSourceBase.query(executeContext, parameter,pageInfo,sortInfo,searchInfo);
            if (result.size() == 0) {
                continue;
            }
            //如果合并的数据源有dataKey，则通过这个key去主数据源中找数据
            if (CollectionUtils.isNotEmpty(dataSourceBase.getDataKeys())
                    && CollectionUtils.isNotEmpty(left.getDataKeys())) {
                //如果datakey一致，则通过索引合并
                if (StringUtils.collectionToCommaDelimitedString(dataSourceBase.getDataKeys()).equals(
                        StringUtils.collectionToCommaDelimitedString(left.getDataKeys()))) {
                    for (Map<String, Object> datum : mainResult.getData()) {
                        String key = datum.getOrDefault(UiBotConstants.DATA_SOURCE_DATA_KEY, "").toString();
                        Map<String, Object> findItem = result.findByKey(key);
                        if (findItem != null) {
                            //如果找到key相同，则直接合并,如果属性相同，则忽略
                            for (Map.Entry<String, Object> stringObjectEntry : findItem.entrySet()) {
                                if (!datum.containsKey(stringObjectEntry.getKey())) {
                                    datum.put(stringObjectEntry.getKey(), stringObjectEntry.getValue());
                                }
                            }
                        }
                    }
                } else {
                    StringBuilder stringBuilder = new StringBuilder();
                    //如果不一致，则需要遍历
                    for (Map<String, Object> datum : mainResult.getData()) {
                        stringBuilder.setLength(0);
                        for (String key : dataSourceBase.getDataKeys()) {
                            if (datum.containsKey(key)) {
                                stringBuilder.append(datum.get(key) == null ? "NAN" : datum.get(key).toString()).append(";");
                            }
                        }
                        if (stringBuilder.length() > 0) {
                            String key = stringBuilder.toString();
                            Map<String, Object> findItem = result.findByKey(key);
                            if (findItem != null) {
                                //如果找到key相同，则直接合并,如果属性相同，则忽略
                                for (Map.Entry<String, Object> stringObjectEntry : findItem.entrySet()) {
                                    if (!datum.containsKey(stringObjectEntry.getKey())) {
                                        datum.put(stringObjectEntry.getKey(), stringObjectEntry.getValue());
                                    }
                                }
                            }
                        }
                    }
                }
            } else {
                MessageUtils messageUtils = SpringUtil.getBean(MessageUtils.class);
                throw BusinessException.create(ErrorCodeEnum.NUM_500_0044.getErrCode(), messageUtils.getMessage("exception.define.data.primary.key"));
            }
        }
        return mainResult;
    }

    /**
     * 附加数据的元数据
     *
     * @param executeContext 执行上下文
     * @param parameter      参数
     * @param queryResult    查询结果
     * @return
     */
    @Override
    protected ApiMetadataCollection queryMetaDataCore(ExecuteContext executeContext,  Map<String, Object> parameter, QueryResult queryResult) {
        if (queryResult.getData() != null && queryResult.getData().size() ==0 ){
            return null;
        }
        ApiMetadataCollection apiMetadataCollection =  left.queryMetaData(executeContext, parameter,queryResult);
        if (CollectionUtils.isEmpty(rightList)){
            return apiMetadataCollection;
        }
        for (DataSourceBase dataSourceBase : rightList) {
            ApiMetadataCollection metadataCollection =  dataSourceBase.queryMetaData(executeContext, parameter,queryResult);
            if (metadataCollection!=null && metadataCollection.getMasterApiMetadata() != null && metadataCollection.getMasterApiMetadata().getResponseFields().size() >0) {
                apiMetadataCollection.getMasterApiMetadata().addResponseFields(metadataCollection.getMasterApiMetadata().getResponseFields().get(0).getSubFields());
            }
        }
        return apiMetadataCollection;
    }

    @Override
    protected DataSourceBase copyWithoutProcessorCore() {
        MergeDataSource mergeDataSource = new MergeDataSource();
        mergeDataSource.setName(this.getName());
        mergeDataSource.setAction(this.getAction());
        mergeDataSource.setType(this.getType());
        mergeDataSource.setDataKeys(this.getDataKeys());
        mergeDataSource.setActionId(this.getActionId());
        if (this.left != null) {
            mergeDataSource.setLeft(this.left.copyWithoutProcessor());
        }
        if (CollectionUtils.isNotEmpty(rightList)) {
            List<DataSourceBase> dataSourceBases = new ArrayList<>();
            for (DataSourceBase dataSourceBase : this.rightList) {

                dataSourceBases.add( dataSourceBase.copyWithoutProcessorCore());
            }
            mergeDataSource.setRightList(dataSourceBases);
        }

        return mergeDataSource;
    }
}
