package com.digiwin.athena.atdm.action.apiError;

import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.exception.BusinessException;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.digiwin.athena.atdm.action.dto.ApiErrorInlfoDTO;
import com.digiwin.athena.atdm.atmc.CommonAtmcService;
import com.digiwin.athena.atdm.mongodb.CommonMongodbService;
import com.digiwin.athena.atdm.constant.ErrorCodeEnum;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.data.mongodb.core.query.Update;
import org.springframework.stereotype.Service;
import org.springframework.util.StringUtils;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import static com.digiwin.athena.atdm.action.constant.CallbackConstants.*;

/**
 * api错误数据处理
 */
@Slf4j
@Service
public class ApiErrorInfoServiceImpl implements ApiErrorInfoService {

    @Autowired
    private CommonMongodbService commonMongodbService;

    @Autowired(required = false)
    @Qualifier("espRequestWorkItemTemplate")
    private MongoTemplate espRequestWorkItemTemplate;

    @Autowired
    private CommonAtmcService atmcService;

    private static String ERROR_INFO = "errorInfo";

    private MongoTemplate getEspRequestWorkItemTemplate() {
        if (null != espRequestWorkItemTemplate) {
            return espRequestWorkItemTemplate;
        }
        throw BusinessException.create("创建espRequestWorkItemTemplate失败，请检查数据源配置是否正确~");
    }


    /**
     * 保存请求和任务的关联关系
     * @param collectionName 租户id
     * @param dataMap
     */
    @Override
    public void insertRequestWorkItem(String collectionName, Map<String,Object> dataMap){
//        log.info("insertRequestWorkItem,collectionName:{},dataMap:{}",collectionName,JsonUtils.objectToString(dataMap));
        // 集合名不能为空
        if (StringUtils.isEmpty(collectionName)) {
            log.error("insertRequestWorkItem:collectionName is null");
            return;
        }
        // 数据不能为空
        if (checkData(dataMap)) {
            log.error("insertRequestWorkItem-dataMap:{}", JsonUtils.objectToString(dataMap));
            return;
        }
        //创建集合和索引
        Map<String, Object> indexMap = new HashMap<>();
        indexMap.put(REQ_ID, 1);
        indexMap.put(PTM_BACKLOG_ID, 1);
        indexMap.put(PTM_WORK_ITEM_ID, 1);
        commonMongodbService.createCollectionLock(getEspRequestWorkItemTemplate(),collectionName, indexMap);

        // 插入数据
        dataMap.put(STATUS,0);
        dataMap.put(CREATE_TIME, LocalDateTime.now());
        dataMap.put(MODIFY_TIME, LocalDateTime.now());
        commonMongodbService.insert(getEspRequestWorkItemTemplate(),dataMap,collectionName);
    }

    /**
     * 修改状态
     * @param collectionName
     * @param dataMap
     */
    @Override
    public void updateDataStatus(String collectionName,Map<String,Object> dataMap){
//        log.info("updateDataStatus,collectionName:{},dataMap:{}",collectionName,JsonUtils.objectToString(dataMap));
        // 集合名不能为空
        if (StringUtils.isEmpty(collectionName)) {
            log.error("updateDataStatus:collectionName is null");
            return;
        }
        // 数据不能为空
        if (checkData(dataMap)) {
            log.error("updateDataStatus-dataMap:{}", JsonUtils.objectToString(dataMap));
            return;
        }
        // 查询条件
        Query query = Query.query(Criteria.where(REQ_ID).is(dataMap.get(REQ_ID))
                .and(PTM_BACKLOG_ID).is(dataMap.get(PTM_BACKLOG_ID)).and(PTM_WORK_ITEM_ID).is(dataMap.get(PTM_WORK_ITEM_ID)));
        // 更新字段
        Update update = new Update();
        update.set(ERROR_INFO,dataMap.get(ERROR_INFO));
        update.set(STATUS,1);
        update.set(MODIFY_TIME,LocalDateTime.now());
        commonMongodbService.updateFirst(getEspRequestWorkItemTemplate(),collectionName,query,update);
    }

    /**
     * 查询数据
     * @return
     */
    @Override
    public Map<String,Object> queryDataByReqId(String collectionName, String reqId){
        if(StringUtils.isEmpty(reqId)){
            throw BusinessException.create(ErrorCodeEnum.API_ERR_PARAM_IS_NULL.getErrCode(), "reqId is null");
        }
        Query query = Query.query(Criteria.where(REQ_ID).is(reqId));
        List<Map> list = commonMongodbService.query(getEspRequestWorkItemTemplate(),collectionName, query);
        if (CollectionUtils.isNotEmpty(list)) {
            return list.get(0);
        }
        return null;
    }

    /**
     * 向运维模组发送error_info
     */
    @Override
    public void sendApiErrorInfo(List<Object> errorInfos, Long ptmBacklogId, Long ptmWorkItemId, String ApiName) {
        if (CollectionUtils.isEmpty(errorInfos)) {
            return;
        }
        ApiErrorInlfoDTO apiErrorInlfoDTO = new ApiErrorInlfoDTO();
        apiErrorInlfoDTO.setErrorInfos(errorInfos);
        apiErrorInlfoDTO.setPtmBacklogId(ptmBacklogId);
        apiErrorInlfoDTO.setPtmWorkItemId(ptmWorkItemId);
        apiErrorInlfoDTO.setApiName(ApiName);
        atmcService.sendApiErrorInfoMq(apiErrorInlfoDTO);
    }

    /**
     * esp全异步回调，向运维模组发送错误数据
     * @param reqId
     */
    @Override
    public void callBackApiErrorInfo(AuthoredUser user, String reqId, String data) {
        log.info("callBackApiErrorInfo-reqId:{}，user:{}，data:{}",reqId,
                JsonUtils.objectToString(user),JsonUtils.objectToString(data));
        // 请求id不能为空
        if (StringUtils.isEmpty(reqId)) {
            throw BusinessException.create(ErrorCodeEnum.API_ERR_PARAM_IS_NULL.getErrCode(), "reqId is null");
        }
        // 用户信息为空
        if (null == user) {
            throw BusinessException.create(ErrorCodeEnum.API_ERR_PARAM_IS_NULL.getErrCode(), "user is null");
        }

        // 根据请求id查询任务信息
        Map<String,Object> requestWorkItemData = queryDataByReqId(user.getTenantId(),reqId);
        if (MapUtils.isEmpty(requestWorkItemData)) {
            throw BusinessException.create(ErrorCodeEnum.API_ERR_PARAM_IS_NULL.getErrCode(), "requestWorkItemData is null");
        }
        //  消息内容
        Long ptmBacklogId = Long.valueOf(requestWorkItemData.get("ptmBacklogId").toString());
        Long ptmWorkItemId = Long.valueOf(requestWorkItemData.get("ptmWorkItemId").toString());
        String apiName = String.valueOf(requestWorkItemData.get("apiName"));
        List<Object> errorInfos = getErrorInfos(data);
        sendApiErrorInfo(errorInfos,ptmBacklogId,ptmWorkItemId,apiName);

        // 记录此请求已处理过回调
        Map<String,Object> mongoMap = new HashMap<>();
        mongoMap.put(REQ_ID,reqId);
        mongoMap.put(PTM_BACKLOG_ID,ptmBacklogId);
        mongoMap.put(PTM_WORK_ITEM_ID,ptmWorkItemId);
        mongoMap.put(ERROR_INFO,errorInfos);
        updateDataStatus(user.getTenantId(),mongoMap);
    }

    /**
     * 检查数据完整性
     * @param dataMap
     * @return
     */
    private boolean checkData(Map<String,Object> dataMap){
        return MapUtils.isEmpty(dataMap) || !dataMap.containsKey(REQ_ID)
                || !dataMap.containsKey(PTM_WORK_ITEM_ID) || !dataMap.containsKey(PTM_BACKLOG_ID);
    }

    /**
     * 拿到回调里的error_info
     * @param data
     * @return
     */
    private List<Object> getErrorInfos(String data){
        if (StringUtils.isEmpty(data)) {
           return new ArrayList<>();
        }
        Map<String,Object> dataMap = JsonUtils.jsonToObject(data,Map.class);
        if (!dataMap.containsKey("std_data") || null == dataMap.get("std_data")) {
            return new ArrayList<>();
        }
        Map<String,Object> stdDataMap = (Map<String,Object>) dataMap.get("std_data");
        if (!stdDataMap.containsKey("parameter") || null == stdDataMap.get("parameter")) {
            return new ArrayList<>();
        }
        Map<String,Object> parameterMap = (Map<String,Object>) stdDataMap.get("parameter");
        if (!parameterMap.containsKey("error_info") || null == parameterMap.get("error_info")) {
            return new ArrayList<>();
        }
        return (List<Object>) parameterMap.get("error_info");
    }

}
