package com.digiwin.athena.atdm.action.executor;


import cn.hutool.core.lang.Tuple;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.appcore.constant.LogConstant;
import com.digiwin.athena.appcore.domain.log.LogDto;
import com.digiwin.athena.appcore.exception.BusinessException;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.digiwin.athena.atdm.UiBotConstants;
import com.digiwin.athena.atdm.action.impl.MergeActionExecuteService;
import com.digiwin.athena.atdm.activity.domain.MergeSubmitActionDTO;
import com.digiwin.athena.atdm.activity.domain.SubmitActionDTO;
import com.digiwin.athena.atdm.atmc.CommonAtmcService;
import com.digiwin.athena.atdm.constant.ErrorCodeEnum;
import com.digiwin.athena.atdm.datasource.ActionExecuteReq;
import com.digiwin.athena.atdm.datasource.domain.ExecuteResult;
import com.digiwin.athena.atdm.datasource.domain.SubmitAction;
import com.digiwin.athena.atdm.datasource.domain.SubmitExecuteContext;
import com.digiwin.athena.atdm.datasource.domain.SubmitType;
import com.digiwin.athena.atdm.datasource.dto.DataSourceDTO;
import com.digiwin.athena.atdm.log.LogRecordDTO;
import com.digiwin.athena.atdm.log.LogRecordEvent;
import com.digiwin.athena.base.sdk.common.application.util.MessageUtil;
import com.google.common.collect.Maps;
import com.google.common.eventbus.AsyncEventBus;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.compress.utils.Lists;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.stereotype.Service;
import org.springframework.util.StringUtils;

import javax.servlet.http.HttpServletRequest;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import static com.digiwin.athena.atdm.log.LogRecordSubscriber.URL;

@Slf4j
@Service
public class DataSubmissionServiceWrapper {
    public static final String MERGE_WHITE_LIST = "mergeSubmitActivityIdWhiteList";

    //最大提交笔数
    public static final Integer MAX_COMMIT_SIZE = 2000;

    public static final String WHITE_LIST = "submitActivityIdWhiteList";


    @Autowired
    private CommonAtmcService atmcService;

    @Autowired
    private DataSubmissionService dataSubmissionService;

    @Autowired
    private ManualTaskActionExecutor manualTaskActionExecutor;

    @Autowired
    private AsyncEventBus asyncEventBus;

    @Autowired
    private MergeActionExecuteService mergeActionExecuteService;

    /**
     * 提交数据
     */
    public ExecuteResult executeAction(HttpServletRequest request, SubmitActionDTO submitAction) {
        try {
            actionCheck(request, submitAction);
            // 预处理API级别附件
            AttachmentActionExecutor.preprocessApiAttachment(com.google.common.collect.Lists.newArrayList(submitAction));
            SubmitExecuteContext executeContext = submitAction.getAction().getExecuteContext();
            SubmitAction action = submitAction.getAction();
            Map<String, Object> data = submitAction.getData();
            log.info("actionExecuteService.submit start");
            return this.dataSubmissionService.submit(new ActionExecuteReq(executeContext, null, action, data, submitAction.getWorkItemIdToData()));//actionService.submit(actionExecuteCommand);
        } catch (Exception e) {
            logRecord(request, submitAction);
            throw e;
        }
    }

    private void actionCheck(HttpServletRequest request, SubmitActionDTO submitAction) {
        if (submitAction.getAction() == null) {
            throw new IllegalArgumentException("submitAction.action为null");
        }
        if (submitAction.getData() == null) {
            throw new IllegalArgumentException("submitAction.data为null");
        }

        SubmitExecuteContext executeContext = submitAction.getAction().getExecuteContext();
        if (executeContext == null) {
            executeContext = SubmitExecuteContext.createByHttpRequest(request);
            submitAction.getAction().setExecuteContext(executeContext);
        } else {
            executeContext.appendHttpRequest(request);
        }
        if (StringUtils.hasText(executeContext.getProxyToken())) {
            AppAuthContextHolder.getContext().setProxyToken(executeContext.getProxyToken());
        }

        executeContext.setOperateAuthoredUser(submitAction.getOperateAuthoredUser());

        if (CollectionUtils.isNotEmpty(submitAction.getAction().getAttachActions())) {
            List<SubmitAction> attachActions = submitAction.getAction().getAttachActions();
            for (SubmitAction attachAction : attachActions) {
                SubmitExecuteContext attachActionExecuteContext = attachAction.getExecuteContext();
                if (attachActionExecuteContext == null) {
                    attachAction.setExecuteContext(executeContext);
                } else {
                    attachActionExecuteContext.appendHttpRequest(request);
                }
            }
        }

        SubmitAction action = submitAction.getAction();
        if (action == null) {
            throw new IllegalArgumentException("submitAction.getAction()");
        }

        //新增不超过2，000笔数据提交逻辑校验
        List<SubmitActionDTO> list = new ArrayList<>();
        list.add(submitAction);
        if (!validateSizeOrWhiteList(list)) {
            throw BusinessException.create(ErrorCodeEnum.BUSINESS_600_0003.getErrCode(), MessageUtil.getMessageByLocale("submit.task.size.overflow", LocaleContextHolder.getLocale().toString()));
        }

    }

    private boolean validateSizeOrWhiteList(List<SubmitActionDTO> submitActions) {

        // 全在在白名单中，就不考虑数据量了
        if (atmcService.operationConfig(submitActions.get(0).getAction().getExecuteContext().getTmActivityId(), 0, WHITE_LIST)) {
            return true;
        }

        // 不在白名单中，考虑是否有数据量大于2,000的
        return submitActions.stream().filter(e -> e.getData() != null).noneMatch(e -> e.getData().values().stream().filter(List.class::isInstance).map(el -> (List<?>) el).anyMatch(el -> el.size() > MAX_COMMIT_SIZE));
    }


    /**
     * 仅执行action
     * <br>不会判断是否处理完成
     * <br>不会关闭待办等
     */
    public ExecuteResult executeOpenTaskWindowAction(HttpServletRequest request, SubmitActionDTO submitAction) {
        if (submitAction.getAction() == null) {
            throw new IllegalArgumentException("submitAction.action为null");
        }
        if (submitAction.getData() == null) {
            throw new IllegalArgumentException("submitAction.data为null");
        }

        SubmitExecuteContext executeContext = submitAction.getAction().getExecuteContext();
        if (executeContext == null) {
            executeContext = SubmitExecuteContext.createByHttpRequest(request);
            executeContext.setBusinessUnit(submitAction.getAction().getBusinessUnit());
            submitAction.getAction().setExecuteContext(executeContext);
        } else {
            executeContext.appendHttpRequest(request);
        }
        if (StringUtils.hasText(executeContext.getProxyToken())) {
            AppAuthContextHolder.getContext().setProxyToken(executeContext.getProxyToken());
        }
        if (CollectionUtils.isNotEmpty(submitAction.getAction().getAttachActions())) {
            List<SubmitAction> attachActions = submitAction.getAction().getAttachActions();
            for (SubmitAction attachAction : attachActions) {
                SubmitExecuteContext attachActionExecuteContext = attachAction.getExecuteContext();
                if (attachActionExecuteContext == null) {
                    attachAction.setExecuteContext(executeContext);
                } else {
                    attachActionExecuteContext.appendHttpRequest(request);
                }
            }
        }
        preExecute(executeContext, submitAction);
        return doExecute(executeContext, submitAction);
    }

    private ExecuteResult doExecute(SubmitExecuteContext executeContext, SubmitActionDTO submitAction) {
        return dataSubmissionService.execute(executeContext, submitAction.getAction(), submitAction.getData());
    }

    private void preExecute(SubmitExecuteContext executeContext, SubmitActionDTO submitAction) {
        // 行事历手动任务的前置处理
        manualTaskActionExecutor.preExecute(executeContext, submitAction.getAction(), submitAction.getData());
    }

    /**
     * 项目卡退回接口
     */
    public List<ExecuteResult> executeReturnAction(HttpServletRequest request, List<SubmitActionDTO> submitActionList) throws Throwable {
        // 退回场景需要合并
        List<ExecuteResult> resultList = null;
        // 构造新的action 执行关卡逻辑
        if (CollectionUtils.isNotEmpty(submitActionList)) {
            SubmitActionDTO firstAction = submitActionList.get(0);
            // 数量为1不执行合并
            if (submitActionList.size() == 1) {
                return executeMergeAction(request, submitActionList);
            }
            // 合并提交的数据
            SubmitAction action = firstAction.getAction();
            actionCheck(request, firstAction);
            String category = action.getCategory();
            // 如果第一个是调用应用的api接口
            Map<String, Object> firstData = firstAction.getData();
            boolean isFirstEspApi = UiBotConstants.ACTION_CATEGORY_ESP.equals(category);
            if (isFirstEspApi) {
                SubmitType submitType = action.getSubmitType();
                // 数据源名称
                String dataSourceName;
                //  clone数据
                Map<String, Object> cloneData = JsonUtils.jsonToObject(JsonUtils.objectToString(firstData), Map.class);
                if (submitType != null) {
                    dataSourceName = submitType.getSchema();
                } else {
                    // 如果没有取第一个
                    dataSourceName = firstData.keySet().toArray()[0].toString();
                }
                // 将其他的数据放入第一个入参中
                List<Object> params = (List<Object>) cloneData.get(dataSourceName);
                for (int i = 1; i < submitActionList.size(); i++) {
                    SubmitActionDTO submitActionDto = submitActionList.get(i);
                    Map<String, Object> data = submitActionDto.getData();
                    Object o = data.get(dataSourceName);
                    if (o instanceof List) {
                        params.addAll((Collection<?>) o);
                    }
                }
                SubmitAction submitAction = firstAction.getAction();
                List<SubmitAction> attachActions = submitAction.getAttachActions();
                DataSourceDTO checkCompleteAction = submitAction.getCheckCompleteAction();
                // 不执行attach
                submitAction.setAttachActions(null);
                // 不执行检查
                submitAction.setCheckCompleteAction(null);
                // 不发送事件
                submitAction.setExecuteEventPost(true);
                this.dataSubmissionService.submit(firstAction.getAction().getExecuteContext(), action, cloneData);
                submitAction.setAttachActions(attachActions);
                submitAction.setCheckCompleteAction(checkCompleteAction);
                submitAction.setExecuteEventPost(false);
            }
            List<SubmitActionDTO> returnActionList = Lists.newArrayList();
            // 获取退回的api配置
            mergeReturnTaskAction(returnActionList, submitActionList);
            if (CollectionUtils.isEmpty(returnActionList)) {
                throw new IllegalArgumentException("actionId:task-return 不存在");
            }
            // 执行退回
            Map<String, Object> param = Maps.newHashMapWithExpectedSize(1);
            SubmitActionDTO returnActionDto = returnActionList.get(0);
            SubmitAction returnAction = JsonUtils.jsonToObject(JsonUtils.objectToString(returnActionDto.getAction()), SubmitAction.class);
            returnAction.setActionParams(null);
            returnAction.setParas(null);
            returnAction.setExecuteEventPost(true);
            param.put(UiBotConstants.ACTION_RETURN_PARAMS_KEY, returnActionList);
            // 执行合并后的退回；executeContext从最外面取
            this.dataSubmissionService.submit(returnAction.getExecuteContext(), returnAction, param);
            // 分批执行dispatchAction
            if (isFirstEspApi) {
                for (SubmitActionDTO actionDTO : submitActionList) {
                    SubmitAction submitAction = actionDTO.getAction();
                    //为了当前api不执行，但是为了执行关闭关卡isDispatchAction
                    submitAction.setExecuteAction(true);
                }
            }
            // 执行dispatch
            Tuple tuple = executeSubmitActionList(request, submitActionList);
            resultList = tuple.get(2);
        }
        return resultList;
    }

    private void mergeReturnTaskAction(List<SubmitActionDTO> returnActionList, List<SubmitActionDTO> submitActionList) {
        Iterator<SubmitActionDTO> iterator = submitActionList.iterator();
        while (iterator.hasNext()) {
            SubmitActionDTO actionDTO = iterator.next();
            SubmitAction action = actionDTO.getAction();
            String actionId = action.getActionId();
            if ("task-return".equals(actionId)) {
                returnActionList.add(actionDTO);
                iterator.remove();
            } else if (CollectionUtils.isNotEmpty(action.getAttachActions())) {
                List<SubmitAction> attachActions = action.getAttachActions();
                Iterator<SubmitAction> actionIterator = attachActions.iterator();
                while (actionIterator.hasNext()) {
                    SubmitAction next = actionIterator.next();
                    if ("task-return".equals(next.getActionId())) {
                        SubmitActionDTO submitActionDTO = new SubmitActionDTO();
                        submitActionDTO.setAction(next);
                        submitActionDTO.setData(actionDTO.getData());
                        submitActionDTO.setOperateAuthoredUser(actionDTO.getOperateAuthoredUser());
                        returnActionList.add(submitActionDTO);
                        actionIterator.remove();
                        break;
                    }
                }
            }
        }
    }


    private Tuple executeSubmitActionList(HttpServletRequest request, List<SubmitActionDTO> submitActionList) {
        List<ExecuteResult> resultList = new ArrayList<>();
        String backlogId = "";
        String tenantId = "";
        //执行 所有合并数据的任务之外的action
        for (SubmitActionDTO submitActionDTO : submitActionList) {
            SubmitAction taskAction = submitActionDTO.getAction();
            SubmitExecuteContext executeContext = taskAction.getExecuteContext();
            if (executeContext == null) {
                executeContext = SubmitExecuteContext.createByHttpRequest(request);
            }
            if (StringUtils.hasText(executeContext.getProxyToken())) {
                AppAuthContextHolder.getContext().setProxyToken(executeContext.getProxyToken());
            }

            executeContext.setOperateAuthoredUser(submitActionDTO.getOperateAuthoredUser());
            setExecuteContext(request, taskAction, executeContext);

            backlogId = String.valueOf(executeContext.getBacklogId());
            tenantId = submitActionDTO.getOperateAuthoredUser() == null ? "" : submitActionDTO.getOperateAuthoredUser().getTenantId();
            Map<String, Object> data = submitActionDTO.getData();
            LogDto logDto = new LogDto("执行所有合并数据的任务之外的action开始，backlogId：" + backlogId, tenantId + LogConstant.TRACE_SEPARATOR + backlogId);
            log.info(logDto.toString());
            ExecuteResult executeResult = this.dataSubmissionService.submit(executeContext, taskAction, data);//actionService.submit(actionExecuteCommand);
            LogDto logDtoEnd = new LogDto("执行所有合并数据的任务之外的action结束，backlogId：" + backlogId, tenantId + LogConstant.TRACE_SEPARATOR + backlogId);
            log.info(logDtoEnd.toString());

            Map<String, Object> extendResult = new HashMap<>();
            extendResult.put("backlogId", executeContext.getBacklogId());
            executeResult.setExtendResult(extendResult);
            resultList.add(executeResult);
        }
        return new Tuple(backlogId, tenantId, resultList);

    }

    /**
     * 提交合并任务的数据
     */
    public List<ExecuteResult> executeMergeAction(HttpServletRequest request, List<SubmitActionDTO> submitActionList) throws Throwable {
        if (CollectionUtils.isEmpty(submitActionList)) {
            throw new IllegalArgumentException("submitActionList为null或为空");
        }
        for (SubmitActionDTO submitActionDTO : submitActionList) {
            if (submitActionDTO.getAction() == null) {
                throw new IllegalArgumentException("submitActionList.action为null");
            }
        }
        //新增不超过2，000笔数据提交逻辑校验
        if (!validateSizeOrWhiteList(submitActionList)) {
            throw BusinessException.create(ErrorCodeEnum.BUSINESS_600_0003.getErrCode(), MessageUtil.getMessageByLocale("submit.task.size.overflow", LocaleContextHolder.getLocale().toString()));
        }

        // 预处理API级别附件
        AttachmentActionExecutor.preprocessApiAttachment(submitActionList);
        //走新的逻辑
        return mergeActionExecuteService.executeMergeActionWapper(request, submitActionList);
    }

    /**
     * 提交合并任务的数据_支持跨BK
     *
     * @return
     */
    public List<ExecuteResult> executeMergeActionCrossBk(HttpServletRequest request, List<SubmitActionDTO> submitActionList) throws Throwable {
        if (CollectionUtils.isEmpty(submitActionList)) {
            throw new IllegalArgumentException("submitActionList为null或为空");
        }
        for (SubmitActionDTO submitActionDTO : submitActionList) {
            if (submitActionDTO.getAction() == null) {
                throw new IllegalArgumentException("submitActionList.action为null");
            }
        }
        //新增不超过2，000笔数据提交逻辑校验
        if (!validateSizeOrWhiteList(submitActionList)) {
            throw BusinessException.create(ErrorCodeEnum.BUSINESS_600_0003.getErrCode(), MessageUtil.getMessageByLocale("submit.task.size.overflow", LocaleContextHolder.getLocale().toString()));
        }

        //走新的逻辑
        if (atmcService.operationConfig(MERGE_WHITE_LIST + "_" + submitActionList.get(0).getAction().getExecuteContext().getTmActivityId(), 0, MERGE_WHITE_LIST)) {
            LogDto logDto = new LogDto("执行支持跨BK的合并任务新逻辑");
            log.info(logDto.toString());
            return mergeActionExecuteService.executeMergeActionWapper(request, submitActionList);
        }

        log.info("start to deal find support submit merge data action start");
        //所有合并数据的任务
        List<SubmitAction> mergeDataActions = new ArrayList<>();
        for (SubmitActionDTO submitActionDTO : submitActionList) {
            SubmitAction taskAction = submitActionDTO.getAction();
            findSupportSubmitMergeDataActions(mergeDataActions, taskAction);
        }
        log.info("start to deal find support submit merge data action end");

        Collection<SubmitAction> needExecuteMergeActions = initSubmitMergeDataActions(mergeDataActions);
        log.info("init submit merge data action end");
        Tuple tuple = executeSubmitActionList(request, submitActionList);
        String backlogId = tuple.get(0);
        String tenantId = tuple.get(1);
        log.info("execute non-merge action  end");
        //合并执行所有任务的 start-new-process
        LogDto logDto = new LogDto("执行所有支持跨BK的合并数据的任务action开始，backlogId：" + backlogId, tenantId + LogConstant.TRACE_SEPARATOR + backlogId);
        log.info(logDto.toString());
        executeMergeAction(submitActionList.get(0), needExecuteMergeActions);
        log.info("execute merge action  end");
        LogDto logDtoEnd = new LogDto("执行所有支持跨BK的合并数据的任务action结束，backlogId：" + backlogId, tenantId + LogConstant.TRACE_SEPARATOR + backlogId);
        log.info(logDtoEnd.toString());
        return tuple.get(2);
    }

    public List<ExecuteResult> executeMergeActionRetry(HttpServletRequest request, MergeSubmitActionDTO mergeSubmitAction) throws Throwable {
        try {
            return mergeActionExecuteService.executeMergeAction(request, mergeSubmitAction);
        } catch (Throwable e) {
            mergeActionExecuteService.logRecord(request, mergeSubmitAction);
            throw e;
        }
    }

    private void setExecuteContext(HttpServletRequest request, SubmitAction taskAction, SubmitExecuteContext parentExecuteContext) {
        SubmitExecuteContext executeContext = taskAction.getExecuteContext();
        if (executeContext == null) {
            executeContext = parentExecuteContext;
            taskAction.setExecuteContext(parentExecuteContext);
        } else {
            executeContext.appendHttpRequest(request);
        }

        if (CollectionUtils.isNotEmpty(taskAction.getAttachActions())) {
            List<SubmitAction> attachActions = taskAction.getAttachActions();
            for (SubmitAction attachAction : attachActions) {
                setExecuteContext(request, attachAction, executeContext);
            }
        }
    }

    private Collection<SubmitAction> initSubmitMergeDataActions(List<SubmitAction> mergeDataActions) {
        if (CollectionUtils.isEmpty(mergeDataActions)) {
            return mergeDataActions;
        }
        Map<String, SubmitAction> firstActions = new LinkedHashMap<>();

        for (SubmitAction submitAction : mergeDataActions) {
            //设置模拟执行
            if (!firstActions.containsKey(submitAction.getActionId())) {
                firstActions.put(submitAction.getActionId(), submitAction);
            }
            submitAction.setMergeDataAction(firstActions.get(submitAction.getActionId()));
        }
        return firstActions.values();
    }


    private void findSupportSubmitMergeDataActions(List<SubmitAction> list, SubmitAction taskAction) {
        if (this.dataSubmissionService.isSupportSubmitMergeData(taskAction)) {
            list.add(taskAction);
        }
        if (CollectionUtils.isNotEmpty(taskAction.getAttachActions())) {
            for (SubmitAction attachAction : taskAction.getAttachActions()) {
                findSupportSubmitMergeDataActions(list, attachAction);
            }
        }
    }

    private void executeMergeAction(SubmitActionDTO submitActionDTO, Collection<SubmitAction> needExecuteMergeActions) {
        if (CollectionUtils.isEmpty(needExecuteMergeActions)) {
            return;
        }
        Map<String, Object> startData = submitActionDTO.getData();
        for (SubmitAction needExecuteMergeAction : needExecuteMergeActions) {
            needExecuteMergeAction.setAttachActions(null);
            this.dataSubmissionService.submit(needExecuteMergeAction.getExecuteContext(), needExecuteMergeAction, startData);//actionService.submit(actionExecuteCommand);
        }

    }

    private void logRecord(HttpServletRequest request, SubmitActionDTO submitAction) {
        try {
            LogRecordDTO logRecordDTO = new LogRecordDTO();
            logRecordDTO.setInputValue(JsonUtils.objectToString(submitAction));
            logRecordDTO.setCreateTime(new Date());
            logRecordDTO.setUrl(URL);
            LogRecordEvent logRecordEvent = new LogRecordEvent(request, logRecordDTO, submitAction.getOperateAuthoredUser());
            asyncEventBus.post(logRecordEvent);
        } catch (Exception e) {
            log.error("记录日志EventBus任务异常：{}", e.getMessage());
        }
    }
}
