package com.digiwin.athena.atdm.action.executor;

import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.constant.LogConstant;
import com.digiwin.athena.appcore.domain.log.LogDto;
import com.digiwin.athena.appcore.exception.BusinessException;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.atdm.ActionConstants;
import com.digiwin.athena.atdm.UiBotConstants;
import com.digiwin.athena.atdm.ptm.dto.PtmWithdrawReqDTO;
import com.digiwin.athena.atdm.ptm.dto.PtmWithdrawRespDTO;
import com.digiwin.athena.atdm.uibot.dto.UiBotTaskWithdrawReqDTO;
import com.digiwin.athena.atdm.action.ActionExecutor;
import com.digiwin.athena.atdm.atmc.CommonAtmcService;
import com.digiwin.athena.atdm.ptm.CommonPtmService;
import com.digiwin.athena.atdm.uibot.CommonUiBotService;
import com.digiwin.athena.atdm.activity.ActivityConstants;
import com.digiwin.athena.atdm.activity.service.BpmServiceInvokeType;
import com.digiwin.athena.atdm.constant.ErrorCodeEnum;
import com.digiwin.athena.atdm.datasource.datasource.DataSourceSet;
import com.digiwin.athena.atdm.datasource.datasource.converter.DataSourceConverter;
import com.digiwin.athena.atdm.retrieveData.CommonRetrieveDataService;
import com.digiwin.athena.atdm.smartdata.CommonSmartDataService;
import com.digiwin.athena.atdm.retrieveData.dto.RetrieveDataDTO;
import com.digiwin.athena.atdm.smartdata.dto.RetractCompareReqDTO;
import com.digiwin.athena.atdm.datasource.domain.ExecuteContext;
import com.digiwin.athena.atdm.datasource.domain.ExecuteResult;
import com.digiwin.athena.atdm.datasource.domain.QueryResultSet;
import com.digiwin.athena.atdm.datasource.domain.SubmitAction;
import com.digiwin.athena.atdm.datasource.domain.SubmitExecuteContext;
import com.digiwin.athena.atdm.datasource.dto.DataSourceSetDTO;
import com.fasterxml.jackson.core.type.TypeReference;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.BooleanUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.annotation.Order;
import org.springframework.stereotype.Service;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * TaskWithdrawExecutor Description
 *
 * @author majianfu
 * @date 2022/10/12
 * @since
 */
@Slf4j
@Order(101)
@Service("uibot_task_withdraw")
public class TaskWithdrawExecutor implements ActionExecutor {
    @Autowired
    private CommonPtmService commonPtmService;

    @Autowired
    private CommonUiBotService commonUiBotService;

    @Autowired
    private MessageUtils messageUtils;

    @Autowired
    private CommonAtmcService atmcService;

    @Autowired
    private CommonSmartDataService commonSmartDataService;

    @Autowired
    private CommonRetrieveDataService commonRetrieveDataService;

    @Autowired
    private DataSubmissionService dataSubmissionService;

    @Override
    public String supportKey() {
        return UiBotConstants.ACTION_CATEGORY_UIBOT + ActionConstants.SPLIT + BpmServiceInvokeType.TaskWithdraw.getName();
    }

    @Override
    public boolean hasActionMetadata() {
        return false;
    }

    @Override
    public ExecuteResult execute(SubmitExecuteContext executeContext, ExecuteResult parentExecuteResult, SubmitAction action, Map<String, Object> parameter) {
//        log.info("[ptm-task-withdraw] 入参：{}", JsonUtils.objectToString(action));
        // 参数校验
        this.checkWithdrawAction(action);

        // 提交数据中属于业务数据的字段名
        String submitVarName = action.getExtendParas().get("submitVariableName").toString();
        // 提交数据中业务数据的主键字段名称集合
        String[] dataKeys = StringUtils.split(action.getExtendParas().get("dataKeys").toString(), ";");

        // 将提交的业务数据按 workItemId归纳好
        List<Map<String, Object>> submitPageDataList = parseSubmitData(submitVarName, parameter);
        Map<Long, List<Map<String, Object>>> workItemSubmitDataMap = classifySubmitDataByWorkItemId(submitPageDataList);
        // 解析出backlogId
        Long backlogId = parseBacklogId(workItemSubmitDataMap);

        // 查询当前撤回任务关联的所有任务
        List<PtmWithdrawRespDTO> relativeWithdrawTaskList = this.queryRelativeWithdrawTask(executeContext.getTenantId(),backlogId, dataKeys, workItemSubmitDataMap);
        if (CollectionUtils.isEmpty(relativeWithdrawTaskList)) {
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0032.getErrCode(), messageUtils.getMessage("exception.task.withdraw.failed.to.query.relative.task"));
        }

        // 当前登录用户
        AuthoredUser authoredUser = executeContext.getAuthoredUser();
        // 保存当前任务 + 后置关联任务的“撤回”数据
        saveRelativeWithdrawnData(authoredUser, backlogId, relativeWithdrawTaskList, dataKeys, workItemSubmitDataMap);

        List<Map<String, Object>> bkList = this.parseBK(dataKeys, submitPageDataList);
        // 撤回数据一致性数据
        this.retractComparisonData(authoredUser, backlogId, relativeWithdrawTaskList, executeContext.getTmActivityId(), executeContext.getBusinessUnit(), bkList);

        // 当前执行者userId
        String performerId = this.parsePerformerId(action.getParas(), authoredUser);
        // 当前执行者类型：执行者、代理人、辅助执行者、超级管理员
        Integer performerType = this.parsePerformerType(action.getParas(),executeContext.getPageCode());
        // 流程变量key
        String processVarName = action.getExtendParas().get("processVariableName").toString();
        // 执行当前任务 + 后置关联任务的撤回
        this.executeRelativeTaskWithdraw(performerId, performerType, relativeWithdrawTaskList, processVarName, dataKeys, backlogId, workItemSubmitDataMap);

        return ExecuteResult.ok();
    }

    private void retractComparisonData(AuthoredUser authoredUser, Long backlogId, List<PtmWithdrawRespDTO> relativeWithdrawTaskList,
                                       String tmActivityId, Map<String, Object> eocMap, List<Map<String, Object>> bkList) {
        // 类型擦除
        String bkListJson = JsonUtils.objectToString(bkList);
        List<Map> tmpBkList = JsonUtils.jsonToListObject(bkListJson, Map.class);
        // 请求ATMC拿到BK entityName
        List<Map> cardBKInfoList = atmcService.queryCardListByBk("workitem", backlogId, tmpBkList, true);
        if (CollectionUtils.isEmpty(cardBKInfoList)) {
            log.warn("[ptm-task-withdraw-{}] 未能获取到BK信息: tmActivityId: {}, bkList: {}", backlogId, tmActivityId, bkListJson);
            return;
        }

        for (PtmWithdrawRespDTO relativeWithdrawTask : relativeWithdrawTaskList) {
            if (CollectionUtils.isNotEmpty(relativeWithdrawTask.getNextWorkItems())) {
                for (PtmWithdrawRespDTO.Task nextWorkItem : relativeWithdrawTask.getNextWorkItems()) {
                    Optional<Map> cardBkOpt = cardBKInfoList.stream()
                            .filter(MapUtils::isNotEmpty)
                            .filter(cardBKInfo -> Objects.equals(backlogId, cardBKInfo.get("backlogId"))
                                    && Objects.equals(nextWorkItem.getWorkItemId(), cardBKInfo.get("workItemId"))
                                    && null != cardBKInfo.get("entityName"))
                            .findFirst();
                    if (!cardBkOpt.isPresent() || MapUtils.isEmpty(cardBkOpt.get())) {
                        continue;
                    }

                    Map cardBKInfo = cardBkOpt.get();
                    RetractCompareReqDTO retractCompareReqDTO = new RetractCompareReqDTO();
                    retractCompareReqDTO.setTenantId(authoredUser.getTenantId());
                    retractCompareReqDTO.setInstanceId(String.valueOf(cardBKInfo.get("workItemId")));
                    retractCompareReqDTO.setTempId(String.valueOf(cardBKInfo.get("tmActivityId")));
                    retractCompareReqDTO.setEocMap(eocMap);
                    retractCompareReqDTO.setBks((List<Map>) cardBKInfo.get("bkList"));
                    retractCompareReqDTO.setEntityName(String.valueOf(cardBKInfo.get("entityName")));
//                    log.info("[ptm-task-withdraw-{}] atdm访问SD执行撤销数据对比, 请求参数：{}", backlogId, JsonUtils.objectToString(retractCompareReqDTO));
                    this.commonSmartDataService.retractComparisonData(retractCompareReqDTO);
                }
            }
        }
    }

    private Long parseBacklogId(Map<Long, List<Map<String, Object>>> workItemSubmitDataMap) {
        Long backlogId = null;
        for (Map.Entry<Long, List<Map<String, Object>>> workItemSubmitData : workItemSubmitDataMap.entrySet()) {
            // 取第一行数据的backlogId，所有行的backlogId都一样的
            Map<String, Object> submitData = workItemSubmitData.getValue().get(0);
            Map<String, Object> backlogData = (Map<String, Object>) submitData.get("activity__backLog__data");

            backlogId = (Long) backlogData.get("ptmBacklogId");
            if(backlogId != null) {
                break;
            }
        }
        return backlogId;
    }

    private Map<Long, List<Map<String, Object>>> classifySubmitDataByWorkItemId(List<Map<String, Object>> submitDataList) {
        // [workItemId, List<提交数据：没行数据中，可能workItemId是一样的>]
        Map<Long, List<Map<String, Object>>> workItemSubmitDataMap = new HashMap<>();
        for (Map<String, Object> submitData : submitDataList) {
            if (submitData.containsKey("activity__backLog__data")) {
                Map<String, Object> backlogData = (Map<String, Object>) submitData.get("activity__backLog__data");
                if (null != backlogData
                        && null != backlogData.get("ptmBacklogId")
                        && null != backlogData.get("ptmWorkItemId")) { // work_item_id
                    List<Map<String, Object>> workItemList = workItemSubmitDataMap.computeIfAbsent((Long) backlogData.get("ptmWorkItemId"), workItemId -> new ArrayList<>());
                    workItemList.add(submitData);
                } else {
                    // 抛出异常
                    throw BusinessException.create(ErrorCodeEnum.NUM_500_0033.getErrCode(), messageUtils.getMessage("exception.task.withdraw.data.lack.necessary.config"));
                }
            } else {
                throw BusinessException.create(ErrorCodeEnum.NUM_500_0033.getErrCode(),messageUtils.getMessage("exception.task.withdraw.data.lack.necessary.config"));
            }
        }
        return workItemSubmitDataMap;
    }

    private void saveRelativeWithdrawnData(AuthoredUser authoredUser, Long backlogId, List<PtmWithdrawRespDTO> relativeTaskList,
                                           String[] dataKeys, Map<Long, List<Map<String, Object>>> workItemSubmitDataMap) {
        // 为每个workItem解析出bk列表
        Map<Long, List<Map<String, Object>>> workItemBKMap = parseWorkItemBK(dataKeys, workItemSubmitDataMap);

        // 找出后置任务列表
        List<PtmWithdrawRespDTO> needParseRelativeTaskList = parsePostRelativeTask(relativeTaskList, workItemBKMap);

        LocalDateTime withdrawDateTime = LocalDateTime.now();

        // 解析并保存后置任务撤回的数据
        parseAndSavePostRelativeTaskWithdrawData(authoredUser, withdrawDateTime, backlogId, dataKeys, workItemBKMap, needParseRelativeTaskList);

        // 保存当前任务撤回数据
        for (Map.Entry<Long, List<Map<String, Object>>> workItemSubmitData : workItemSubmitDataMap.entrySet()) {
            // 保存任务撤回的业务数据
            saveRetrievedData(authoredUser, withdrawDateTime, backlogId, workItemSubmitData.getKey(), dataKeys, workItemSubmitData.getValue());
        }
    }

    private List<PtmWithdrawRespDTO> parsePostRelativeTask(List<PtmWithdrawRespDTO> relativeTaskList, Map<Long, List<Map<String, Object>>> workItemBKMap) {
        List<PtmWithdrawRespDTO> needParseRelativeTaskList = new ArrayList<>();
        // 收集当前待办项所属的taskId，关联任务中taskId和当前待办项所属taskId相同的，都认为不是后置任务
        List<List<Long>> rowCurrentWorkTaskIds = new ArrayList<>(relativeTaskList.size());
        for (PtmWithdrawRespDTO relativeTask : relativeTaskList) {
            List<Long> currWorkTaskId = new ArrayList<>();
            if (CollectionUtils.isNotEmpty(relativeTask.getNextWorkItems())) {
                for (PtmWithdrawRespDTO.Task nextWorkItem : relativeTask.getNextWorkItems()) {
                    // 收集当前提交任务的taskId
                    if (workItemBKMap.containsKey(nextWorkItem.getWorkItemId())) {
                        currWorkTaskId.add(nextWorkItem.getTaskId());
                    }
                }
            }
            rowCurrentWorkTaskIds.add(currWorkTaskId);
        }

        for (int idx = 0; idx < relativeTaskList.size(); idx++) {
            PtmWithdrawRespDTO relativeTask = relativeTaskList.get(idx);
            if (CollectionUtils.isEmpty(relativeTask.getNextWorkItems())) {
                continue;
            }

            List<PtmWithdrawRespDTO.Task> needParseNextWorkItems = new ArrayList<>();
            List<Long> currentWorkTaskId = rowCurrentWorkTaskIds.get(idx);
            for (PtmWithdrawRespDTO.Task nextWorkItem : relativeTask.getNextWorkItems()) {
                // 不是当前提交的workItem
                if (!workItemBKMap.containsKey(nextWorkItem.getWorkItemId())) {
                    // 和当前提交待办项是否属于同一个taskId
                    nextWorkItem.setInSameTask(currentWorkTaskId.contains(nextWorkItem.getTaskId()));
                    needParseNextWorkItems.add(nextWorkItem);
                }
            }

            if (CollectionUtils.isNotEmpty(needParseNextWorkItems)) {
                PtmWithdrawRespDTO needParseRelativeTask = new PtmWithdrawRespDTO();
                needParseRelativeTask.setWorkItemId(relativeTask.getWorkItemId());
                needParseRelativeTask.setState(relativeTask.getState());
                needParseRelativeTask.setNextWorkItems(needParseNextWorkItems);

                needParseRelativeTaskList.add(needParseRelativeTask);
            }
        }
        return needParseRelativeTaskList;
    }

    private void parseAndSavePostRelativeTaskWithdrawData(AuthoredUser authoredUser, LocalDateTime withdrawDateTime, Long backlogId, String[] dataKeys
            , Map<Long, List<Map<String, Object>>> workItemBKMap, List<PtmWithdrawRespDTO> needParseRelativeTaskList) {
        if (CollectionUtils.isEmpty(needParseRelativeTaskList)) {
            return;
        }
        // 解析关联撤回任务的PageDefine定义
        Map<Long, List<Map>> workItemPageDefineMap = parseRelativeWithdrawTaskPageDefine(backlogId, needParseRelativeTaskList, workItemBKMap);
        if (MapUtils.isEmpty(workItemPageDefineMap)) {
            return;
        }

        // 执行数据源查询
        for (Map.Entry<Long, List<Map>> workItemPageDefine : workItemPageDefineMap.entrySet()) {
            Long workItemId = workItemPageDefine.getKey();
            List<Map> pageDefines = workItemPageDefine.getValue();
            if (CollectionUtils.isEmpty(pageDefines)) {
                continue;
            }

            SubmitAction terminateTaskAction = null;
            // 一般而言，一条数据要么在待处理、要么在已完成；只要从任意一个标签中查询到数据，另外一个标签就无需在查询了
            for (Map pageDefine : pageDefines) {
                // 必须包含executeContext，dataSourceSet，submitActions
                if (null == pageDefine || null == pageDefine.get("executeContext") || null == pageDefine.get("dataSourceSet")) {
                    log.error("[ptm-task-withdraw-{}-{}] 未能成功解析出页签定义", backlogId, workItemId);
                    continue;
                }

                ExecuteContext executeContext = convertJson(pageDefine.get("executeContext"), ExecuteContext.class);
                executeContext.setAuthoredUser(authoredUser);
                if (StringUtils.isBlank(executeContext.getTenantId())) {
                    executeContext.setTenantId(executeContext.getAuthoredUser().getTenantId());
                }
                DataSourceSetDTO dataSourceSet = convertJson(pageDefine.get("dataSourceSet"), DataSourceSetDTO.class);
                Map<String, Object> parameter = null != pageDefine.get("parameter") ?
                        JsonUtils.jsonToObject(JsonUtils.objectToString(pageDefine.get("parameter")), new TypeReference<Map<String, Object>>() {
                        }) : new HashMap<>();
                // 执行数据源查询
                QueryResultSet queryResultSet = this.executeDataSource(executeContext, dataSourceSet, parameter);
                // 任务提交数据中业务数据Key
                Optional<String> taskSubmitVarName = Optional.ofNullable((Map) pageDefine.get("extendedFields"))
                        .map(extendedFields -> extendedFields.get("submitVariableName"))
                        .map(submitVariableName -> submitVariableName.toString());
                // 任务提交的业务数据
                List<Map<String, Object>> submitDataList = this.parseSubmitData(taskSubmitVarName.isPresent() ? taskSubmitVarName.get() : "", queryResultSet.getPageData());
                if (CollectionUtils.isNotEmpty(submitDataList)) {
                    // 保存任务撤回的业务数据
                    saveRetrievedData(authoredUser, withdrawDateTime, backlogId, workItemId, dataKeys, submitDataList);
                }

                if (null != pageDefine.get("submitActions") && CollectionUtils.isNotEmpty((List) pageDefine.get("submitActions"))) {
                    terminateTaskAction = JsonUtils.jsonToObject(JsonUtils.objectToString(((List) pageDefine.get("submitActions")).get(0)), SubmitAction.class);
                }
            }

            // 执行checkCompleteAction + terminate-task
            if (null != terminateTaskAction) {
                if (null == terminateTaskAction.getExecuteContext().getAuthoredUser()) {
                    terminateTaskAction.getExecuteContext().setAuthoredUser(authoredUser);
                }
                if (StringUtils.isBlank(terminateTaskAction.getExecuteContext().getTenantId())) {
                    terminateTaskAction.getExecuteContext().setTenantId(authoredUser.getTenantId());
                }
//                log.info("[ptm-task-withdraw-{}-{}] 执行checkCompleteAction + terminateTaskAction: {}", backlogId, workItemId, JsonUtils.objectToString(terminateTaskAction));
                dataSubmissionService.submit(terminateTaskAction.getExecuteContext(), terminateTaskAction, null);
            }
        }
    }

    private <T> T convertJson(Object param, Class<T> clz) {
        return JsonUtils.jsonToObject(JsonUtils.objectToString(param), clz);
    }

    private void saveRetrievedData(AuthoredUser authoredUser, LocalDateTime withdrawDateTime, Long backlogId, Long workItemId
            , String[] dataKeys, List<Map<String, Object>> submitDataList) {
//        log.info("[ptm-task-withdraw-{}-{}] 保存撤回数据: {}", backlogId, workItemId, JsonUtils.objectToString(submitDataList));
        RetrieveDataDTO retrieveData = new RetrieveDataDTO();

        retrieveData.setUserId(authoredUser.getUserId());
        retrieveData.setUserName(authoredUser.getUserName());
        retrieveData.setTenantId(authoredUser.getTenantId());

        retrieveData.setActivityId(String.valueOf(workItemId));
        // 发起“撤回”的任务backlogId
        retrieveData.setInitiateActivityId(String.valueOf(backlogId));

        retrieveData.setBkList(Arrays.asList(dataKeys));
        retrieveData.setBkDataList(this.parseBK(dataKeys, submitDataList));

        retrieveData.setTerminateTime(withdrawDateTime);

        retrieveData.setPageData(submitDataList);
        retrieveData.setPageDataSize(submitDataList.size());

        commonRetrieveDataService.saveRetrieveData(retrieveData);
    }

    /**
     * 解析出所有关联撤回任务的PageDefine定义
     */
    private Map<Long, List<Map>> parseRelativeWithdrawTaskPageDefine(Long backlogId, List<PtmWithdrawRespDTO> relativeTaskList, Map<Long, List<Map<String, Object>>> workItemBKMap) {
        List<UiBotTaskWithdrawReqDTO> uiBotTaskWithdrawReqList = new ArrayList<>();

        for (PtmWithdrawRespDTO ptmWithdrawRespDTO : relativeTaskList) {
            for (PtmWithdrawRespDTO.Task nextTask : ptmWithdrawRespDTO.getNextWorkItems()) {
                UiBotTaskWithdrawReqDTO uiBotTaskWithdrawReqDTO = new UiBotTaskWithdrawReqDTO();

                uiBotTaskWithdrawReqDTO.setBacklogId(backlogId);
                uiBotTaskWithdrawReqDTO.setWorkItemId(nextTask.getWorkItemId());

                uiBotTaskWithdrawReqDTO.setTaskDefCode(nextTask.getTaskDefCode());
                uiBotTaskWithdrawReqDTO.setProjectDefCode(nextTask.getProjectDefCode());
                // 注意：这边是发起撤回的待办项id
                uiBotTaskWithdrawReqDTO.setBkList(workItemBKMap.get(ptmWithdrawRespDTO.getWorkItemId()));
                // 塞入待办项状态
                uiBotTaskWithdrawReqDTO.setState(nextTask.getState());
                // nextTask和当前执行撤回的待办项属于同一个任务，则不需要校验是否有待处理数据 + 关闭任务卡
                uiBotTaskWithdrawReqDTO.setCheckCompleted(BooleanUtils.isFalse(nextTask.getInSameTask()));

                uiBotTaskWithdrawReqList.add(uiBotTaskWithdrawReqDTO);
            }
        }



//        log.info("[ptm-task-withdraw-{}] 从UiBot获取任务定义，请求待办项列表PageDefine定义: {}", backlogId, JsonUtils.objectToString(uiBotTaskWithdrawReqList));

        Map<Long, List<Map>> taskPageDefineMap = commonUiBotService.queryTaskWithdrawPageDefine(uiBotTaskWithdrawReqList);

//        log.info("[ptm-task-withdraw-{}] 从UiBot获取任务定义，响应待办项id列表: {}", backlogId,
//                MapUtils.isNotEmpty(taskPageDefineMap) ? taskPageDefineMap.keySet() : new HashSet<>());
        return taskPageDefineMap;
    }

    /**
     * 执行数据源查询，查出关联撤回任务的业务数据
     */
    private QueryResultSet executeDataSource(ExecuteContext executeContext, DataSourceSetDTO dataSourceSetDTO, Map<String, Object> parameter) {
        if (dataSourceSetDTO == null) {
            return QueryResultSet.empty();
        }
        DataSourceSet dataSourceSet = DataSourceConverter.convert(dataSourceSetDTO);
        if (dataSourceSet == null) {
            return QueryResultSet.empty();
        }

        return dataSourceSet.query(executeContext, parameter, null, null, null);
    }

    /**
     * 解析出所有撤回任务的BK信息
     */
    private Map<Long, List<Map<String, Object>>> parseWorkItemBK(String[] dataKeys, Map<Long, List<Map<String, Object>>> workItemSubmitDataMap) {
        Map<Long, List<Map<String, Object>>> workItemBKMap = new HashMap<>();

        for (Map.Entry<Long, List<Map<String, Object>>> workItemSubmitData : workItemSubmitDataMap.entrySet()) {
            // 待办项id
            Long workItemId = workItemSubmitData.getKey();
            // bk列表
            List<Map<String, Object>> bkList = this.parseBK(dataKeys, workItemSubmitData.getValue());

            workItemBKMap.put(workItemId, bkList);
        }

        return workItemBKMap;
    }

    /**
     * 从提交的业务数据中解析出BK集合
     */
    private List<Map<String, Object>> parseBK(String[] dataKeys, List<Map<String, Object>> submitDataList) {
        List<Map<String, Object>> bkList = new ArrayList<>(submitDataList.size());
        for (Map<String, Object> submitData : submitDataList) {
            Map<String, Object> BK = new HashMap<>();
            for (String key : dataKeys) {
                BK.put(key, submitData.get(key));
            }
            bkList.add(BK);
        }
        return bkList;
    }

    /**
     * 解析出执行者类型：非代理人、代理人、辅助这行者、超级管理员
     */
    private Integer parsePerformerType(Map<String, Object> actionParas,String pageCode) {
        if(ActivityConstants.PROJECT_DETAIL.equals(pageCode)){
            return 999;
        }
        if (MapUtils.isEmpty(actionParas)) {
            // 非代理人
            return 0;
        }

        Object performerType = actionParas.get("performerType");
        if (null == performerType) {
            // 非代理人
            return 0;
        } else {
            // 0：非代理人，1：代理人，2：辅助执行者，999：超级管理员
            return Integer.valueOf(performerType.toString());
        }
    }

    /**
     * 解析出当前执行者userId
     */
    private String parsePerformerId(Map<String, Object> actionParas, AuthoredUser authoredUser) {
        if (MapUtils.isEmpty(actionParas)) {
            // 非代理人
            return authoredUser.getUserId();
        }

        Object performerId = actionParas.get("agentPerformerId");
        if (null == performerId) {
            // 非代理人
            return authoredUser.getUserId();
        } else {
            return performerId.toString();
        }
    }

    /**
     * 执行关联任务的撤回
     */
    private void executeRelativeTaskWithdraw(String performerId, Integer performerType, List<PtmWithdrawRespDTO> relativeWithdrawTaskList,
                                             String processVarName, String[] dataKeys, Long backlogId, Map<Long, List<Map<String, Object>>> workItemDataMap) {
        // <workItemId, taskId列表>
        Map<Long, Set<Long>> workItemTaskMap = relativeWithdrawTaskList.stream()
                .collect(Collectors.toMap(PtmWithdrawRespDTO::getWorkItemId, ptmWithdrawRespDTO -> {
                    if (CollectionUtils.isNotEmpty(ptmWithdrawRespDTO.getTaskIds())) {
                        return ptmWithdrawRespDTO.getTaskIds();
                    }

                    if (CollectionUtils.isNotEmpty(ptmWithdrawRespDTO.getNextWorkItems())) {
                        return ptmWithdrawRespDTO.getNextWorkItems().stream()
                                .map(PtmWithdrawRespDTO.Task::getWorkItemId)
                                .collect(Collectors.toSet());
                    }
                    return new HashSet();
                }, (newValue, oldValue) -> newValue));

        PtmWithdrawReqDTO withdrawReqDTO = new PtmWithdrawReqDTO();
        withdrawReqDTO.setBacklogId(backlogId);
        withdrawReqDTO.setUniKeys(Arrays.asList(dataKeys));
        withdrawReqDTO.setParameter(new ArrayList<>());

        withdrawReqDTO.setPerformerId(performerId);
        withdrawReqDTO.setPerformerType(performerType);
        withdrawReqDTO.setComment("数据撤回");

        workItemDataMap.forEach((workItemId, submitData) -> {
            PtmWithdrawReqDTO.ParamReqDTO paramReqDTO = new PtmWithdrawReqDTO.ParamReqDTO();
            paramReqDTO.setWorkItemId(workItemId);
            paramReqDTO.setNextWorkItemIdList(workItemTaskMap.get(workItemId));
            paramReqDTO.setData(submitData);

            withdrawReqDTO.getParameter().add(paramReqDTO);
        });
        withdrawReqDTO.setDataVariableKey(processVarName);

//        log.info("[ptm-task-withdraw-{}] atdm访问PTM执行任务撤回, 请求参数：{}", backlogId, JsonUtils.objectToString(withdrawReqDTO));
        commonPtmService.executeRelativeTaskWithdraw(withdrawReqDTO);
    }

    /**
     * 查询需关联撤回的所有任务信息 例如A-B-C-D，B撤回会查询到
     */
    private List<PtmWithdrawRespDTO> queryRelativeWithdrawTask(String tenantId,Long backlogId, String[] dataKeys, Map<Long, List<Map<String, Object>>> workItemDataMap) {
        PtmWithdrawReqDTO withdrawReqDTO = new PtmWithdrawReqDTO();
        withdrawReqDTO.setBacklogId(backlogId);
        withdrawReqDTO.setUniKeys(Arrays.asList(dataKeys));
        withdrawReqDTO.setParameter(new ArrayList<>());

        workItemDataMap.forEach((workItemId, submitData) -> {
            PtmWithdrawReqDTO.ParamReqDTO paramReqDTO = new PtmWithdrawReqDTO.ParamReqDTO();
            paramReqDTO.setWorkItemId(workItemId);
            paramReqDTO.setData(submitData);

            withdrawReqDTO.getParameter().add(paramReqDTO);
        });

        LogDto logDto = new LogDto("执行ptm-task-withdraw：" + backlogId+",请求参数参数:"+JsonUtils.objectToString(withdrawReqDTO), tenantId + LogConstant.TRACE_SEPARATOR + backlogId);
        log.info(logDto.toString());

//        log.info("[ptm-task-withdraw-{}] 查询所有需关联撤回的任务, 请求参数：{}", backlogId, JsonUtils.objectToString(withdrawReqDTO));

        List<PtmWithdrawRespDTO> ptmWithdrawRespDTOS = commonPtmService.queryRelativeWithdrawTask(withdrawReqDTO);
        LogDto resDto = new LogDto("执行ptm-task-withdraw：" + backlogId+",查询所有需关联撤回的任务, 响应内容:"+JsonUtils.objectToString(ptmWithdrawRespDTOS), tenantId + LogConstant.TRACE_SEPARATOR + backlogId);
        log.info(resDto.toString());


//        log.info("[ptm-task-withdraw-{}] 查询所有需关联撤回的任务, 响应内容：{}", backlogId, JsonUtils.objectToString(ptmWithdrawRespDTOS));
        return ptmWithdrawRespDTOS;
    }

    /**
     * 从提交数据中解析出业务数据
     */
    private List<Map<String, Object>> parseSubmitData(String submitVarName, Map<String, Object> data) {
        if (MapUtils.isEmpty(data) || null == data.get(submitVarName)) {
            return new ArrayList<>();
        }

        List<Map<String, Object>> submitDataList = new ArrayList<>();

        Object submitDataObj = data.get(submitVarName);
        // 提交的数据，兼容Map和List两种情况
        if (submitDataObj instanceof Collection) {
            submitDataList.addAll((List<Map<String, Object>>) submitDataObj);
        } else {
            submitDataList.add((Map<String, Object>) submitDataObj);
        }
        return submitDataList;
    }

    /**
     * 检查撤回按钮的配置，其extendParas中必须有：dataKeys、processVariableName、submitVariableName等配置
     */
    private void checkWithdrawAction(SubmitAction action) {
        // 检查“撤回”按钮中是否有主键等等必要配置
        if (MapUtils.isEmpty(action.getExtendParas())
                || null == action.getExtendParas().get("dataKeys")
                || null == action.getExtendParas().get("processVariableName")
                || null == action.getExtendParas().get("submitVariableName")) {
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0033.getErrCode(), messageUtils.getMessage("exception.task.withdraw.lack.necessary.config"));
        }
    }
}
