package com.digiwin.athena.atdm.action.executor;

import com.digiwin.athena.appcore.domain.BaseResultDTO;
import com.digiwin.athena.appcore.exception.BusinessException;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.atdm.ActionConstants;
import com.digiwin.athena.atdm.action.ActionExecutor;
import com.digiwin.athena.atdm.atmc.CommonAtmcService;
import com.digiwin.athena.atdm.constant.ErrorCodeEnum;
import com.digiwin.athena.atdm.datasource.domain.ExecuteResult;
import com.digiwin.athena.atdm.datasource.domain.SubmitAction;
import com.digiwin.athena.atdm.datasource.domain.SubmitExecuteContext;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Component;
import org.springframework.web.client.RestTemplate;

import javax.annotation.Resource;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

/**
 * ApiExecuteErrorActionExecutor Description
 *
 * @author majianfu
 * @date 2021/6/19
 * @since
 */
@Slf4j
@Component
public class ApiExecuteErrorActionExecutor implements ActionExecutor {
    private static final String SUPPORT_KEY = "ATMC_API_EXECUTE_ERROR" + ActionConstants.SPLIT;

    @Resource
    private CommonAtmcService atmcService;

    @Resource
    private RestTemplate restTemplate;

    @Resource
    private MessageUtils messageUtils;

    @Override
    public String supportKey() {
        return SUPPORT_KEY;
    }

    @Override
    public ExecuteResult execute(SubmitExecuteContext executeContext,ExecuteResult parentExecuteResult, SubmitAction action, Map<String, Object> data) {
        if ("atmc.api.execute.error.notify.maintainer".equals(action.getActionId())) {
            notifyMaintainer(executeContext, action);
            return ExecuteResult.ok();
        } else if ("atmc.api.execute.error.solved".equals(action.getActionId())) {
            notifySolved(executeContext, action);
            return ExecuteResult.ok();
        } else {
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0037.getErrCode(), messageUtils.getMessage("exception.unsupport.actionId") + action.getActionId());
        }
    }

    private void notifySolved(SubmitExecuteContext executeContext, SubmitAction action) {
        // 获取全局配置
        if (isNotifySolvedByReExecute()) {
            notifySolvedByReExecute(executeContext, action);
        } else {
            notifySolvedBySendingEmail(executeContext, action);
        }
    }

    private void notifySolvedByReExecute(SubmitExecuteContext executeContext, SubmitAction action) {
        // 关闭任务卡
        closeWorkItem(executeContext, action);
        // 发送重启请求
        reExecute(executeContext, action);
    }

    private void reExecute(SubmitExecuteContext executeContext, SubmitAction action) {
        Map<String, Object> reExecuteApi = Optional.ofNullable(executeContext.getBpmData())
                .map(bpmData -> (Map<String, Object>) (bpmData.get("reExecute")))
                .orElse(new HashMap<>());
        if (MapUtils.isEmpty(reExecuteApi)) {
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0038.getErrCode(), messageUtils.getMessage("exception.lack.restart.interface.info"));
        }

        String url = String.valueOf(reExecuteApi.get("url"));
        Map<String, String> headerMap = (Map<String, String>) reExecuteApi.get("headers");

        HttpHeaders headers = new HttpHeaders();
        if (MapUtils.isNotEmpty(headerMap)) {
            for (Map.Entry<String, String> entry : headerMap.entrySet()) {
                headers.add(entry.getKey(), entry.getValue());
            }
        }
        HttpEntity httpEntity = new HttpEntity<>(reExecuteApi.get("request"), headers);
        try {
            ResponseEntity<BaseResultDTO> respEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity
                    , new ParameterizedTypeReference<BaseResultDTO>() {
                    });
            respEntity.getBody().getResponseWithException("");
        } catch (Exception ex) {
            log.error("body: {}, error: ", JsonUtils.objectToString(reExecuteApi.get("request")), ex);
            throw ex;
//            String error = String.format(messageUtils.getMessage("exception.restart.interface.error") +
//                    ",%s--%s" ,url,ex.getMessage());
//            throw  BusinessException.create(599,"599",error,error,"",ex);
        }
    }

    private void closeWorkItem(SubmitExecuteContext executeContext, SubmitAction action) {
        atmcService.closeApiExecuteErrorWorkItem(executeContext.getProcessSerialNumber());
    }

    private void notifySolvedBySendingEmail(SubmitExecuteContext executeContext, SubmitAction action) {
//        try {
            Object emailData = getEmailDataWithException(executeContext);
            atmcService.sendEmailToHandleApiExecuteError("notifySolved", emailData);
//        } catch (Exception ex) {
//            String error = messageUtils.getMessage("exception.send.email.fail");
//            throw  BusinessException.create(599,"599",error,error,"",ex);
//        }
    }

    private boolean isNotifySolvedByReExecute() {
        List<Map> configs = atmcService.getConfig(Collections.singletonList("notifySolvedByReExecute"));
        if (CollectionUtils.isEmpty(configs) || null == configs.get(0)) {
            return false;
        }

        return StringUtils.equalsIgnoreCase("true", String.valueOf(configs.get(0).get("value")));
    }

    private void notifyMaintainer(SubmitExecuteContext executeContext, SubmitAction action) {
//        try {
            Object emailData = getEmailDataWithException(executeContext);
            atmcService.sendEmailToHandleApiExecuteError("notifyMaintainer", emailData);
//        } catch (Exception ex) {
//            throw BusinessException.create(messageUtils.getMessage("exception.send.notice.DevOps.email.fail"), ex);
//        }
    }

    private Object getEmailDataWithException(SubmitExecuteContext executeContext) {
        // 直接发送邮件通知运维
        Optional<Object> emailData = Optional.ofNullable(executeContext.getBpmData())
                .map(bpmData -> bpmData.get("data"))
                .map(data -> ((Map<String, Object>) data).get("emailData"));
        if (emailData.isPresent()) {
            return emailData.get();
        } else {
            throw  BusinessException.create(ErrorCodeEnum.NUM_500_0039.getErrCode(), messageUtils.getMessage("exception.lack.email.param"));
        }
    }
}
