package com.digiwin.athena.atdm.action.executor;

import com.digiwin.athena.appcore.exception.BusinessException;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.atdm.ActionConstants;
import com.digiwin.athena.atdm.UiBotConstants;
import com.digiwin.athena.atdm.action.dto.ManualTaskConsistencyResp;
import com.digiwin.athena.atdm.action.ActionExecutor;
import com.digiwin.athena.atdm.atmc.CommonAtmcService;
import com.digiwin.athena.atdm.constant.ErrorCodeEnum;
import com.digiwin.athena.atdm.datasource.domain.ExecuteResult;
import com.digiwin.athena.atdm.datasource.domain.SubmitAction;
import com.digiwin.athena.atdm.datasource.domain.SubmitExecuteContext;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.ArrayUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.annotation.Order;
import org.springframework.stereotype.Service;
import org.springframework.util.PropertyPlaceholderHelper;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;

/**
 * ManualTaskActionExecutor Description
 *
 * @author majianfu
 * @date 2022/1/11
 * @since
 */
@Order(0)
@Service("ManualTaskActionExecutor")
@Slf4j
public class ManualTaskActionExecutor implements ActionExecutor {
    private static final PropertyPlaceholderHelper PLACEHOLDER_HELPER = new PropertyPlaceholderHelper("${", "}", (String) null, true);

    private static final String ACTION_SUBMIT = "uibot_manual.task.submit";

    private static final String ACTION_DELETE = "uibot_manual.task.delete";

    private static final String ACTION_EDIT = "uibot_manual.task.edit";

    private static final int SOURCE_TYPE_MANUAL = 1;

    private static final int STATE_CLOSE = 3;

    private static final int SUB_STATE_DELETE = 9999;

    private static final List<String> MANUAL_TASK_ACTION_TYPE_LIST = Arrays.asList("manual-task-submit", "manual-task-edit", "manual-task-delete");

    @Autowired
    private CommonAtmcService atmcService;

    @Autowired
    private MessageUtils messageUtils;

    @Override
    public String supportKey() {
        return UiBotConstants.ACTION_CATEGORY_MANUAL_TASK + ActionConstants.SPLIT;
    }

    /**
     * 前置执行处理：根据执行人数量，分裂相应数据量的提交data
     *
     * @param executeContext
     * @param action
     * @param data
     */
    public void preExecute(SubmitExecuteContext executeContext, SubmitAction action, Map<String, Object> data) {
        if (null == action || !MANUAL_TASK_ACTION_TYPE_LIST.contains(action.getActionType())) {
            return;
        }

        if (MapUtils.isEmpty(data) || null == data.get("data")) {
            return;
        }

        Map<String, Object> submitData = (Map<String, Object>) data.get("data");
        if (MapUtils.isEmpty(submitData) || null == submitData.get("executorList")) {
            return;
        }

        List<Map<String, String>> executorList = (List<Map<String, String>>) submitData.get("executorList");
        checkManualTaskConsistency(executorList, submitData, executeContext.getBacklogId());
        if (null == executorList) {
            // 老数据的提交，不做任何处理
        }
        // 单执行人，直接将executorList平铺
        else if (1 == executorList.size()) {
            submitData.remove("executorList");
            submitData.putAll(executorList.get(0));
        }
        // 多个执行人时，才需要分裂处理
        else if (2 <= executorList.size()) {
            // 将submitData转换为List，重新赋值给submitAction.data.data中
            List<Map<String, Object>> newSubmitData = new ArrayList<>();
            for (Map<String, String> executor : executorList) {
                Map<String, Object> subData = new HashMap<>();
                subData.putAll(submitData);
                subData.remove("executorList");
                subData.putAll(executor);
                newSubmitData.add(subData);
            }
            data.put("data", newSubmitData);
        }
    }

    private void checkManualTaskConsistency(List<Map<String, String>> executorList, Map<String, Object> submitData, Long backlogId) {
        String executorUserId = "executorUserId";
        Long teamId;
        try {
            teamId = Long.valueOf(submitData.get("teamId").toString());
        } catch (Exception e) {
            log.info(" no teamId");
            return;
        }
        ManualTaskConsistencyResp freshData = atmcService.checkConsistency(teamId, backlogId);
        if (!freshData.getItemBelongsToTeam()) {
            throw BusinessException.create(ErrorCodeEnum.BUSINESS_600_0004.getErrCode(), messageUtils.getMessage("manual.task.invalid.data"));
        }
        executorList.removeIf(e -> !freshData.getTeamMemberUserIds().contains(e.get(executorUserId)));
    }

    @Override
    public ExecuteResult execute(SubmitExecuteContext executeContext, ExecuteResult parentExecuteResult, SubmitAction action, Map<String, Object> data) {
        // data中必须只包含 稳敏太API的响应数据 + 提交数据
        if (null != data && data.size() > 2) {
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0013.getErrCode(), messageUtils.getMessage("exception.calendar.activity.data.error"));
        }

        // 提交：创建 or 提交
        if (StringUtils.equals(ACTION_SUBMIT, action.getActionId())) {
            Optional<Long> backlogIdOpt = Optional.ofNullable(action.getExecuteContext()).map(SubmitExecuteContext::getBacklogId);
            if (backlogIdOpt.isPresent()) {
                updateManualTask(executeContext, action, data);
            } else {
                createManualTask(executeContext, action, data);//NOSONAR
            }
        }
        // 编辑
        else if (StringUtils.equals(ACTION_EDIT, action.getActionId())) {
            editManualTask(executeContext, action, data);
        }
        // 删除
        else if (StringUtils.equals(ACTION_DELETE, action.getActionId())) {
            deleteManualTask(executeContext, action, data);
        } else {
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0014.getErrCode(), "");
        }

        //todo:这里有bug，只能处理2层嵌套，如果是多层，则有问题
        if (parentExecuteResult != null) {
            //兼容下以前的逻辑，后续删除
            if (parentExecuteResult.getRequestData() == null) {
                parentExecuteResult.setRequestData((HashMap) data);
            } else {
                parentExecuteResult.getRequestData().putAll((HashMap) data);
            }
            if (parentExecuteResult.getData() == null) {
                parentExecuteResult.setData((HashMap) data);
            } else {
                parentExecuteResult.getData().putAll((HashMap) data);
            }
            return ExecuteResult.ok();
        } else {
            ExecuteResult result = ExecuteResult.withData((HashMap) data);
            //兼容下以前的逻辑，后续删除
            if (result.getRequestData() == null) {
                result.setRequestData((HashMap) data);
            } else {
                result.getRequestData().putAll((HashMap) data);
            }
            return result;
        }
    }

    private void editManualTask(SubmitExecuteContext executeContext, SubmitAction action, Map<String, Object> data) {
        String customTaskName = parseCustomTaskName(executeContext.getParentAction(), data);
        Map<String, Object> submitDataMap = mergeActionResult(data);
        if (StringUtils.isNotBlank(customTaskName)) {
            submitDataMap.put("taskName", customTaskName);
        }

        submitDataMap.put("_planEndTime", submitDataMap.get("endTime"));
        submitDataMap.put("tmActivityId", submitDataMap.get("taskId"));

        Map<String, Object> json = new HashMap<>();
        json.put("data", submitDataMap);
        json.put("createTime", submitDataMap.get("startTime"));

        // 编辑时，勾选“已执行”
        Object executeState = submitDataMap.get("executeState");
        if (null != executeState && Objects.equals("1", String.valueOf(executeState))) {
            json.put("state", STATE_CLOSE);
            json.put("subState", 10);
        } else {
            json.put("state", 1);
            json.put("subState", 0);
        }

        // 手动任务
        json.put("sourceType", 1);
        json.put("actionId", ACTION_EDIT);
        json.put("teamId", submitDataMap.get("teamId"));
        json.put("workItemId", action.getExecuteContext().getBacklogId());

        json.put("memberUserId", submitDataMap.get("executorUserId"));
        json.put("memberUserName", submitDataMap.get("executorUserName"));

        atmcService.manualTaskStateChanged(Collections.singletonList(json));
    }

    private void deleteManualTask(SubmitExecuteContext executeContext, SubmitAction action, Map<String, Object> data) {
        Map<String, Object> submitDataMap = mergeActionResult(data);

        Map<String, Object> json = new HashMap<>();
        json.put("data", submitDataMap);

        json.put("state", STATE_CLOSE);
        json.put("subState", SUB_STATE_DELETE);

        // 手动任务
        json.put("sourceType", SOURCE_TYPE_MANUAL);
        json.put("actionId", ACTION_DELETE);
        json.put("teamId", submitDataMap.get("teamId"));
        json.put("workItemId", action.getExecuteContext().getBacklogId());

        json.put("memberUserId", submitDataMap.get("executorUserId"));
        json.put("memberUserName", submitDataMap.get("executorUserName"));

        atmcService.manualTaskStateChanged(Collections.singletonList(json));
    }

    private void updateManualTask(SubmitExecuteContext executeContext, SubmitAction action, Map<String, Object> data) {
        String customTaskName = parseCustomTaskName(executeContext.getParentAction(), data);
        Map<String, Object> submitDataMap = mergeActionResult(data);
        if (StringUtils.isNotBlank(customTaskName)) {
            submitDataMap.put("taskName", customTaskName);
        }

        Map<String, Object> json = new HashMap<>();
        json.put("data", submitDataMap);

        json.put("state", STATE_CLOSE);
        json.put("subState", 10);

        // 手动任务
        json.put("sourceType", 1);
        json.put("actionId", ACTION_SUBMIT);
        json.put("teamId", submitDataMap.get("teamId"));
        json.put("workItemId", action.getExecuteContext().getBacklogId());

        json.put("memberUserId", submitDataMap.get("executorUserId"));
        json.put("memberUserName", submitDataMap.get("executorUserName"));


        atmcService.manualTaskStateChanged(Collections.singletonList(json));
    }

    private Map<String, Object> mergeActionResult(Map<String, Object> data) {
        Map<String, Object> submitDataMap = (Map<String, Object>) data.get("data");
        for (Map.Entry<String, Object> entry : data.entrySet()) {
            if (!"data".equals(entry.getKey())) {
                submitDataMap.put(entry.getKey(), entry.getValue());
            }
        }
        return submitDataMap;
    }

    private String parseCustomTaskName(SubmitAction parentAction, Map<String, Object> data) {
        String taskNameExpression = Optional.ofNullable(parentAction)
                .map(action -> action.getExtendParas())
                .map(extendParas -> (String) extendParas.get("calendarTaskNameExpression"))
                .orElse("");
        if (StringUtils.isBlank(taskNameExpression)) {
            return "";
        }

        /**
         * data的数据结构为：
         * {
         *     "data": [
         *          {
         *              "xxx": "xxx",
         *              "calendar_task_info": [
         *                  {....;// 稳敏太API请求内容}
         *              ]
         *          }
         *     ],
         *     "calendar_task_info": [
         *          {....;// 稳敏太API响应内容}
         *     ]
         * }
         */
        return PLACEHOLDER_HELPER.replacePlaceholders(taskNameExpression, placeholderName -> {
            if (StringUtils.isBlank(placeholderName)) {
                return "";
            }
            // 优先在提交请求数据中寻找；再在稳敏太返回内容中寻找
            // data.data 为行事历任务提交的请求内容
            Map<String, Object> requestPlanMap = new HashMap<>();
            flatData(requestPlanMap, data.get("data"), "");
            Object placeholderValue = requestPlanMap.get(placeholderName);

            String finalValue = String.valueOf(placeholderValue);
            if (null != placeholderValue && StringUtils.isNotBlank(finalValue)) {
                return finalValue;
            }

            // 非data.data 为稳敏太API的返回内容
            Map<String, Object> responseMap = new HashMap<>(data);
            // 踢掉 行事历任务提交的请求内容（data），在上一步已经尝试解析过placeholderName了
            responseMap.remove("data");
            Map<String, Object> responsePlanMap = new HashMap<>();
            flatData(responsePlanMap, responseMap, "");
            placeholderValue = responsePlanMap.get(placeholderName);
            finalValue = String.valueOf(placeholderValue);
            if (null != placeholderValue && StringUtils.isNotBlank(finalValue)) {
                return String.valueOf(placeholderValue);
            }
            return "";
        });
    }

    private void flatData(Map<String, Object> planMap, Object data, String keyPrefix) {
        if (null == data) {
            return;
        }

        if (data instanceof Map) {
            Map dataMap = (Map) data;
            Set<Map.Entry> entrySet = dataMap.entrySet();
            for (Map.Entry entry : entrySet) {
                String path = StringUtils.isNotBlank(keyPrefix) ? keyPrefix + "." + String.valueOf(entry.getKey()) : String.valueOf(entry.getKey());
                flatData(planMap, entry.getValue(), path);
            }
        } else if (data instanceof Collection) {
            Collection collection = (Collection) data;
            if (CollectionUtils.isNotEmpty(collection)) {
                for (Object element : collection) {
                    // 取第一个非空元素
                    if (null != element) {
                        flatData(planMap, element, keyPrefix);
                        break;
                    }
                }
            }
        } else if (data.getClass().isArray()) {
            Object[] array = (Object[]) data;
            if (ArrayUtils.isNotEmpty(array)) {
                for (Object element : array) {
                    // 取第一个非空元素
                    if (null != element) {
                        flatData(planMap, element, keyPrefix);
                        break;
                    }
                }
            }
        } else {
            planMap.put(keyPrefix, data);
        }
    }

    private void createManualTask(SubmitExecuteContext executeContext, SubmitAction action, Map<String, Object> data) {
        if (data.get("data") instanceof List) {
            String parentRespDataKey = null;
            // <executorEmpId, parentResp>
            Map<String, Map<String, Object>> parentRespDataMap = new HashMap<>();
            parentRespDataKey = flatParentRespByExecutorNo(data, parentRespDataMap);

            List<Map<String, Object>> manualTaskDataList = (List<Map<String, Object>>) data.get("data");
            // 将data.data（ArrayList)
            List<Map<String, Object>> manualTaskList = new ArrayList<>();
            for (Map<String, Object> manualTaskData : manualTaskDataList) {
                Map<String, Object> tmpSubmitAndParentRespData = new HashMap<>();
                tmpSubmitAndParentRespData.put("data", manualTaskData);

                // 塞入应用API为该执行人任务创建的主键
                String executorEmpId = manualTaskData.get("executorEmpId").toString();
                if (StringUtils.isNotBlank(parentRespDataKey)) {
                    tmpSubmitAndParentRespData.put(parentRespDataKey, Collections.singletonList(parentRespDataMap.get(executorEmpId)));
                }

                Map<String, Object> manualTask = createManualTaskData(executeContext, action, tmpSubmitAndParentRespData);
                manualTaskList.add(manualTask);
            }

            atmcService.manualTaskStateChanged(manualTaskList).getResponse();
            // 无需返回backlogId信息
        }
        // data.data（Map）
        else {
            Map<String, Object> manualTask = createManualTaskData(executeContext, action, data);

            Long backlogId = (Long) atmcService.manualTaskStateChanged(Collections.singletonList(manualTask)).getResponse();
            ((Map<String, Object>) manualTask.get("data")).put("backlogId", backlogId);
        }
    }

    private String flatParentRespByExecutorNo(Map<String, Object> data, Map<String, Map<String, Object>> parentRespDataMap) {
        if (data.size() <= 1) {
            return null;
        }

        String parentRespDataKey = null;
        for (Map.Entry<String, Object> entry : data.entrySet()) {
            if (StringUtils.equals(entry.getKey(), "data")) {
                continue;
            }

            parentRespDataKey = entry.getKey();
            // 稳敏太返回的不是List，则报错
            if (!(entry.getValue() instanceof List)) {
                throw BusinessException.create(ErrorCodeEnum.NUM_500_0015.getErrCode(), messageUtils.getMessage("exception.create.calendar.activity.api.result"));
            }

            List<Map<String, Object>> parentRespDataList = (List<Map<String, Object>>) entry.getValue();
            if (CollectionUtils.isNotEmpty(parentRespDataList)) {
                for (Map<String, Object> parentRespData : parentRespDataList) {
                    if (null == parentRespData.get("executor_no")) {
                        String errMsg = messageUtils.getMessage("manual.task.not.support.multi.executor");
                        throw BusinessException.create(ErrorCodeEnum.BUSINESS_600_0001.getErrCode(), errMsg);
                    }
                    parentRespDataMap.put(parentRespData.get("executor_no").toString(), parentRespData);
                }
            }
        }

        return parentRespDataKey;
    }

    private Map<String, Object> createManualTaskData(SubmitExecuteContext executeContext, SubmitAction action, Map<String, Object> data) {
        String customTaskName = parseCustomTaskName(executeContext.getParentAction(), data);
        Map<String, Object> submitDataMap = mergeActionResult(data);
        submitDataMap.put("tmActivityName", submitDataMap.get("taskName"));
        if (StringUtils.isNotBlank(customTaskName)) {
            submitDataMap.put("taskName", customTaskName);
        }

        submitDataMap.put("_planEndTime", submitDataMap.get("endTime"));
        submitDataMap.put("tmActivityId", submitDataMap.get("taskId"));

        Map<String, Object> json = new HashMap<>();
        json.put("data", submitDataMap);
        json.put("createTime", submitDataMap.get("startTime"));

        // 已完成
        Object executeState = submitDataMap.get("executeState");
        if (null != executeState && Objects.equals("1", String.valueOf(executeState))) {
            json.put("state", STATE_CLOSE);
            json.put("subState", 10);
        } else {
            json.put("state", 1);
            json.put("subState", 0);
        }

        // 手动任务
        json.put("sourceType", 1);
        json.put("actionId", ACTION_SUBMIT);
        json.put("teamId", submitDataMap.get("teamId"));

        json.put("memberUserId", submitDataMap.get("executorUserId"));
        json.put("memberUserName", submitDataMap.get("executorUserName"));

        return json;
    }
}
