package com.digiwin.athena.atdm.action.executor;

import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.domain.log.LogDto;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.digiwin.athena.atdm.ActionConstants;
import com.digiwin.athena.atdm.UiBotConstants;
import com.digiwin.athena.atdm.ptm.dto.PtmWithdrawCrossBkReqDTO;
import com.digiwin.athena.atdm.ptm.dto.PtmWorkItemDTO;
import com.digiwin.athena.atdm.action.ActionExecutor;
import com.digiwin.athena.atdm.ptm.CommonPtmService;
import com.digiwin.athena.atdm.uibot.CommonUiBotService;
import com.digiwin.athena.atdm.activity.ActivityConstants;
import com.digiwin.athena.atdm.activity.service.BpmServiceInvokeType;
import com.digiwin.athena.atdm.datasource.domain.*;
import com.digiwin.athena.atdm.util.TrackingUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.annotation.Order;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * TaskWithdrawCrossBkExecutor Description
 *
 * @author lisheng
 * @date 2024/03/29
 * @since
 */
@Slf4j
@Order(102)
@Service("uibot_task_withdraw_cross_bk")
public class TaskWithdrawCrossBkExecutor implements ActionExecutor {
    @Autowired
    private CommonPtmService commonPtmService;

    @Autowired
    private CommonUiBotService commonUiBotService;

    @Autowired
    private DataSubmissionService dataSubmissionService;
    @Autowired
    private TrackingUtil trackingUtil;

    /**
     * 任务状态：关闭
     */
    private static final int STATE_CLOSE = 3;

    @Override
    public String supportKey() {
        return UiBotConstants.ACTION_CATEGORY_TASK_ENGINE + ActionConstants.SPLIT + BpmServiceInvokeType.TaskWithdrawCrossBk.getName();
    }

    @Override
    public boolean hasActionMetadata() {
        return false;
    }

    @Override
    public ExecuteResult execute(SubmitExecuteContext executeContext, ExecuteResult parentExecuteResult, SubmitAction action, Map<String, Object> parameter) {
        LogDto logDto = new LogDto("[ptm-task-crossbk-withdraw] 入参：" + JsonUtils.objectToString(action));
        log.info(logDto.toString());

        // 当前登录用户
        AuthoredUser authoredUser = executeContext.getAuthoredUser();
        // 当前执行者userId
        String performerId = this.parsePerformerId(action.getParas(), authoredUser);
        // 当前执行者类型：执行者、代理人、辅助执行者、超级管理员
        Integer performerType = this.parsePerformerType(action.getParas(),executeContext.getPageCode());

        // 1、执行PTM撤回接口 当前任务 + 后置关联任务的撤回
        List<PtmWorkItemDTO> withdrawReqDTOList = this.executeRelativeTaskWithdrawCrossBk(performerId, performerType, action);

        // 2、校验关联的被撤回任务是否需要关闭
        checkRelativeTaskCompleted(authoredUser, withdrawReqDTOList);
        return ExecuteResult.ok();
    }

    /**
     * 校验关联的被撤回任务是否需要关闭
     * @param authoredUser
     * @param needParseRelativeTaskList
     */
    private void checkRelativeTaskCompleted(AuthoredUser authoredUser, List<PtmWorkItemDTO> needParseRelativeTaskList) {
        if (CollectionUtils.isEmpty(needParseRelativeTaskList)) {
            return;
        }
        // 任务不是进行中的，直接不需要处理，因为本身已关闭
        needParseRelativeTaskList = needParseRelativeTaskList.stream().filter(ptmWorkItemDTO -> STATE_CLOSE > ptmWorkItemDTO.getState()).collect(Collectors.toList());

        if (CollectionUtils.isEmpty(needParseRelativeTaskList)) {
            return;
        }

        // 解析关联撤回任务的PageDefine定义
        Map<Long, List<Map>> workItemPageDefineMap = commonUiBotService.queryTaskCrossBkWithdrawPageDefine(needParseRelativeTaskList);
        if (MapUtils.isEmpty(workItemPageDefineMap)) {
            return;
        }

        // 执行数据源查询
        for (Map.Entry<Long, List<Map>> workItemPageDefine : workItemPageDefineMap.entrySet()) {
            Long workItemId = workItemPageDefine.getKey();
            List<Map> pageDefines = workItemPageDefine.getValue();
            if (CollectionUtils.isEmpty(pageDefines)) {
                continue;
            }

            SubmitAction dispatchAction = null;
            // 一般而言，一条数据要么在待处理、要么在已完成；只要从任意一个标签中查询到数据，另外一个标签就无需在查询了
            for (Map pageDefine : pageDefines) {
                // 必须包含executeContext，dataSourceSet，submitActions
                if (null == pageDefine || null == pageDefine.get("executeContext") || null == pageDefine.get("dataSourceSet")) {
                    log.error("[ptm-task-crossbk-withdraw{}] 未能成功解析出页签定义", workItemId);
                    continue;
                }

                ExecuteContext executeContext = convertJson(pageDefine.get("executeContext"), ExecuteContext.class);
                executeContext.setAuthoredUser(authoredUser);
                if (StringUtils.isBlank(executeContext.getTenantId())) {
                    executeContext.setTenantId(executeContext.getAuthoredUser().getTenantId());
                }

                if (null != pageDefine.get("submitActions") && CollectionUtils.isNotEmpty((List) pageDefine.get("submitActions"))) {
                    dispatchAction = JsonUtils.jsonToObject(JsonUtils.objectToString(((List) pageDefine.get("submitActions")).get(0)), SubmitAction.class);
                }
            }

            // 执行checkCompleteAction + terminate-task
            if (null != dispatchAction) {
                if (null == dispatchAction.getExecuteContext().getAuthoredUser()) {
                    dispatchAction.getExecuteContext().setAuthoredUser(authoredUser);
                }
                if (StringUtils.isBlank(dispatchAction.getExecuteContext().getTenantId())) {
                    dispatchAction.getExecuteContext().setTenantId(authoredUser.getTenantId());
                }
//                log.info("[ptm-task-withdraw-{}-{}] 执行checkCompleteAction + dispatchAction: {}", backlogId, workItemId, JsonUtils.objectToString(dispatchAction));
                // 待关闭的任务没有业务数据，所以传空map
                dataSubmissionService.submit(dispatchAction.getExecuteContext(), dispatchAction, new HashMap<>());
            }
        }
    }

    /**
     * 解析出执行者类型：非代理人、代理人、辅助这行者、超级管理员
     */
    private Integer parsePerformerType(Map<String, Object> actionParas,String pageCode) {
        if(ActivityConstants.PROJECT_DETAIL.equals(pageCode)){
            return 999;
        }
        if (MapUtils.isEmpty(actionParas)) {
            // 非代理人
            return 0;
        }

        Object performerType = actionParas.get("performerType");
        if (null == performerType) {
            // 非代理人
            return 0;
        } else {
            // 0：非代理人，1：代理人，2：辅助执行者，999：超级管理员
            return Integer.valueOf(performerType.toString());
        }
    }

    /**
     * 解析出当前执行者userId
     */
    private String parsePerformerId(Map<String, Object> actionParas, AuthoredUser authoredUser) {
        if (MapUtils.isEmpty(actionParas)) {
            // 非代理人
            return authoredUser.getUserId();
        }

        Object performerId = actionParas.get("agentPerformerId");
        if (null == performerId) {
            // 非代理人
            return authoredUser.getUserId();
        } else {
            return performerId.toString();
        }
    }

    /**
     * 调用PTM 跨BK撤回
     * @param performerId
     * @param performerType
     * @param action
     * @return
     */
    private List<PtmWorkItemDTO> executeRelativeTaskWithdrawCrossBk(String performerId, Integer performerType, SubmitAction action) {
        Map<String, Object> rowParas = action.getParas();
        ActionServiceId serviceId = action.getServiceId() == null ? ActionServiceId.empty() : action.getServiceId();

        List<PtmWorkItemDTO> uiBotTaskCrossBkWithdrawReqDTOList = new ArrayList<>();
        if(MapUtils.isEmpty(rowParas) || null == rowParas.get("params") || StringUtils.isEmpty(serviceId.getServiceUri())){
            return uiBotTaskCrossBkWithdrawReqDTOList;
        }
        String workitemId = String.valueOf(rowParas.get("workitemId"));
        List<PtmWithdrawCrossBkReqDTO> retrieveReqDTOList = JsonUtils.jsonToListObject(JsonUtils.objectToString(rowParas.get("params")),PtmWithdrawCrossBkReqDTO.class);

        if(StringUtils.isEmpty(workitemId) ||CollectionUtils.isEmpty(retrieveReqDTOList)){
            return uiBotTaskCrossBkWithdrawReqDTOList;
        }

        // 只撤回合并任务前的数据
        retrieveReqDTOList = retrieveReqDTOList.stream().filter(ptmWithdrawCrossBkReqDTO -> workitemId.equals(ptmWithdrawCrossBkReqDTO.getWorkitemId())).collect(Collectors.toList());
        rowParas.put("params", retrieveReqDTOList);

        retrieveReqDTOList.forEach(withdrawReqDTO -> {
            withdrawReqDTO.setPerformerId(performerId);
            withdrawReqDTO.setPerformerType(performerType);
            withdrawReqDTO.setComment("数据撤回");
        });
        LogDto logDto = new LogDto("[ptm-task-withdraw-{}] atdm访问PTM执行任务撤回, 请求参数:" + JsonUtils.objectToString(retrieveReqDTOList));
        log.info(logDto.toString());
        List<PtmWorkItemDTO> ptmWorkItemDTOS = commonPtmService.executeRelativeTaskWithdrawCrossBk(retrieveReqDTOList, serviceId.getServiceUri());
        // 跨BK撤回 增加埋点
        if (CollectionUtils.isNotEmpty(ptmWorkItemDTOS)) {
            String code = ptmWorkItemDTOS.stream().map(PtmWorkItemDTO::getTaskDefCode).collect(Collectors.joining(","));
            trackingUtil.reportUserTrack("task", code, "跨BK撤回");
        }
        return ptmWorkItemDTOS;
    }

    private <T> T convertJson(Object param, Class<T> clz) {
        return JsonUtils.jsonToObject(JsonUtils.objectToString(param), clz);
    }
}
