package com.digiwin.athena.atdm.action.executor;

import com.digiwin.athena.appcore.exception.BusinessException;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.atdm.ActionConstants;
import com.digiwin.athena.atdm.AttachmentConstants;
import com.digiwin.athena.atdm.UiBotConstants;
import com.digiwin.athena.atdm.aam.CommonAttachmentService;
import com.digiwin.athena.atdm.aam.dto.DmcAccount;
import com.digiwin.athena.atdm.action.ActionExecutor;
import com.digiwin.athena.atdm.activity.domain.SubmitActionDTO;
import com.digiwin.athena.atdm.constant.ErrorCodeEnum;
import com.digiwin.athena.atdm.datasource.domain.ExecuteResult;
import com.digiwin.athena.atdm.datasource.domain.SubmitAction;
import com.digiwin.athena.atdm.datasource.domain.SubmitExecuteContext;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

/**
 * AttachmentActionExecutor Description
 *
 * @author majianfu
 * @date 2022/3/4
 * @since
 */
@Service("attachmentActionExecutor")
@Slf4j
public class AttachmentActionExecutor implements ActionExecutor {
    @Resource
    private CommonAttachmentService commonAttachmentService;

    @Autowired
    private MessageUtils messageUtils;

    @Override
    public String supportKey() {
        return UiBotConstants.ACTION_CATEGORY_UIBOT + ActionConstants.SPLIT + "upload-or-save-attachment";
    }

    @Override
    public boolean hasActionMetadata() {
        return false;
    }

    @Override
    public ExecuteResult execute(SubmitExecuteContext executeContext, ExecuteResult parentExecuteResult, SubmitAction action, Map<String, Object> data) {
        Map<String, Object> extendParas = action.getExtendParas();
        checkActionExtendParas(extendParas);

        String submitVariableName = (String) extendParas.get("submitVariableName");
        // 平台附件提交后需要处理的字段列表
        List<String> targetSchemaList = (List<String>) extendParas.get("targetSchemaList");
        // 需要删除的API级别附件
        List<Map> needDeleteFilesAPI = new ArrayList<>();
        if (null != extendParas.get(AttachmentConstants.NEED_DELETE_FILES_API)) {
            needDeleteFilesAPI = (List<Map>) extendParas.get(AttachmentConstants.NEED_DELETE_FILES_API);
        }

        // 应用dmc账密
        Map<String, DmcAccount> schemaDmcAccountMap = new HashMap<>();
        if (null != extendParas.get(AttachmentConstants.SCHEMA_DMC_ACCOUNT_MAP)) {
            schemaDmcAccountMap = (Map<String, DmcAccount>) extendParas.get(AttachmentConstants.SCHEMA_DMC_ACCOUNT_MAP);
        }
        // 获取附件字段名称
        List<Map> uploadedFiles = new ArrayList<>();
        List<Map> deletedFiles = new ArrayList<>();
        // 检查附件标志，分类新增、删除附加列表
        if (CollectionUtils.isNotEmpty(targetSchemaList)) {
            classifyFiles(data.get(submitVariableName), targetSchemaList, uploadedFiles, deletedFiles);
        }


        // 调用aam接口
        if (CollectionUtils.isNotEmpty(uploadedFiles) || CollectionUtils.isNotEmpty(deletedFiles)) {
            commonAttachmentService.uploadOrDeleteAttachment(uploadedFiles, deletedFiles, Boolean.TRUE);
        }

        // API级附件删除
        if (CollectionUtils.isNotEmpty(needDeleteFilesAPI)) {
            // 调用aam接口
            commonAttachmentService.deleteAttachment(needDeleteFilesAPI, schemaDmcAccountMap);
        }
        return ExecuteResult.ok();
    }

    private void classifyFiles(Object submitData, List<String> targetSchemaList, List<Map> uploadedFiles, List<Map> deletedFiles) {
        // submitData必须是Map 或者 集合类型
        if (submitData instanceof Map) {
            Map<String, Object> data = (Map<String, Object>) submitData;
            for (String targetSchema : targetSchemaList) {
                classifyFiles(data, targetSchema, uploadedFiles, deletedFiles);
            }
        } else {
            List<Map<String, Object>> dataList = (List<Map<String, Object>>) submitData;
            if (CollectionUtils.isNotEmpty(dataList)) {
                for (Map<String, Object> data : dataList) {
                    for (String targetSchema : targetSchemaList) {
                        classifyFiles(data, targetSchema, uploadedFiles, deletedFiles);
                    }
                }
            }
        }
    }

    private void classifyFiles(Map submitData, String targetSchema, List<Map> uploadedFiles, List<Map> deletedFiles) {
        if (MapUtils.isEmpty(submitData) || null == submitData.get(targetSchema)) {
            return;
        }

        Map attachment = (Map) submitData.get(targetSchema);
        List<Map> attachmentDataList = (List<Map>) attachment.get("data");
        if (CollectionUtils.isEmpty(attachmentDataList)) {
            return;
        }

        for (Map attachmentData : attachmentDataList) {
            String status = (String) attachmentData.get("uibot__status");
            if (StringUtils.equalsIgnoreCase(status, "NEW")) {
                uploadedFiles.add(attachmentData);
            } else if (StringUtils.equalsIgnoreCase(status, "DELETE")) {
                deletedFiles.add(attachmentData);
            }
        }
    }

    private void checkActionExtendParas(Map<String, Object> extendParas) {
        if (MapUtils.isEmpty(extendParas) || null == extendParas.get("submitVariableName") || null == extendParas.get("targetSchemaList")) {
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0043.getErrCode(), messageUtils.getMessage("exception.effectAfterSubmit.config.empty"));
        }

        String submitVariableName = (String) extendParas.get("submitVariableName");
        if (StringUtils.isBlank(submitVariableName)) {
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0043.getErrCode(), messageUtils.getMessage("exception.effectAfterSubmit.config.empty"));
        }
    }

    /**
     * 预处理API级别附件
     *
     * @param submitActionList
     */
    public static void preprocessApiAttachment(List<SubmitActionDTO> submitActionList) {
        for (SubmitActionDTO submitActionDTO : submitActionList) {
            SubmitAction action = submitActionDTO.getAction();

            // 1、解析出“upload-or-save-attachm” action
            SubmitAction attachementAction = parseAction(AttachmentConstants.ATTACHMENT_ACTION_ID, action);
            if (attachementAction == null) {
                continue;
            }
            Map<String, Object> extendParas = attachementAction.getExtendParas();
            if (MapUtils.isEmpty(extendParas) || null == extendParas.get(AttachmentConstants.SUBMIT_VARIABLE_NMAE) || null == extendParas.get(AttachmentConstants.TARGET_SCHEMA_LIST_API)) {
                continue;
            }
            String submitVariableName = (String) extendParas.get(AttachmentConstants.SUBMIT_VARIABLE_NMAE);
            // API级别附件提交后需要处理的字段列表
            List<String> targetSchemaListAPI = (List<String>) extendParas.get(AttachmentConstants.TARGET_SCHEMA_LIST_API);
            if (StringUtils.isBlank(submitVariableName) || CollectionUtils.isEmpty(targetSchemaListAPI)) {
                continue;
            }
            List<Map> deletedFiles = new ArrayList<>();

            // 2、处理API级附件
            dealApiAttachementFiles(submitActionDTO.getData().get(submitVariableName), targetSchemaListAPI, deletedFiles);
            if (CollectionUtils.isNotEmpty(deletedFiles)) {
                // 后面在AttachmentActionExecutor中调用dmc接口删除附件
                extendParas.put(AttachmentConstants.NEED_DELETE_FILES_API, deletedFiles);
            }
        }
    }

    /**
     * @param submitData
     * @param targetSchemaList
     * @param deletedFiles
     */
    private static void dealApiAttachementFiles(Object submitData, List<String> targetSchemaList, List<Map> deletedFiles) {
        // submitData必须是Map 或者 集合类型
        if (submitData instanceof Map) {
            Map<String, Object> data = (Map<String, Object>) submitData;
            for (String targetSchema : targetSchemaList) {
                LinkedList<String> targetSchemaSplitList = getTargetSchemaSplitList(targetSchema);
                dealApiAttachementFiles(data, targetSchema, targetSchemaSplitList, deletedFiles);
            }
        } else {
            List<Map<String, Object>> dataList = (List<Map<String, Object>>) submitData;
            if (CollectionUtils.isNotEmpty(dataList)) {
                for (Map<String, Object> data : dataList) {
                    for (String targetSchema : targetSchemaList) {
                        LinkedList<String> targetSchemaSplitList = getTargetSchemaSplitList(targetSchema);
                        dealApiAttachementFiles(data, targetSchema, targetSchemaSplitList, deletedFiles);
                    }
                }
            }
        }
    }

    /**
     * 将targetSchema按照分隔符解析成有序list
     *
     * @param targetSchema
     * @return
     */
    private static LinkedList getTargetSchemaSplitList(String targetSchema) {
        // targetSchema 可以配置单头（aaa）、单身(aaa.bbb)、子单身(aaa.bbb.ccc)
        String[] targetSchemaArray = targetSchema.split("\\.");
        return new LinkedList(Arrays.asList(targetSchemaArray));
    }

    /**
     * 处理API级附件
     *
     * @param submitData
     * @param targetSchema
     * @param targetSchemaSplitList
     * @param deletedFiles
     */
    private static void dealApiAttachementFiles(Map submitData, String targetSchema, LinkedList<String> targetSchemaSplitList, List<Map> deletedFiles) {
        if (MapUtils.isEmpty(submitData)) {
            return;
        }

        Iterator<String> targetSchemaIterator = targetSchemaSplitList.iterator();
        while (targetSchemaIterator.hasNext()) {
            String currentSchema = targetSchemaSplitList.poll();
            Object obj = submitData.get(currentSchema);
            if (null == obj) {
                return;
            }

            if (obj instanceof Map) {
                if (CollectionUtils.isEmpty(targetSchemaSplitList)) {
                    // 处理单头
                    doDealApiAttachementFiles(submitData, targetSchema, currentSchema, deletedFiles);
                } else {
                    // 还有单身或子单身，则继续递归
                    dealApiAttachementFiles(submitData, targetSchema, targetSchemaSplitList, deletedFiles);
                }
            } else {
                // 处理单身、子单身
                List<Map<String, Object>> dataList = (List<Map<String, Object>>) submitData.get(currentSchema);
                if (CollectionUtils.isNotEmpty(dataList)) {
                    String schema = targetSchemaSplitList.poll();
                    for (Map<String, Object> data : dataList) {
                        doDealApiAttachementFiles(data, targetSchema, schema, deletedFiles);
                    }
                }
            }
        }
    }

    /**
     * @param data
     * @param targetSchema
     * @param schema
     * @param deletedFiles
     */
    private static void doDealApiAttachementFiles(Map<String, Object> data, String targetSchema, String schema, List<Map> deletedFiles) {
        if (StringUtils.isEmpty(schema)) {
            return;
        }

        Map attachment = (Map) data.get(schema);
        if (null == attachment) {
            return;
        }

        List<Map> attachmentDataList = (List<Map>) attachment.get("data");
        if (CollectionUtils.isEmpty(attachmentDataList)) {
            return;
        }

        Iterator<Map> attachmentDataIterator = attachmentDataList.iterator();
        while (attachmentDataIterator.hasNext()) {
            Map attachmentData = attachmentDataIterator.next();
            String status = (String) attachmentData.get("uibot__status");
            attachmentData.remove("uibot__status");
            if (StringUtils.equalsIgnoreCase(status, "DELETE")) {
                // 过滤出“DELETE”状态的api级别附件数据，后面在AttachmentActionExecutor中调用dmc接口删除附件
                attachmentData.put("targetSchema", targetSchema);
                deletedFiles.add(attachmentData);
                // 过滤掉“DELETE”状态的api级别附件数据，不需要在调用esp接口传入
                attachmentDataIterator.remove();
            }
        }
    }

    /**
     * 解析出需要的定义
     *
     * @param submitAction
     * @return
     */
    public static SubmitAction parseAction(String targetActionId, SubmitAction submitAction) {
        if (submitAction == null) {
            return null;
        }
        if (targetActionId.equals(submitAction.getActionId())) {
            return submitAction;
        }

        List<SubmitAction> attachActions = submitAction.getAttachActions();
        if (CollectionUtils.isNotEmpty(attachActions)) {
            for (SubmitAction attachAction : attachActions) {
                SubmitAction targetAction = parseAction(targetActionId, attachAction);
                if (null != targetAction) {
                    return targetAction;
                }
            }
        }
        return null;
    }

}
