package com.digiwin.athena.atdm.action.executor;

import com.digiwin.athena.atdm.ActionConstants;
import com.digiwin.athena.atdm.UiBotConstants;
import com.digiwin.athena.atdm.action.ActionExecutor;
import com.digiwin.athena.atdm.recycle.CommonRecycleService;
import com.digiwin.athena.atdm.datasource.domain.ExecuteResult;
import com.digiwin.athena.atdm.datasource.domain.SubmitAction;
import com.digiwin.athena.atdm.datasource.domain.SubmitExecuteContext;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.UUID;

/**
 * @description: 从基础录入列表删除数据插入到回收站
 */
@Service
public class RecycleSaveExecutor implements ActionExecutor {
    @Autowired
    private CommonRecycleService commonRecycleService;

    @Override
    public String supportKey() {
        return UiBotConstants.ACTION_CATEGORY_RECYCLE + ActionConstants.SPLIT + "recycle.save";
    }

    @Override
    public ExecuteResult execute(SubmitExecuteContext executeContext, ExecuteResult parentExecuteResult, SubmitAction action, Map<String, Object> data) {
        String submitVariableName = Optional.ofNullable(action.getExtendParas())
                .map(extendParas -> (String) extendParas.get("submitVariableName"))
                .orElse("");
        List<Map<String, Object>> submitDataList = getSubmitDataList(submitVariableName, data);
        if (CollectionUtils.isEmpty(submitDataList)) {
            return ExecuteResult.ok();
        }

        /**
         * 提交的每条数据，没有DataKey就通过UUID生成一个RECYCLE__DATA_KEY，以保证回收站还原时能成功剔除该条数据。
         * 之所以使用RECYCLE__DATA_KEY，因为即使这边塞入一个__DATA_KEY，在回收站界面展示时，也会根据数据源定义生成一个__DATA_KEY，
         * 从而冲掉这边生成的__DATA_KEY，导致还原时__DATA_KEY不一致，无法剔除回收站中的数据。
         * @since p2s1
         * @authored majfa
         */
        for (Map<String, Object> submitData : submitDataList) {
            if (!submitData.containsKey(UiBotConstants.DATA_SOURCE_DATA_KEY)) {
                submitData.put(UiBotConstants.RECYCLE_DATA_KEY, UUID.randomUUID().toString());
            }
        }

        commonRecycleService.insertRecycle(executeContext.getAuthoredUser(), executeContext.getTmActivityId(), submitDataList);
        return ExecuteResult.ok();
    }

    private List<Map<String, Object>> getSubmitDataList(String submitVariableName, Map<String, Object> data) {
        if (MapUtils.isEmpty(data)) {
            return Collections.emptyList();
        }

        Optional<Object> dataOptional;
        if (StringUtils.isNotBlank(submitVariableName)) {
            dataOptional = Optional.ofNullable(data.get(submitVariableName));
        } else {
            dataOptional = data.values().stream().findFirst();
        }

        if (dataOptional.isPresent()) {
            Object submitData = dataOptional.get();
            if (submitData instanceof Map) {
                return Collections.singletonList((Map<String, Object>) submitData);
            } else {
                return (List<Map<String, Object>>) submitData;
            }
        } else {
            return Collections.emptyList();
        }
    }
}
