package com.digiwin.athena.atdm.action.executor;


import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.appcore.exception.BusinessException;
import com.digiwin.athena.atdm.action.util.ActionUtils;
import com.digiwin.athena.atdm.activity.domain.SubmitActionDTO;
import com.digiwin.athena.atdm.atmc.CommonAtmcService;
import com.digiwin.athena.atdm.constant.ErrorCodeEnum;
import com.digiwin.athena.atdm.datasource.ActionExecuteReq;
import com.digiwin.athena.atdm.datasource.domain.ExecuteResult;
import com.digiwin.athena.atdm.datasource.domain.SubmitAction;
import com.digiwin.athena.atdm.datasource.domain.SubmitExecuteContext;
import com.digiwin.athena.base.sdk.common.application.util.MessageUtil;
import com.google.common.eventbus.AsyncEventBus;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.stereotype.Service;
import org.springframework.util.StringUtils;

import javax.servlet.http.HttpServletRequest;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;


@Slf4j
@Service
public class DataSubmissionServiceWrapper {
    //最大提交笔数
    public static final Integer MAX_COMMIT_SIZE = 2000;

    public static final String WHITE_LIST = "submitActivityIdWhiteList";

    @Autowired
    private CommonAtmcService atmcService;

    @Autowired
    private DataSubmissionService dataSubmissionService;

    @Autowired
    private AsyncEventBus asyncEventBus;

    /**
     * 提交数据
     */
    public ExecuteResult executeActionSubmit(HttpServletRequest request, SubmitActionDTO submitAction) {
        actionCheck(request, submitAction);
        // 预处理API级别附件
        AttachmentActionExecutor.preprocessApiAttachment(com.google.common.collect.Lists.newArrayList(submitAction));
        SubmitExecuteContext executeContext = submitAction.getAction().getExecuteContext();
        SubmitAction action = submitAction.getAction();
        Map<String, Object> data = submitAction.getData();
        log.info("actionExecuteService.submit start");
        return this.dataSubmissionService.submit(new ActionExecuteReq(executeContext, null, action, data, submitAction.getWorkItemIdToData()));//actionService.submit(actionExecuteCommand);
    }

    private void actionCheck(HttpServletRequest request, SubmitActionDTO submitAction) {
        if (submitAction.getAction() == null) {
            throw new IllegalArgumentException("submitAction.action为null");
        }
        if (submitAction.getData() == null) {
            throw new IllegalArgumentException("submitAction.data为null");
        }

        SubmitExecuteContext executeContext = submitAction.getAction().getExecuteContext();
        if (executeContext == null) {
            executeContext = SubmitExecuteContext.createByHttpRequest(request);
            submitAction.getAction().setExecuteContext(executeContext);
        } else {
            executeContext.appendHttpRequest(request);
        }
        if (StringUtils.hasText(executeContext.getProxyToken())) {
            AppAuthContextHolder.getContext().setProxyToken(executeContext.getProxyToken());
        }

        executeContext.setOperateAuthoredUser(submitAction.getOperateAuthoredUser());

        if (CollectionUtils.isNotEmpty(submitAction.getAction().getAttachActions())) {
            List<SubmitAction> attachActions = submitAction.getAction().getAttachActions();
            for (SubmitAction attachAction : attachActions) {
                SubmitExecuteContext attachActionExecuteContext = attachAction.getExecuteContext();
                if (attachActionExecuteContext == null) {
                    attachAction.setExecuteContext(executeContext);
                } else {
                    attachActionExecuteContext.appendHttpRequest(request);
                }
            }
        }

        SubmitAction action = submitAction.getAction();
        if (action == null) {
            throw new IllegalArgumentException("submitAction.getAction()");
        }

        // 新增不超过2，000笔数据提交逻辑校验
        List<SubmitActionDTO> list = new ArrayList<>();
        list.add(submitAction);
        if (!validateSizeOrWhiteList(list)) {
            throw BusinessException.create(ErrorCodeEnum.BUSINESS_600_0003.getErrCode(), MessageUtil.getMessageByLocale("submit.task.size.overflow", LocaleContextHolder.getLocale().toString()));
        }

    }

    private boolean validateSizeOrWhiteList(List<SubmitActionDTO> submitActions) {
        // 全在在白名单中，就不考虑数据量了
        if (atmcService.operationConfig(submitActions.get(0).getAction().getExecuteContext().getTmActivityId(), 0, WHITE_LIST)) {
            return true;
        }

        // 不在白名单中，考虑是否有数据量大于2,000的
        return submitActions.stream().filter(e -> e.getData() != null).noneMatch(e -> e.getData().values().stream().filter(List.class::isInstance).map(el -> (List<?>) el).anyMatch(el -> el.size() > MAX_COMMIT_SIZE));
    }

    /**
     * 仅执行action
     * <br>不会判断是否处理完成
     * <br>不会关闭待办等
     */
    public ExecuteResult executeAction(HttpServletRequest request, SubmitActionDTO submitAction) {
        if (submitAction.getAction() == null) {
            throw new IllegalArgumentException("submitAction.action为null");
        }
        if (submitAction.getData() == null) {
            throw new IllegalArgumentException("submitAction.data为null");
        }

        // 判断若是行事历任务卡的提交，将请求转发至atmc新接口
        if (ActionUtils.isCalendarSubmitAction(submitAction.getAction())) {
            return atmcService.executeCalendarAction(submitAction);
        }

        // 包含非通用（ESP、附件）等类型Action，将请求转发至atmc接口
        if (ActionUtils.notCommonSubmitAction(submitAction.getAction())) {
            return atmcService.executeAction(submitAction);
        }

        SubmitExecuteContext executeContext = submitAction.getAction().getExecuteContext();
        if (executeContext == null) {
            executeContext = SubmitExecuteContext.createByHttpRequest(request);
            executeContext.setBusinessUnit(submitAction.getAction().getBusinessUnit());
            submitAction.getAction().setExecuteContext(executeContext);
        } else {
            executeContext.appendHttpRequest(request);
        }
        if (StringUtils.hasText(executeContext.getProxyToken())) {
            AppAuthContextHolder.getContext().setProxyToken(executeContext.getProxyToken());
        }
        if (CollectionUtils.isNotEmpty(submitAction.getAction().getAttachActions())) {
            List<SubmitAction> attachActions = submitAction.getAction().getAttachActions();
            for (SubmitAction attachAction : attachActions) {
                SubmitExecuteContext attachActionExecuteContext = attachAction.getExecuteContext();
                if (attachActionExecuteContext == null) {
                    attachAction.setExecuteContext(executeContext);
                } else {
                    attachActionExecuteContext.appendHttpRequest(request);
                }
            }
        }
        preExecute(executeContext, submitAction);
        return doExecute(executeContext, submitAction);
    }

    private ExecuteResult doExecute(SubmitExecuteContext executeContext, SubmitActionDTO submitAction) {
        return dataSubmissionService.execute(executeContext, submitAction.getAction(), submitAction.getData());
    }

    private void preExecute(SubmitExecuteContext executeContext, SubmitActionDTO submitAction) {
        // do nothing
    }
}
