package com.digiwin.athena.itsys.dao;

import java.io.Serializable;
import java.util.List;


/**
 * 通用性的DAO，內含POJO共用的CRUD方法
 *
 * 使用方法：在需要保護物件(Model)的型態之狀況下，繼承此介面
 *
 * @author Roger
 * @param <T> a type variable
 * @param <PK> the primary key for that type
 */
public interface GenericDao<T, PK extends Serializable> {

    /**
     * 取得Table內所有的物件資料
     * 請小心使用，以免Out of Memory
     * @return List 資料
     */
    List<T> getAll ();

    public List<T> getDistinct (List<T> list);
    /**
     * 透過id取得單一特定的Object
     *
     * @param id PK型態
     * @return 特定單一model
     * @see org.springframework.orm.ObjectRetrievalFailureException
     */
    T get (PK id);

    /**
     * 檢查某筆資料是否存在
     * @param id PK型態
     * @return - true if it exists, false if it doesn't
     */
    boolean exists (PK id);

    /**
     * 存入物件資料為Table資料 - 會同時處理更新和插入
     *
     * @param object 存入物件資料
     * @return 存入物件
     */
    T save (T object);

    /**
     * 透過id刪除特定的資料
     * @param id PK型態
     */
    void remove (PK id);

    /**
     * 以物件方式查詢資訊,比對條件式必須完全相同
     * @param pObject 要查詢的物件
     * @return a list of the records found
     */
    public List<T> findByExample (T pObject);

    /**
     * 模糊方式以物件方式查詢資訊
     * @param pObject 要查詢的物件
     * @param pClazz 物件類別
     * @return a list of the records found
     */
    public List<T> findByExampleLike (T pObject, Class<T> pClazz);

    /**
     * 處理更新和插入
     *
     * @param object 存入物件資料
     *
     */
    void saveOrupdate (T object);
}