package com.digiwin.athena.itsys.dao.impl;

import com.digiwin.athena.itsys.dao.MonitorLogDao;
import com.digiwin.athena.itsys.enums.EocLevelEnum;
import com.digiwin.athena.itsys.model.MonitorLog;
import org.hibernate.Criteria;
import org.hibernate.HibernateException;
import org.hibernate.Session;
import org.hibernate.criterion.Restrictions;
import org.hibernate.query.Query;
import org.springframework.orm.hibernate5.HibernateCallback;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.Collection;
import java.util.List;
import java.util.Map;

@Service
@Transactional("itsysTransactionManager")
public class MonitorLogDaoImpl extends GenericDaoImpl<MonitorLog, Long> implements MonitorLogDao {

	public MonitorLogDaoImpl() {
		super(MonitorLog.class);
	}

	@Override
	public List<MonitorLog> getMonitorLogsByTraceIds(List<String> pTraceIds, String pTenantId) {
		return getHibernateTemplate().execute(new HibernateCallback<List<MonitorLog>>() {
			@Override
			public List<MonitorLog> doInHibernate(Session pSession) throws HibernateException {

				Criteria criteria = pSession.createCriteria(MonitorLog.class);
				criteria.add(Restrictions.eq("tenantId", pTenantId));
				criteria.add(Restrictions.in("traceId", pTraceIds));
				criteria.add(Restrictions.eq("initial", false));
				return criteria.list();

			}
		});

	}

	@Override
	public List<MonitorLog> getMonitorLogsByTraceIdsAndRuleCode(Collection<String> pTraceIds, String pTenantId, String ruleCode) {
		return getHibernateTemplate().execute(new HibernateCallback<List<MonitorLog>>() {
			@Override
			public List<MonitorLog> doInHibernate(Session pSession) throws HibernateException {

				Criteria criteria = pSession.createCriteria(MonitorLog.class);
				criteria.add(Restrictions.eq("tenantId", pTenantId));
				criteria.add(Restrictions.in("traceId", pTraceIds));
				criteria.add(Restrictions.eq("initial", false));
				criteria.add(Restrictions.eq("ruleCode", ruleCode));
				criteria.add(Restrictions.eq("state", true));
				return criteria.list();
			}
		});
	}

	@Override
	public List<MonitorLog> getMonitorLogsBySourceIdAndTenantId(String pSourceId, String pTenantId) {
		return getHibernateTemplate().execute(new HibernateCallback<List<MonitorLog>>() {
			@Override
			public List<MonitorLog> doInHibernate(Session pSession) throws HibernateException {

				StringBuilder tHqlSB = new StringBuilder();
				tHqlSB.append(
						"from MonitorLog where tenantId=:tenant_id and sourceId=:source_id and initial=:initial");

				Query<MonitorLog> tQuery = pSession.createQuery(tHqlSB.toString(), MonitorLog.class);

				tQuery.setParameter("tenant_id", pTenantId).setParameter("source_id", "%" + pSourceId + "%");

				tQuery.setParameter("initial", false);

				return tQuery.getResultList();
			}
		});

	}

	@Override
	public List<MonitorLog> getRecentInitialLog(String pTenantId, String pRuleCode,
                                                Map<String, String> pEocConditionMap) {
		return getHibernateTemplate().execute(new HibernateCallback<List<MonitorLog>>() {
			@Override
			public List<MonitorLog> doInHibernate(Session pSession) throws HibernateException {

				StringBuilder tHqlSB = new StringBuilder();
				tHqlSB.append(
						"from MonitorLog where tenantId=:tenant_id and ruleCode=:rule_code and initial=:initial ");

				for (String eocLevel : pEocConditionMap.keySet()) {
					if (eocLevel.equals(EocLevelEnum.COMPANY.getLevel())) {
						tHqlSB.append("and eocCompanyId=:eoc_company_id ");
					}
					if (eocLevel.equals(EocLevelEnum.SITE.getLevel())) {
						tHqlSB.append("and eocSiteId=:eoc_site_id ");
					}
					if (eocLevel.equals(EocLevelEnum.REGION.getLevel())) {
						tHqlSB.append("and eocRegionId=:eoc_region_id ");
					}
				}

				tHqlSB.append("order by id desc");

				Query<MonitorLog> tQuery = pSession.createQuery(tHqlSB.toString(), MonitorLog.class);

				tQuery.setMaxResults(1);
				tQuery.setParameter("tenant_id", pTenantId);
				tQuery.setParameter("rule_code", pRuleCode);
				tQuery.setParameter("initial", true);

				for (String eocLevel : pEocConditionMap.keySet()) {
					if (eocLevel.equals(EocLevelEnum.COMPANY.getLevel())) {
						tQuery.setParameter("eoc_company_id", pEocConditionMap.get(EocLevelEnum.COMPANY.getLevel()));
					}
					if (eocLevel.equals(EocLevelEnum.SITE.getLevel())) {
						tQuery.setParameter("eoc_site_id", pEocConditionMap.get(EocLevelEnum.SITE.getLevel()));
					}
					if (eocLevel.equals(EocLevelEnum.REGION.getLevel())) {
						tQuery.setParameter("eoc_region_id", pEocConditionMap.get(EocLevelEnum.REGION.getLevel()));
					}
				}

				return tQuery.getResultList();
			}
		});

	}

	@Override
	public List<MonitorLog> getRecentFinishedLog(String pTenantId, String pRuleCode,
                                                 Map<String, String> pEocConditionMap) {
		return getHibernateTemplate().execute(new HibernateCallback<List<MonitorLog>>() {
			@Override
			public List<MonitorLog> doInHibernate(Session pSession) throws HibernateException {

				StringBuilder tHqlSB = new StringBuilder();
				tHqlSB.append(
						"from MonitorLog where tenantId=:tenant_id and ruleCode=:rule_code and initial=:initial ");

				for (String eocLevel : pEocConditionMap.keySet()) {
					if (eocLevel.equals(EocLevelEnum.COMPANY.getLevel())) {
						tHqlSB.append("and eocCompanyId=:eoc_company_id ");
					}
					if (eocLevel.equals(EocLevelEnum.SITE.getLevel())) {
						tHqlSB.append("and eocSiteId=:eoc_site_id ");
					}
					if (eocLevel.equals(EocLevelEnum.REGION.getLevel())) {
						tHqlSB.append("and eocRegionId=:eoc_region_id ");
					}
				}

				tHqlSB.append("order by id desc");

				Query<MonitorLog> tQuery = pSession.createQuery(tHqlSB.toString(), MonitorLog.class);

				tQuery.setMaxResults(1);
				tQuery.setParameter("tenant_id", pTenantId);
				tQuery.setParameter("rule_code", pRuleCode);
				tQuery.setParameter("initial", false);

				for (String eocLevel : pEocConditionMap.keySet()) {
					if (eocLevel.equals(EocLevelEnum.COMPANY.getLevel())) {
						tQuery.setParameter("eoc_company_id", pEocConditionMap.get(EocLevelEnum.COMPANY.getLevel()));
					}
					if (eocLevel.equals(EocLevelEnum.SITE.getLevel())) {
						tQuery.setParameter("eoc_site_id", pEocConditionMap.get(EocLevelEnum.SITE.getLevel()));
					}
					if (eocLevel.equals(EocLevelEnum.REGION.getLevel())) {
						tQuery.setParameter("eoc_region_id", pEocConditionMap.get(EocLevelEnum.REGION.getLevel()));
					}
				}

				return tQuery.getResultList();
			}
		});

	}

	@Override
	public List<MonitorLog> getInitialLogByTraceId(String pTraceId) {
		return getHibernateTemplate().execute(new HibernateCallback<List<MonitorLog>>() {
			@Override
			public List<MonitorLog> doInHibernate(Session pSession) throws HibernateException {

				StringBuilder tHqlSB = new StringBuilder();
				tHqlSB.append("from MonitorLog where traceId=:trace_id and initial=:initial");

				Query<MonitorLog> tQuery = pSession.createQuery(tHqlSB.toString(), MonitorLog.class);

				tQuery.setMaxResults(1);
				tQuery.setParameter("trace_id", pTraceId);
				tQuery.setParameter("initial", true);

				return tQuery.getResultList();
			}
		});
	}

	@Override
	public void deleteMonitorLogsByTenantId(String pTenantId) {

		getHibernateTemplate().execute(new HibernateCallback() {
			@Override
			public Object doInHibernate(Session pSession) throws HibernateException {
				Query tQuery = pSession.createQuery("delete MonitorLog where tenantId =:tenantId");
				tQuery.setParameter("tenantId", pTenantId);

				return tQuery.executeUpdate();
			}
		});

	}

}
