package com.digiwin.athena.itsys.dao.impl;

import com.digiwin.athena.itsys.dao.ProjectLogDao;
import com.digiwin.athena.itsys.enums.AppTypeEnum;
import com.digiwin.athena.itsys.enums.ProjectLogConditionEnum;
import com.digiwin.athena.itsys.model.ProjectLog;
import org.apache.commons.lang3.tuple.Pair;
import org.hibernate.Criteria;
import org.hibernate.HibernateException;
import org.hibernate.Session;
import org.hibernate.criterion.Restrictions;
import org.hibernate.query.Query;
import org.hibernate.transform.Transformers;
import org.springframework.orm.hibernate5.HibernateCallback;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigInteger;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.*;

@Service
@Transactional("itsysTransactionManager")
public class ProjectLogDaoImpl extends GenericDaoImpl<ProjectLog, Long> implements ProjectLogDao {

	public ProjectLogDaoImpl() {
		super(ProjectLog.class);
	}

	SimpleDateFormat formatToDate = new SimpleDateFormat("yyyy-MM-dd");

	// 查詢資料
	@SuppressWarnings("unchecked")
	@Override
	public List<ProjectLog> fetch(Map<String, String> pConditions) {
		List<ProjectLog> tReturn = null;
		tReturn = (List<ProjectLog>) doHibernateQuery(pConditions);
		return tReturn;
	}

	public Object doHibernateQuery(final Map<String, String> pConditions) {

		return getHibernateTemplate().execute(new HibernateCallback<List<ProjectLog>>() {
			@Override
			public List<ProjectLog> doInHibernate(Session pSession) throws HibernateException {

				Criteria criteria = pSession.createCriteria(ProjectLog.class);

				// 有設定時間範圍
				if (pConditions.containsKey("query_start_date")) {
					try {

						Calendar calendar = Calendar.getInstance();
						calendar.setTime(formatToDate.parse(pConditions.get("query_end_date")));
						calendar.add(Calendar.DATE, 1);

						criteria.add(Restrictions.between("createdTime",
                                                          formatToDate.parse(pConditions.get("query_start_date")), calendar.getTime()));

					} catch (ParseException e) {
						log.error(e.getMessage(), e);
					}
				}

				if (pConditions.containsKey(ProjectLogConditionEnum.STATE.toString())) {
					criteria.add(Restrictions.eq("state",
                                                 Integer.valueOf(pConditions.get(ProjectLogConditionEnum.STATE.toString()))));
				}

				if (pConditions.containsKey(ProjectLogConditionEnum.APP_ID.toString())) {
					criteria.add(Restrictions.eq("appId", pConditions.get(ProjectLogConditionEnum.APP_ID.toString())));
				}

				if (pConditions.containsKey(ProjectLogConditionEnum.TENANT_ID.toString())) {
					criteria.add(
                            Restrictions.eq("tenantId", pConditions.get(ProjectLogConditionEnum.TENANT_ID.toString())));
				}

				if (pConditions.containsKey(ProjectLogConditionEnum.PROCESS_SERIAL_NUMBER.toString())) {
					criteria.add(Restrictions.eq("processSerialNumber",
                                                 pConditions.get(ProjectLogConditionEnum.PROCESS_SERIAL_NUMBER.toString())));
				}

				return criteria.list();
			}
		});

	}

	// 不限制項目的狀態，不限制提供消息的組件(taskengine or ptm)，取最新的一筆紀錄
	@Override
	public List<ProjectLog> getProjectLogsBySerialNumbers(List<String> pSerialNumbers, String pTenantId) {
		return getHibernateTemplate().execute(new HibernateCallback<List<ProjectLog>>() {
			@Override
			public List<ProjectLog> doInHibernate(Session pSession) throws HibernateException {

				Criteria criteria = pSession.createCriteria(ProjectLog.class);
				criteria.add(Restrictions.eq("tenantId", pTenantId));
				criteria.add(Restrictions.in("processSerialNumber", pSerialNumbers));
				// criteria.add(Restrictions.in("appId", "ptm"));

				// criteria.add(
				// Restrictions.disjunction().add(Restrictions.eq("state",
				// 1)).add(Restrictions.eq("state", 2)));
				// criteria.setMaxResults(1);
				return criteria.list();

			}
		});

	}

	@Override
	public List<ProjectLog> getErrorProject(String tenantId, Pair<Date, Date> dateRange, String projectCode) {
		return getHibernateTemplate().execute(new HibernateCallback<List<ProjectLog>>() {
			@Override
			public List<ProjectLog> doInHibernate(Session pSession) throws HibernateException {
				StringBuilder tHqlSB = new StringBuilder();
				tHqlSB.append("select distinct p.traceId as traceId,p.processSerialNumber as processSerialNumber from ProjectLog as p where " +
						" p.tenantId = :tenantId and p.appId = :appId and p.projectId = :projectId and p.state != :state " +
						" and exists(select 1 from ErrorMessage as e where e.traceId = p.traceId and e.processSerialNumber = p.processSerialNumber " +
						" and e.createdTime between :startDate and :endDate and e.isError = :isError)");
				Query tQuery = pSession.createQuery(tHqlSB.toString());
				tQuery.setResultTransformer(Transformers.aliasToBean(ProjectLog.class));
				tQuery.setParameter("tenantId", tenantId);
				tQuery.setParameter("appId", AppTypeEnum.TASKENGINE.getType());
				tQuery.setParameter("projectId", projectCode);
				tQuery.setParameter("state", 3);
				tQuery.setParameter("startDate", dateRange.getLeft());
				tQuery.setParameter("endDate", dateRange.getRight());
				tQuery.setParameter("isError", true);
				return tQuery.getResultList();
			}
		});
	}

	/**
	 * 根据租户查询所有项目
	 * @param pTenantId
	 * @return
	 */
	@Override
	public Long getSuccessProjectCnt(String pTenantId) {
		return getHibernateTemplate().execute(new HibernateCallback<Long>() {
			@Override
			public Long doInHibernate(Session pSession) throws HibernateException {
				StringBuffer appendSql = new StringBuffer("");
				appendSql.append("select count(distinct process_serial_number) from project_log pl where tenant_id= :tenantId and state in (1,2) ");
				appendSql.append("and not exists (select id from error_log el where pl.process_serial_number = el.process_serial_number)");
				Query tQuery = pSession.createSQLQuery(appendSql.toString());
				tQuery.setParameter("tenantId", pTenantId);
				return ((BigInteger)tQuery.uniqueResult()).longValue();
			}
		});
	}

	@Override
	public List<ProjectLog> getProjectLogsByTraceId(String pTraceId) {
		return getHibernateTemplate().execute(new HibernateCallback<List<ProjectLog>>() {
			@Override
			public List<ProjectLog> doInHibernate(Session pSession) throws HibernateException {

				List<Integer> states = new ArrayList<Integer>();
				StringBuilder tHqlSB = new StringBuilder();
				tHqlSB.append("from ProjectLog where traceId=:trace_id and state in (:state)");
				Query<ProjectLog> tQuery = pSession.createQuery(tHqlSB.toString(), ProjectLog.class);
				tQuery.setParameter("trace_id", pTraceId);
				states.add(1);
				states.add(2);
				tQuery.setParameterList("state", states);

				return tQuery.getResultList();
			}
		});

	}
}
