package com.digiwin.athena.itsys.dao.impl;

import com.digiwin.athena.itsys.dao.ErrorMessageDao;
import com.digiwin.athena.itsys.enums.AppTypeEnum;
import com.digiwin.athena.itsys.model.ErrorMessage;
import com.digiwin.athena.itsys.util.StringUtil;
import com.fasterxml.jackson.databind.JsonNode;
import org.hibernate.Criteria;
import org.hibernate.HibernateException;
import org.hibernate.Session;
import org.hibernate.criterion.Restrictions;
import org.hibernate.query.Query;
import org.hibernate.transform.Transformers;
import org.springframework.orm.hibernate5.HibernateCallback;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigInteger;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

@Service
@Transactional("itsysTransactionManager")
public class ErrorMessageDaoImpl extends GenericDaoImpl<ErrorMessage, Long> implements ErrorMessageDao {

	public ErrorMessageDaoImpl() {
		super(ErrorMessage.class);
	}

	SimpleDateFormat formatToDate = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss.SSS");

	// 查詢資料
	@SuppressWarnings("unchecked")
	@Override
	public List<ErrorMessage> fetch(Map<String, String> pConditions) {
		List<ErrorMessage> tReturn = null;
		tReturn = (List<ErrorMessage>) doHibernateQuery(pConditions);
		return tReturn;
	}

	public Object doHibernateQuery(final Map<String, String> pConditions) {
		Object tReturn = null;

		tReturn = getHibernateTemplate().execute(new HibernateCallback() {

			@Override
			public Object doInHibernate(Session pSession) throws HibernateException {

				StringBuilder tHqlSB = new StringBuilder();

				List<String> conditionList = new ArrayList<String>();

				tHqlSB.append(
						"select id as id," +
								"traceId as traceId," +
								"appId as appId," +
								"processSerialNumber as processSerialNumber," +
								"processActivityId as processActivityId," +
								"retryUrl as retryUrl," +
								"token as token," +
								"processActivityOid as processActivityOid," +
								"tenantId as tenantId," +
								"exceptionType as exceptionType," +
								"errorCode as errorCode," +
								"isError as isError," +
								"buildTime as buildTime," +
								"lastUpdateTime as lastUpdateTime," +
								"createdTime as createdTime," +
								"idempotency as idempotency," +
								"taskUid as taskUid," +
								"actionId as actionId," +
								"apiErrorInfo as apiErrorInfo," +
								"apiErrorInfoMetadata as apiErrorInfoMetadata from ErrorMessage where createdTime >= :query_start_date AND createdTime <= :query_end_date and tenantId=:tenant_id and isError = :is_error and appId in (:app_id)");

//				if (pConditions.containsKey("query_start_date")) {
//					tHqlSB.append(" and DATE(createdTime)>=:query_start_date and DATE(createdTime)<=:query_end_date");
//				}

				Query tQuery = pSession.createQuery(tHqlSB.toString());
				tQuery.setResultTransformer(Transformers.aliasToBean(ErrorMessage.class));

				// 有指定租戶
				if (pConditions.containsKey("tenant_id")) {
					tQuery.setParameter("tenant_id", pConditions.get("tenant_id"));
				}

				// 有設定時間範圍
				if (pConditions.containsKey("query_start_date")) {
					try {
						tQuery.setParameter("query_start_date",
								formatToDate.parse(pConditions.get("query_start_date")+ " 00:00:00.000"));
						tQuery.setParameter("query_end_date", 
								formatToDate.parse(pConditions.get("query_end_date")+ " 23:59:59.999"));
					} catch (ParseException e) {
						log.error(e.getMessage(), e);
					}
				}

				// 有指定搜尋類型
				// stuck = 異常
				if (pConditions.containsKey("condition")) {
					switch (pConditions.get("condition")) {
					case "stuck":
						// p3s2優化之後，不在看流程引擎數據
						// conditionList.add(AppTypeEnum.BPMCLOUD.getType());
						conditionList.add(AppTypeEnum.TASKENGINE.getType());
						conditionList.add(AppTypeEnum.ATMC.getType());
						tQuery.setParameterList("app_id", conditionList);
						break;
					case "solving":
						conditionList.add(AppTypeEnum.ATMC.getType());
						tQuery.setParameterList("app_id", conditionList);
						break;
					case "total":
						// p3s2優化之後，不在看流程引擎數據
						// conditionList.add(AppTypeEnum.BPMCLOUD.getType());
						conditionList.add(AppTypeEnum.TASKENGINE.getType());
						conditionList.add(AppTypeEnum.ATMC.getType());
						tQuery.setParameterList("app_id", conditionList);
						break;

					default:
						break;
					}
				}

				// 有指定異常解決狀態
				// 1 = 為解決
				// 0 = 已解決
				if (pConditions.containsKey("is_error")) {
					tQuery.setParameter("is_error", Boolean.valueOf(pConditions.get("is_error").toString()));
				}



				return tQuery.getResultList();

			}
		});

		return tReturn;
	}

	@Override
	public List<ErrorMessage> getByBusinessLogicKey(JsonNode tErrorMessage) {
		return getHibernateTemplate().execute(new HibernateCallback<List<ErrorMessage>>() {
			@Override
			public List<ErrorMessage> doInHibernate(Session pSession) throws HibernateException {

				StringBuilder tHqlSB = new StringBuilder();
				tHqlSB.append(
						"from ErrorMessage where (traceId=:traceId and processSerialNumber=:processSerialNumber and processActivityOid=:processActivityOID)");

				Query<ErrorMessage> tQuery = pSession.createQuery(tHqlSB.toString(), ErrorMessage.class);

				tQuery.setParameter("traceId", tErrorMessage.get("traceId").asText())
						.setParameter("processSerialNumber", tErrorMessage.get("processSerialNumber").asText())
						.setParameter("processActivityOID", tErrorMessage.get("processActivityOID").asText());

				return tQuery.getResultList();
			}
		});

	}

	@Override
	public List<ErrorMessage> getErrorsBySerialNumber(String pProcessSerialNumber) {
		return getHibernateTemplate().execute(new HibernateCallback<List<ErrorMessage>>() {
			@Override
			public List<ErrorMessage> doInHibernate(Session pSession) throws HibernateException {

				StringBuilder tHqlSB = new StringBuilder();

				tHqlSB.append(
						"from ErrorMessage where processSerialNumber=:processSerialNumber and isError = :is_error");

				Query<ErrorMessage> tQuery = pSession.createQuery(tHqlSB.toString(), ErrorMessage.class);

				tQuery.setParameter("processSerialNumber", pProcessSerialNumber).setParameter("is_error", true);

				return tQuery.getResultList();
			}
		});

	}

	@Override
	public List<ErrorMessage> getErrorsBySerialNumbers(List<String> pProcessSerialNumbers, String pTenantId) {
		return getHibernateTemplate().execute(new HibernateCallback<List<ErrorMessage>>() {
			@Override
			public List<ErrorMessage> doInHibernate(Session pSession) throws HibernateException {

				Criteria criteria = pSession.createCriteria(ErrorMessage.class);
				criteria.add(Restrictions.eq("tenantId", pTenantId));
				criteria.add(Restrictions.in("isError", true));
				criteria.add(Restrictions.in("processSerialNumber", pProcessSerialNumbers));
				return criteria.list();

			}
		});

	}

	@Override
	public void deleteErrorsByTenantId(String pTenantId) {

		getHibernateTemplate().execute(new HibernateCallback() {
			@Override
			public Object doInHibernate(Session pSession) throws HibernateException {
				Query tQuery = pSession.createQuery("delete ErrorMessage where tenantId =:tenantId");
				tQuery.setParameter("tenantId", pTenantId);

				return tQuery.executeUpdate();
			}
		});

	}

	@Override
	public List<ErrorMessage> getErrorsByTenantId(String pTenantId, Integer pageSize, Integer pageNum) {
		return getHibernateTemplate().execute(new HibernateCallback<List<ErrorMessage>>() {
			@Override
			public List<ErrorMessage> doInHibernate(Session pSession) throws HibernateException {

				StringBuilder tHqlSB = new StringBuilder();

				tHqlSB.append(
						"select id as id," +
								"traceId as traceId," +
								"appId as appId," +
								"processSerialNumber as processSerialNumber," +
								"processActivityId as processActivityId," +
								"retryUrl as retryUrl," +
								"token as token," +
								"processActivityOid as processActivityOid," +
								"tenantId as tenantId," +
								"exceptionType as exceptionType," +
								"errorCode as errorCode," +
								"isError as isError," +
								"buildTime as buildTime," +
								"lastUpdateTime as lastUpdateTime," +
								"createdTime as createdTime," +
								"idempotency as idempotency," +
								"taskUid as taskUid," +
								"actionId as actionId," +
								"apiErrorInfo as apiErrorInfo," +
								"apiErrorInfoMetadata as apiErrorInfoMetadata from ErrorMessage " +
								"where tenantId=:tenant_id and isError = :is_error and (appId = :app_id2 or appId = :app_id3) order by id desc");
				Query tQuery = pSession.createQuery(tHqlSB.toString());
				tQuery.setResultTransformer(Transformers.aliasToBean(ErrorMessage.class));

				tQuery.setParameter("tenant_id", pTenantId);
				// p3s2優化之後，不在看流程引擎數據
//				tQuery.setParameter("app_id1", AppTypeEnum.BPMCLOUD.getType());
				tQuery.setParameter("app_id2", AppTypeEnum.TASKENGINE.getType());
				tQuery.setParameter("app_id3", AppTypeEnum.ATMC.getType());
				tQuery.setParameter("is_error", true);

				//在Page类中我已说明了：每页开始的索引数在hibernate中的计算公式是：(currentPage-1)*perPageRows
				if (null != pageNum && null != pageSize) {
					tQuery.setFirstResult((pageNum-1)*pageSize);
					tQuery.setMaxResults(pageSize);
				}
				return tQuery.getResultList();
			}
		});

	}

	@Override
	public List<ErrorMessage> getErrorBySerialNumberAndActivityOid(String pSerailNumber, String pActivityOid) {
		return getHibernateTemplate().execute(new HibernateCallback<List<ErrorMessage>>() {
			@Override
			public List<ErrorMessage> doInHibernate(Session pSession) throws HibernateException {

				StringBuilder tHqlSB = new StringBuilder();
				tHqlSB.append(
						"from ErrorMessage where isError = :is_error and (processSerialNumber=:processSerialNumber and processActivityOid=:processActivityOID)");

				Query<ErrorMessage> tQuery = pSession.createQuery(tHqlSB.toString(), ErrorMessage.class);

				tQuery.setParameter("processSerialNumber", pSerailNumber)
						.setParameter("processActivityOID", pActivityOid).setParameter("is_error", true);

				return tQuery.getResultList();
			}
		});

	}

	/**
	 * 根据异常ID查询异常明细
	 * @param errorLogId
	 * @return
	 */
	@Override
	public ErrorMessage getExceptionDetail(Long errorLogId, String processSerialNumber) {
		return getHibernateTemplate().execute(new HibernateCallback<ErrorMessage>() {
			@Override
			public ErrorMessage doInHibernate(Session pSession) throws HibernateException {
				StringBuffer appendSql = new StringBuffer();
				appendSql.append("from ErrorMessage as em where isError = 1 ");
				if (null != errorLogId) {
					appendSql.append("and em.id =:id ");
				}
				if (!StringUtil.isEmpty(processSerialNumber)) {
					appendSql.append("and em.processSerialNumber =:processSerialNumber ");
				}
				Query<ErrorMessage> tQuery = pSession.createQuery(appendSql.toString());
				if (null != errorLogId) {
					tQuery.setParameter("id", errorLogId);
				}
				if (!StringUtil.isEmpty(processSerialNumber)) {
					tQuery.setParameter("processSerialNumber", processSerialNumber);
				};

				// add by songwq 针对存在重复数据进行优化
				List<ErrorMessage> errorMessageList = tQuery.list();
				if (null != errorMessageList && errorMessageList.size() > 0) {
					return errorMessageList.get(0);
				}
				return null;
			}
		});
	}

	/**
	 * 获取错误日志次数
	 * @param pTenantId
	 * @return
	 */
	@Override
	public Long getErrorLogCnt(String pTenantId) {
		 return getHibernateTemplate().execute(new HibernateCallback<Long>() {
			@Override
			public Long doInHibernate(Session pSession) throws HibernateException {
				StringBuffer appendSql = new StringBuffer();
				appendSql.append("select ");
				appendSql.append("count(distinct el.id) cnt ");
				appendSql.append("from ");
				appendSql.append("error_log el ");
				appendSql.append("left join project_log pl on ");
				appendSql.append("el.process_serial_number = pl.process_serial_number ");
				appendSql.append("and el.tenant_id = pl.tenant_id ");
				appendSql.append("where ");
				appendSql.append("el.tenant_id = :tenantId ");
				appendSql.append("and is_error = true ");
				appendSql.append("and (el.app_id = :appId1 ");
				appendSql.append("or el.app_id = :appId2)");
				Query tQuery = pSession.createSQLQuery(appendSql.toString());
				tQuery.setParameter("tenantId", pTenantId);
				tQuery.setParameter("appId1", AppTypeEnum.TASKENGINE.getType());
				tQuery.setParameter("appId2", AppTypeEnum.ATMC.getType());
				return ((BigInteger)tQuery.uniqueResult()).longValue();
			}
		});
	}
}
