package com.digiwin.athena.itsys.util;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.io.Writer;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * 設定檔工具,可以保存配置注釋，並且保持原設定檔順序
 * 
 * 
 */
public class PropertiesUtil {
	private final static Logger log = LoggerFactory.getLogger(PropertiesUtil.class);

	private static String POUND = "#";
	private static String EQUAL = "=";
	/**
	 * 行集合（key－value行則只包含key部分）
	 */
	private List<String> keyList = new ArrayList<String>();

	/** 鍵值對集合（設定檔有效鍵值對） */
	private Map<String, String> valueMap = new HashMap<String, String>();

	public Map<String, String> getPropertyMap() {
		return valueMap;
	}

	/**
	 * 得到設定檔
	 * 
	 * @param key 配置鍵
	 * @return 配置值
	 */
	public String getProperty(String key) {
		return valueMap.get(key);
	}

	/**
	 * 設置配置項
	 * 
	 * @param key 配置鍵
	 * @return 配置值
	 */
	public void setProperty(String key, String value) {
		// 處理新增配置項，防止在保存時丟失
		if (!valueMap.containsKey(key)) {
			keyList.add(key);
		}
		valueMap.put(key, value);
	}

	/**
	 * 刪除配置項
	 * 
	 * @param key 配置鍵
	 * @return 配置值
	 */
	public void delProperty(String key, String value) {
		// 處理新增配置項，防止在保存時丟失
		if (valueMap.containsKey(key)) {
			Iterator<String> ikl = keyList.iterator();
			while (ikl.hasNext()) {
				if (ikl.next().equals(key)) {
					ikl.remove();
				}
			}
			valueMap.remove(key);
		}
	}

	/**
	 * 載入.properties文件
	 * 
	 * @param stream 輸入流
	 * @throws Exception IO錯誤
	 */
	public synchronized void load(InputStream stream) throws IOException {
		Reader isr = new InputStreamReader(stream);
		BufferedReader reader = new BufferedReader(isr);
		keyList.clear();
		valueMap.clear();
		try {
			while (reader.ready()) {
				saveLine(reader.readLine());
			}
		} catch (IOException e) {
			log.error(e.getMessage(), e);
		} finally {
			try {
				if (reader != null) {
					reader.close();
				}
				if (isr != null) {
					isr.close();
				}
			} catch (IOException e) {
				log.error(e.getMessage(), e);
			}

		}
	}

	/**
	 * 保存Properties
	 * 
	 * @param writer 檔輸出物件
	 */
	public void store(Writer writer) {
		BufferedWriter bw = new BufferedWriter(writer);
		try {
			for (String key : keyList) {
				bw.write(key);
				if (valueMap.containsKey(key)) {
					bw.write(EQUAL);
					bw.write(valueMap.get(key));
				}
				bw.newLine();
			}
		} catch (IOException e) {
			log.error(e.getMessage(), e);
		} finally {
			try {
				if (bw != null) {
					bw.close();
				}
			} catch (IOException e) {
				log.error(e.getMessage(), e);
			}
		}
	}

	/**
	 * 讀取流轉為行
	 * 
	 * @param istream 輸入流
	 */
	private void saveLine(String line) {
		if (line.trim().startsWith(POUND)) {
			keyList.add(line);
		} else {
			if (line.indexOf(EQUAL) > -1) {
				String k = line.substring(0, line.indexOf(EQUAL)).trim();
				String v = line.substring(line.indexOf(EQUAL) + 1).trim();
				keyList.add(k);
				valueMap.put(k, v);
			} else {
				keyList.add(line);
			}
		}

	}

	@Override
	public String toString() {
		return valueMap.toString();
	}
}
