package com.digiwin.mobile.mobileuibot.common.datetime;

import com.digiwin.mobile.mobileuibot.common.string.StringUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.time.DateUtils;

import javax.validation.constraints.NotNull;
import javax.validation.constraints.Pattern;
import java.text.SimpleDateFormat;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.ZoneOffset;
import java.time.format.DateTimeFormatter;
import java.util.Calendar;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;

/**
 * <p>功能描述：日期与日期时间处理工具类</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: DateTimeUtil
 * @Author: Zaregoto
 * @Date: 2021/4/21 20:17
 */
@Slf4j
public final class DateTimeUtil {

    public static final String DEFAULT_DATE_PATTERN = "yyyy-MM-dd";
    public static final DateTimeFormatter DEFAULT_DATE_FORMATTER =
            DateTimeFormatter.ofPattern(DEFAULT_DATE_PATTERN);

    public static final String DEFAULT_DATETIME_PATTERN = "yyyy-MM-dd HH:mm:ss";
    public static final DateTimeFormatter DEFAULT_DATETIME_FORMATTER =
            DateTimeFormatter.ofPattern(DEFAULT_DATETIME_PATTERN);

    public static final String MIN_DATETIME_USE_DEFAULT_PATTERN = "1970-01-01 00:00:00";

    public static final String MOBILE_SHOW_DEFAULT_DATE_PATTERN = "yyyy.MM.dd";
    public static final DateTimeFormatter MOBILE_SHOW_DEFAULT_DATE_FORMATTER =
            DateTimeFormatter.ofPattern(MOBILE_SHOW_DEFAULT_DATE_PATTERN);

    public static LocalDateTime parseUseDefaultPattern(String dateTime) {
        if (null == dateTime || dateTime.trim().isEmpty()) {
            return LocalDateTime.MIN;
        }
        return LocalDateTime.parse(dateTime, DEFAULT_DATETIME_FORMATTER);
    }

    public static LocalDate parseLocalDateUseDefaultPattern(String date) {
        if (null == date || date.trim().isEmpty()) {
            return LocalDate.MIN;
        }
        return LocalDate.parse(date, DEFAULT_DATE_FORMATTER);
    }

    /**
     * 获取今天的日期字符串，使用移动端默认展示格式
     *
     * @return 使用移动端默认展示格式的今日日期字符串
     */
    public static String getTodayDateTextInMobileShowDefaultPattern() {
        LocalDate localDate = LocalDate.now();
        return String.format("%s.%s.%s", localDate.getYear(),
                DateTimeUtil.getFullMonth(localDate.getMonthValue()),
                DateTimeUtil.getFullDate(localDate.getDayOfMonth()));
    }

    /**
     * 根据传入值，返回使用移动端默认展示格式的日期字符串
     *
     * @param defaultDatePatternText 使用系统默认日期格式的日期字符串。格式参考<code>DEFAULT_DATE_PATTERN</code>
     * @return 使用移动端默认展示格式的日期字符串
     */
    public static String getDateTextInMobileShowDefaultPattern(String defaultDatePatternText) {
        if (null == defaultDatePatternText || defaultDatePatternText.isEmpty()) {
            return "";
        }
        LocalDate localDate;
        try {
            localDate = LocalDate.parse(defaultDatePatternText, DEFAULT_DATE_FORMATTER);
        } catch (Exception e) {
            return defaultDatePatternText;
        }
        return String.format("%s.%s.%s", localDate.getYear(),
                DateTimeUtil.getFullMonth(localDate.getMonthValue()),
                DateTimeUtil.getFullDate(localDate.getDayOfMonth()));
    }

    /**
     * 获取时距。
     * <p>1，显示开始日期-结束日期：MM.DD-MM.DD</p>
     * <p>2，当某个日期不是今年的日期，则显示所属年份。</p>
     * <p>ex：今年是2021年，开始日期是2020年12月1日，结束日期是2021年6月30日，则展示2020.12.01-06.30</p>
     * <p>ex：今年是2024年，开始日期是2023年12月1日，结束日期是2023年12月1日，则展示2023.12.01</p>
     *
     * @param startTime 开始时间。格式为YYYY-MM-DD HH:MM:SS
     * @param endTime   结束时间。格式为YYYY-MM-DD HH:MM:SS
     * @return 格式化后的时距文字
     */
    public static String getTimeDistanceFromStartToEnd(String startTime, String endTime) {
        LocalDateTime start = DateTimeUtil.parseUseDefaultPattern(startTime);
        LocalDateTime end = DateTimeUtil.parseUseDefaultPattern(endTime);

        String dateThisYearFormat = "%s.%s";
        String dateNotThisYearFormat = "%s.%s.%s";
        LocalDate startDate = start.toLocalDate();
        LocalDate endDate = end.toLocalDate();
        LocalDate today = LocalDate.now();

        String startDateString = startDate.getYear() == today.getYear() ?
                String.format(dateThisYearFormat, DateTimeUtil.getFullMonth(startDate.getMonthValue()),
                        DateTimeUtil.getFullDate(startDate.getDayOfMonth())) :
                String.format(dateNotThisYearFormat, startDate.getYear(),
                        DateTimeUtil.getFullMonth(startDate.getMonthValue()),
                        DateTimeUtil.getFullDate(startDate.getDayOfMonth()));

        String endDateString = endDate.getYear() == today.getYear() ?
                String.format(dateThisYearFormat, DateTimeUtil.getFullMonth(endDate.getMonthValue()),
                        DateTimeUtil.getFullDate(endDate.getDayOfMonth())) :
                String.format(dateNotThisYearFormat, endDate.getYear(),
                        DateTimeUtil.getFullMonth(endDate.getMonthValue()),
                        DateTimeUtil.getFullDate(endDate.getDayOfMonth()));

        // 修复 禅道 bug-069
        if (startDateString.trim().equalsIgnoreCase(endDateString.trim())) {
            return startDateString;
        } else {
            return String.format("%s-%s", startDateString, endDateString);
        }
    }

    /**
     * 获取时距。
     *
     * @param startDate
     * @param endDate
     * @return
     * @see DateTimeUtil#getTimeDistanceFromStartToEnd(String, String)
     */
    public static String getDateDistanceFromStartToEnd(String startDate, String endDate) {
        return DateTimeUtil.getTimeDistanceFromStartToEnd(startDate + " 00:00:00",
                endDate + " 00:00:00");
    }

    /**
     * 获取年份字符串，不足4位的在前面补零
     *
     * @param year
     * @return
     */
    public static String getFullYear(Integer year) {
        return String.format("%04d", year);
    }

    /**
     * 获取月份字符串，不足2位的在前面补零
     *
     * @param month
     * @return
     */
    public static String getFullMonth(Integer month) {
        return String.format("%02d", month);
    }

    /**
     * 获取日期字符串，不足2位的在前面补零
     *
     * @param day
     * @return
     */
    public static String getFullDate(Integer day) {
        return String.format("%02d", day);
    }

    /**
     * 获取小时字符串，不足2位的在前面补零
     *
     * @param hour
     * @return
     */
    public static String getFullHour(Integer hour) {
        return String.format("%02d", hour);
    }

    /**
     * 获取分钟字符串，不足2位的在前面补零
     *
     * @param minute
     * @return
     */
    public static String getFullMinute(Integer minute) {
        return String.format("%02d", minute);
    }

    /**
     * 判断传入时间（使用默认时间格式）是否是今天
     *
     * @param otherTimeStringUseDefaultPattern
     * @return true-是，false-不是
     */
    public static Boolean isTodayTime(String otherTimeStringUseDefaultPattern) {
        LocalDateTime otherTime =
                DateTimeUtil.parseUseDefaultPattern(otherTimeStringUseDefaultPattern);
        LocalDate otherDate = otherTime.toLocalDate();
        return otherDate.equals(LocalDate.now());
    }

    /**
     * 判断传入日期时间是否早于当前系统时间（无时区设置，时区依靠运行时环境）
     *
     * @param otherDateTimeStr 用于判断的日期时间字符串。格式：YYYY-MM-DD hh:mm:ss
     * @return true-早于当前系统时间，false-不早于当前系统时间
     */
    public static Boolean isBeforeNowDateTime(String otherDateTimeStr) {
        LocalDateTime localDateTime = DateTimeUtil.parseUseDefaultPattern(otherDateTimeStr);
        return localDateTime.isBefore(LocalDateTime.now());
    }

    /**
     * 判断传入日期时间是否晚于当前系统时间（无时区设置，时区依靠运行时环境）
     *
     * @param otherDateTimeStr 用于判断的日期时间字符串。格式：YYYY-MM-DD hh:mm:ss
     * @return true-早于当前系统时间，false-不早于当前系统时间
     */
    public static Boolean isAfterNowDatetime(String otherDateTimeStr) {
        LocalDateTime localDateTime = DateTimeUtil.parseUseDefaultPattern(otherDateTimeStr);
        return localDateTime.isAfter(LocalDateTime.now());
    }

    /**
     * 判断传入日期时间是否等于当前系统时间（无时区设置，时区依靠运行时环境）
     *
     * @param otherDateTimeStr 用于判断的日期时间字符串。格式：YYYY-MM-DD hh:mm:ss
     * @return true-早于当前系统时间，false-不早于当前系统时间
     */
    public static Boolean isEqualNowDatetime(String otherDateTimeStr) {
        LocalDateTime localDateTime = DateTimeUtil.parseUseDefaultPattern(otherDateTimeStr);
        return localDateTime.isEqual(LocalDateTime.now());
    }

    /**
     * 判断传入的日期是否早于当前系统时间（无时区设置，时区依靠运行时环境）
     *
     * @param otherDateStr 用于判断的日期字符串。格式：YYYY-MM-DD
     * @return true-早于当前系统时间，false-不早于当前系统时间
     */
    public static Boolean isBeforeNowDate(final String otherDateStr) {
        LocalDate otherDate = DateTimeUtil.parseLocalDateUseDefaultPattern(otherDateStr);
        return otherDate.isBefore(LocalDate.now());
    }

    /**
     * 判断传入的日期是否晚于当前系统时间（无时区设置，时区依靠运行时环境）
     *
     * @param otherDateStr 用于判断的日期字符串。格式：YYYY-MM-DD
     * @return true-晚于当前系统时间，false-不晚于当前系统时间
     */
    public static Boolean isAfterNowDate(final String otherDateStr) {
        LocalDate otherDate = DateTimeUtil.parseLocalDateUseDefaultPattern(otherDateStr);
        return otherDate.isAfter(LocalDate.now());
    }

    /**
     * 判断传入的日期是否等于当前系统时间（无时区设置，时区依靠运行时环境）
     *
     * @param otherDateStr 用于判断的日期字符串。格式：YYYY-MM-DD
     * @return true-等于当前系统时间，false-不等于当前系统时间
     */
    public static Boolean isEqualNowDate(final String otherDateStr) {
        LocalDate otherDate =
                DateTimeUtil.parseLocalDateUseDefaultPattern(otherDateStr);
        return otherDate.isEqual(LocalDate.now());
    }

    /**
     * 根据传入的时间字符串（使用默认时间格式），返回对应的月日信息字符串，格式为MM.DD
     *
     * @param otherTimeStringUseDefaultPattern
     * @return
     */
    public static String getMonthDateOfTime(String otherTimeStringUseDefaultPattern) {
        LocalDateTime localDateTime =
                DateTimeUtil.parseUseDefaultPattern(otherTimeStringUseDefaultPattern);
        return String.format("%s.%s", DateTimeUtil.getFullMonth(localDateTime.getMonthValue()),
                DateTimeUtil.getFullDate(localDateTime.getDayOfMonth()));
    }

    /**
     * 根据传入的时间字符串（使用默认时间格式），返回对应的年月信息字符串，格式为YYYY-MM
     *
     * @param otherTimeStringUseDefaultPattern
     * @return
     */
    public static String getYearMonthOfTime(String otherTimeStringUseDefaultPattern) {
        LocalDateTime localDateTime =
                DateTimeUtil.parseUseDefaultPattern(otherTimeStringUseDefaultPattern);
        return String.format("%s-%s", DateTimeUtil.getFullDate(localDateTime.getYear()),
                DateTimeUtil.getFullDate(localDateTime.getMonthValue()));
    }

    /**
     * 使用传入的指定格式返回当前日期
     *
     * @param pattern
     * @return
     */
    public static String getTodayUseSpecifiedPattern(String pattern) {
        DateTimeFormatter dateFormatter = DateTimeFormatter.ofPattern(pattern);
        return LocalDate.now().format(dateFormatter);
    }

    /**
     * 使用默认时间格式返回当前时间字符串
     *
     * @return
     */
    public static String getTodayTimeUseDefaultPattern() {
        return LocalDateTime.now().format(DEFAULT_DATETIME_FORMATTER);
    }

    /**
     * 使用默认日期格式返回当前日期字符串
     *
     * @return
     */
    public static String getTodayUseDefaultPattern() {
        return LocalDate.now().format(DEFAULT_DATE_FORMATTER);
    }

    /**
     * 使用默认日期格式返回当前日期字符串
     *
     * @return
     */
    public static String getTodayByPattern(String pattern) {
        if (StringUtils.isBlank(pattern)) {
            pattern = "yyyy-MM-dd";
        }
        DateTimeFormatter DEFAULT_DATE_FORMATTER =
                DateTimeFormatter.ofPattern(pattern);
        return LocalDate.now().format(DEFAULT_DATE_FORMATTER);
    }

    /**
     * 使用默认日期格式返回日历组件选中的日期字符串
     *
     * @param year
     * @param month
     * @param date
     * @return
     */
    public static String getInputCalendarDateUseDefaultPattern(
            @NotNull(message = "年份不能为空") @Pattern(regexp = "^\\d{4}$", message = "年份格式不符合") String year,
            @NotNull(message = "月份不能为空") @Pattern(regexp = "^0\\d[1-9]|\\d[10-12]$", message = "月份格式不符合") String month,
            @NotNull(message = "日期不能为空") @Pattern(regexp = "^0\\d[1-9]|\\d[10-31]$", message = "日期格式不符合") String date) {
        LocalDate localDate =
                LocalDate.of(Integer.parseInt(year), Integer.parseInt(month), Integer.parseInt(date));
        return DEFAULT_DATE_FORMATTER.format(localDate);
    }

    /**
     * 使用日历组件传入的数据，返回需要提交的数据。传入格式为：YYYY-MM-DD 00:00:00.000
     *
     * @return
     */
    public static String getInputCalendarDateUseDefaultPattern(
            @NotNull(message = "日期不能为空") @Pattern(regexp = "^\\d{4}-\\d{2}-\\d{2}\\s00:00:00.000$", message = "日期格式不符合") String inputCalendarDate) {
        return inputCalendarDate.substring(0, 10).replace(".", "-");
    }

    public static boolean isDateUseDefaultPattern(String dateToTest) {
        java.util.regex.Pattern p = java.util.regex.Pattern.compile(
                "([0-9]{3}[1-9]|[0-9]{2}[1-9][0-9]{1}|[0-9]{1}[1-9][0-9]{2}|[1-9][0-9]{3})-(((0[13578]|1[02])-(0[1-9]|[12][0-9]|3[01]))|((0[469]|11)-(0[1-9]|[12][0-9]|30))|(02-(0[1-9]|[1][0-9]|2[0-9])))");
        return p.matcher(dateToTest).find();
    }

    /**
     * 计算两个日期之间相差的天数。比如2021-07-01与2021-07-11，计算的结果为10，表示这两个日期相差10天。
     * 若任意一个日期格式不正确，无法计算，返回结果为-1
     * 若任意一个日期为空，无法计算，返回结果为-1
     *
     * @param startDateStringUseDefaultPattern
     * @param endDateStringUseDefaultPattern
     * @return 两个日期之间相差的天数
     */
    public static long getDatePeriod(String startDateStringUseDefaultPattern,
                                     String endDateStringUseDefaultPattern) {
        int errorResult = -1;
        if (!DateTimeUtil.isDateUseDefaultPattern(startDateStringUseDefaultPattern)
                || !DateTimeUtil.isDateUseDefaultPattern(endDateStringUseDefaultPattern)) {
            return errorResult;
        }
        if (StringUtil.isEmpty(startDateStringUseDefaultPattern) || StringUtil.isEmpty(endDateStringUseDefaultPattern)) {
            return errorResult;
        }
        LocalDate startDate = DateTimeUtil.parseLocalDateUseDefaultPattern(startDateStringUseDefaultPattern);
        LocalDate endDate = DateTimeUtil.parseLocalDateUseDefaultPattern(endDateStringUseDefaultPattern);

        return Math.abs(endDate.toEpochDay() - startDate.toEpochDay());
    }

    /**
     * 判断第一个时间是否早于第二个时间
     *
     * @param dateTimeStr1
     * @param dateTimeStr2
     * @return
     */
    public static Boolean compareTime(String dateTimeStr1, String dateTimeStr2) {
        if (!DateTimeUtil.isDateUseDefaultPattern(dateTimeStr1)
                || !DateTimeUtil.isDateUseDefaultPattern(dateTimeStr2)) {
            return false;
        }
        LocalDateTime dateTime1 = DateTimeUtil.parseUseDefaultPattern(dateTimeStr1);
        LocalDateTime dateTime2 = DateTimeUtil.parseUseDefaultPattern(dateTimeStr2);
        return dateTime1.isBefore(dateTime2);
    }

    /**
     * 计算两个日期之间相差的天数。
     *
     * @param startDateTimeStr 格式：YYYY-MM-DD hh:mm:ss
     * @param endDateTimeStr   格式：YYYY-MM-DD hh:mm:ss
     * @return end比start晚的天数，早则返回0，不足一天向下取余
     */
    public static int getDateTimePeriod(String startDateTimeStr, String endDateTimeStr) {
        if (!DateTimeUtil.isDateUseDefaultPattern(startDateTimeStr)
                || !DateTimeUtil.isDateUseDefaultPattern(endDateTimeStr)) {
            return 0;
        }
        LocalDateTime startDateTime = DateTimeUtil.parseUseDefaultPattern(startDateTimeStr);
        LocalDateTime endDateTime = DateTimeUtil.parseUseDefaultPattern(endDateTimeStr);
        if (startDateTime.isAfter(endDateTime)) {
            return 0;
        }
        long secondPeriod =
                endDateTime.toEpochSecond(ZoneOffset.of("+8")) - startDateTime.toEpochSecond(
                        ZoneOffset.of("+8"));
        return (int) (secondPeriod / 60 / 60 / 24);
    }


    public static String getSevthDate(String pattern) {
        if (StringUtils.isBlank(pattern)) {
            pattern = "yyyy-MM-dd";
        }
        SimpleDateFormat sdf = new SimpleDateFormat(pattern);
        Date date = DateUtils.addDays(new Date(), +7);
        return sdf.format(date);
    }

    public static String getOneMonthDate(String pattern) {
        if (StringUtils.isBlank(pattern)) {
            pattern = "yyyy-MM-dd";
        }
        SimpleDateFormat sdf = new SimpleDateFormat(pattern);
        Date date = DateUtils.addDays(new Date(), +30);
        return sdf.format(date);
    }

    public static String currentDateAddYears(String pattern, int years) {
        if (StringUtils.isBlank(pattern)) {
            pattern = "yyyy-MM-dd";
        }

        SimpleDateFormat sdf = new SimpleDateFormat(pattern);
        Date date = DateUtils.addYears(new Date(), years);
        return sdf.format(date);
    }

    /**
     * 获取之前monthsToSubtract月的第一天
     *
     * @param pattern
     * @param monthsToSubtract the months to subtract, may be negative
     * @return
     */
    public static String firstDayOfPreviousMonth(String pattern, int monthsToSubtract) {
        LocalDate date = monthsToSubtract == 0 ? LocalDate.now() : LocalDate.now().minusMonths(monthsToSubtract);
        LocalDate firstDay = LocalDate.of(date.getYear(), date.getMonth(), 1);
        LocalDateTime startTime = LocalDateTime.of(firstDay, LocalTime.MIN);
        return startTime.format(DateTimeFormatter.ofPattern(pattern));
    }

    /**
     * 获取之前monthsToSubtract月的时间
     *
     * @param dateTimeFormatter
     * @param monthsToSubtract  the months to subtract, may be negative
     * @return
     */
    public static String dateOfPreviousMonth(DateTimeFormatter dateTimeFormatter, int monthsToSubtract) {
        LocalDate date = monthsToSubtract == 0 ? LocalDate.now() : LocalDate.now().minusMonths(monthsToSubtract);
        return date.format(dateTimeFormatter);
    }

    public static String convert(String time, String defaultDatePattern) {
        StringBuffer newTime = new StringBuffer();
        if (time.contains(".")) {
            String[] times = time.split("\\.");
            newTime.append(times[0]).append("-").append(times[1]).append("-").append(times[2]);
            return newTime.toString();
        } else {
            return time;
        }
    }

    public static Map<String, Object> getNowDateMap() {
        Map<String, Object> dateMap = new HashMap<>(3);
        LocalDate localDate = LocalDate.now();
        dateMap.put("year", String.valueOf(localDate.getYear()));
        dateMap.put("month", String.valueOf(localDate.getMonthValue()));
        dateMap.put("date", String.valueOf(localDate.getDayOfMonth()));
        return dateMap;
    }

    /**
     * 比较time的时分
     *
     * @param startTime
     * @param endTime
     * @return Boolean
     * @author yanfeng
     */
    public static Boolean compareTimeHourMinute(String startTime, String endTime) {
        LocalDateTime start =
                DateTimeUtil.parseUseDefaultPattern(startTime);
        LocalDateTime end =
                DateTimeUtil.parseUseDefaultPattern(endTime);

        String startTimeText = String.format("%s:%s", DateTimeUtil.getFullHour(start.getHour()),
                DateTimeUtil.getFullMinute(start.getMinute()));
        String endTimeText = String.format("%s:%s", DateTimeUtil.getFullHour(end.getHour()),
                DateTimeUtil.getFullMinute(end.getMinute()));
        int i = startTimeText.compareTo(endTimeText);
        return i >= 0;
    }

    public static int getWorkingDaysBetweenDates(String startDateStr, String endDateStr) {
        int workingDays = 0;
        try {
            SimpleDateFormat sdf = new SimpleDateFormat(DEFAULT_DATE_PATTERN);
            Date startDate = sdf.parse(startDateStr);
            Date endDate = sdf.parse(endDateStr);

            Calendar startCal = Calendar.getInstance();
            startCal.setTime(startDate);
            Calendar endCal = Calendar.getInstance();
            endCal.setTime(endDate);

            while (startCal.before(endCal) || startCal.equals(endCal)) {
                if (startCal.get(Calendar.DAY_OF_WEEK) != Calendar.SATURDAY && startCal.get(Calendar.DAY_OF_WEEK) != Calendar.SUNDAY) {
                    workingDays++;
                }
                startCal.add(Calendar.DAY_OF_MONTH, 1);
            }
        } catch (Exception e) {
            log.error("getWorkingDaysBetweenDates error ", e);
            ;
        }
        return workingDays;
    }
}