package com.digiwin.mobile.mobileuibot.common.file;

import com.digiwin.mobile.mobileuibot.common.string.RegexUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.io.ClassPathResource;
import org.springframework.util.StringUtils;
import org.springframework.web.multipart.MultipartFile;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.util.Objects;

/**
 * <p>功能描述：文件工具类</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: FileUtil
 * @Author: Zaregoto
 * @Date: 2021/6/7 14:05
 */
public final class FileUtil {

    private static final Logger logger = LoggerFactory.getLogger(FileUtil.class);

    private static final String IMAGE_JPEG = "jpeg";

    private static final String IMAGE_JPG = "jpg";

    private static final String IMAGE_BMP = "bmp";

    private static final String IMAGE_PNG = "png";
    /**
     * 图片类文件后缀名
     */
    private static final String[] FILE_IMAGE_EXTENSIONS = new String[]{IMAGE_JPEG, IMAGE_JPG, IMAGE_BMP, IMAGE_PNG};


    public static String readText(String filePath) {
        InputStream inputStream = null;
        StringBuilder data = new StringBuilder();
        try {
            ClassPathResource classPathResource = new ClassPathResource(filePath);
            inputStream = classPathResource.getInputStream();
            byte[] buffer = new byte[4096];
            int count = 0;
            while ((count = inputStream.read(buffer)) > 0) {
                data.append(new String(buffer, 0, count, StandardCharsets.UTF_8));
            }
        } catch (IOException e) {
            e.printStackTrace();
            logger.error(e.getLocalizedMessage());
        } finally {
            if (null != inputStream) {
                try {
                    inputStream.close();
                } catch (IOException e) {
                    e.printStackTrace();
                    logger.error(e.getLocalizedMessage());
                }
            }
        }
        return data.toString();
    }

    /**
     * MultipartFile转File
     *
     * @param file
     * @return
     */
    public static File multipartFileToFile(MultipartFile file) {
        File toFile = null;
        try {
            if (file.getName().isEmpty() || file.getSize() <= 0) {
                file = null;
            } else {
                InputStream ins = null;
                ins = file.getInputStream();
                toFile = new File(Objects.requireNonNull(file.getOriginalFilename()));
                inputStreamToFile(ins, toFile);
                ins.close();
            }
        } catch (IOException e) {
            e.printStackTrace();
        }
        return toFile;
    }

    /**
     * 获取流文件
     *
     * @param ins
     * @param file
     */
    private static void inputStreamToFile(InputStream ins, File file) {
        OutputStream os = null;
        try {
            os = Files.newOutputStream(file.toPath());
            int bytesRead = 0;
            byte[] buffer = new byte[8192];
            while ((bytesRead = ins.read(buffer, 0, 8192)) != -1) {
                os.write(buffer, 0, bytesRead);
            }
            os.close();
            ins.close();
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            if (os != null) {
                try {
                    os.close();
                } catch (IOException e) {
                    logger.error("OutputStream close exception");
                }
            }
            if (ins != null) {
                try {
                    ins.close();
                } catch (IOException e) {
                    logger.error("InputStream close exception");
                }
            }

        }
    }

    /**
     * 获取文件后缀名
     *
     * @param fileName 文件完整名称。如1.pdf、2.docx
     * @return 文件后缀名。英文小写，不带点。如：pdf,docx
     */
    public static String getFileExtension(String fileName) {
        int fileExtensionIdx = fileName.lastIndexOf(".");
        if (fileExtensionIdx == -1) {
            return "";
        }
        String fileExtension = fileName.substring(fileExtensionIdx + 1);
        if (RegexUtil.isMatch(RegexUtil.PATTERN_REGEX_NUMBER_ALPHABET_WORD, fileExtension)) {
            return fileExtension;
        } else {
            return "";
        }
    }

    /**
     * 根据文件扩展名判断文件是否图片格式
     *
     * @param fileName 文件名称
     * @return 是否是图片
     */
    public static boolean isImage(String fileName) {
        String extension = FileUtil.getFileExtension(fileName);
        for (String e : FILE_IMAGE_EXTENSIONS) {
            if (extension.toLowerCase().equals(e)) {
                return true;
            }
        }
        return false;
    }

    /**
     * 根据文件扩展名判断文件是否图片格式
     *
     * @param fileExtendName 文件名称
     * @return 是否是图片
     */
    public static boolean fileExtendNameIsImage(String fileExtendName) {
        if (!StringUtils.hasLength(fileExtendName)) {
            return false;
        }
        for (String e : FILE_IMAGE_EXTENSIONS) {
            if (fileExtendName.toLowerCase().equals(e)) {
                return true;
            }
        }
        return false;
    }
}
