package com.digiwin.mobile.mobileuibot.common.math;

import java.math.BigDecimal;
import java.math.RoundingMode;

/**
 * 数学计算工具类
 *
 * @author zhangjj
 * @date 2021/6/23 0023 17:39
 */
public class MathUtil {

    /**
     * 取小数点后保留几位小数后的值。
     * 2021.07.07 调整：向下取整，不做四舍五入
     *
     * @param d
     * @param point
     * @return
     */
    public static double getDecimalPoint(double d, int point) {
        BigDecimal bg = BigDecimal.valueOf(d);
        return bg.setScale(point, RoundingMode.FLOOR).doubleValue();
    }

    /**
     * 取小数点后保留几位小数后的值。
     *
     * @param d
     * @param point
     * @param hasRound
     * @return
     */
    public static double getDecimalPoint(double d, int point, Boolean hasRound) {
        BigDecimal bg = BigDecimal.valueOf(d);
        double f = 0.0;
        /**
         *hasRound：为true则需要做四舍五入，为false则走getDecimalPoint方法
         *ROUND_UP：非0时，舍弃小数后（整数部分）加1，比如12.49结果为13，-12.49结果为 -13
         *ROUND_DOWN：直接舍弃小数
         *ROUND_CEILING：如果 BigDecimal 是正的，则做 ROUND_UP 操作；如果为负，则做 ROUND_DOWN 操作 （一句话：取附近较大的整数）
         *ROUND_FLOOR: 如果 BigDecimal 是正的，则做 ROUND_DOWN 操作；如果为负，则做 ROUND_UP 操作（一句话：取附近较小的整数）
         *ROUND_HALF_UP：四舍五入（取更近的整数）
         *ROUND_HALF_DOWN：跟ROUND_HALF_UP 差别仅在于0.5时会向下取整
         *ROUND_HALF_EVEN：取最近的偶数
         *ROUND_UNNECESSARY：不需要取整，如果存在小数位，就抛ArithmeticException 异常
         */
        if (hasRound) {
            //先上取整，四舍五入
            f = bg.setScale(point, RoundingMode.HALF_UP).doubleValue();
        } else {
            //向下取整，不做四舍五入. 对于正数而言，ROUND_UP  = ROUND_CEILING，ROUND_DOWN = ROUND_FLOOR
            f = getDecimalPoint(d, point);
        }
        return f;
    }

    /**
     * double类型计算，避免精度丢失。
     *
     * @param d1
     * @param d2
     * @param operatorType,目前支持“+”、“-”、“*”、“/”
     * @return
     */
    public static double doubleMathCalculation(double d1, double d2, String operatorType) {
        BigDecimal bigDecimal1 = new BigDecimal(String.valueOf(d1));
        BigDecimal bigDecimal2 = new BigDecimal(String.valueOf(d2));
        double resultValue = 0.0;
        switch (operatorType) {
            case "+":
                resultValue = bigDecimal1.add(bigDecimal2).doubleValue();
                break;
            case "-":
                resultValue = bigDecimal1.subtract(bigDecimal2).doubleValue();
                break;
            case "*":
                resultValue = bigDecimal1.multiply(bigDecimal2).doubleValue();
                break;
            case "/":
                // 除法保留2位小数，向上取整，四舍五入
                resultValue = bigDecimal1.divide(bigDecimal2, 2, BigDecimal.ROUND_HALF_UP).doubleValue();
                break;
            default:
                break;
        }
        return resultValue;
    }

    /**
     * 针对Double转为String的字符串，去除尾部多余的0字符串。
     * 如：传入"1232424.3383023248000"，返回结果是"1232424.3383023248"
     *
     * @param doubleString
     * @return
     */
    public static String stripTrailingZerosInDoubleString(String doubleString) {
        if (null == doubleString || doubleString.trim().isEmpty()) {
            return "";
        }
        if ("0".equals(doubleString)) {
            return "0";
        }
        int length = doubleString.length();
        int index = 0;
        for (int i = length - 1; i >= 0; i--) {
            char c = doubleString.charAt(i);
            if (c != '0') {
                if (c == '.') {
                    index = i;
                } else {
                    index = i + 1;
                }
                break;
            }
        }
        String trailedZero = doubleString.substring(0, index);
        length = trailedZero.length();
        if (trailedZero.charAt(length - 1) == '.') {
            trailedZero = trailedZero.substring(0, length - 1);
        }
        return trailedZero;
    }

    public enum IrregularValueSearchStrategy {
        /**
         * 基于统计的方法：通过使用3σ原则，将距离均值超过3倍标准差的数值判定为异常值。
         */
        STATS_BASED,

        /**
         * 基于四分位数的方法：通过计算数据的四分位数（Q1，Q2，Q3）和四分位距（IQR）来判断异常值。
         * 将小于Q1-1.5IQR或大于Q3+1.5IQR的数值判定为异常值。
         */
        QUARTILE_BASED,

    }

    public static Double[] findIrregularValues(Double[] rawValues, IrregularValueSearchStrategy strategy) {
        // TODO
        return null;
    }


    /**
     * 根据数据的位数与取整策略，得到一个5的倍数的结果。<p>
     * 数据预处理：<p>
     * 1.个位数或两位数时：往下正式处理；<p>
     * 2.超过两位数时，取最高位和次高位两个数，变成两位数，走第1步处理。最后结束补尾零；<p>
     * 3.小数时，先乘以10的倍数变成个位数，再走第1步处理。最后除以10恢复成小数；<p>
     * 正式处理：<p>
     * 1.若是1，表示向上取整。取大于输入值，且是5的倍数的数<p>
     * 2.若是0，表示自动依数据值取整。若是正数时，除5取余小于3，则向下取整5的倍数，大于等于3则向上取整5的倍数；若是负数则相反，除5取余小于3，则向上取整5的倍数，大于等于3则向下取整5的倍数<p>
     * 3.若是-1，表示向下取整。取小于输入值，且是5的倍数的数。<p>
     * <p>
     * 示例参考MathUtilTest#testGetFiveMultiplesRoundedValueByDigits方法
     *
     * @param value
     * @param roundingStrategy 取整策略(rs)：1表示向上取整；0表示自动取整；-1表示向下取整<br/>
     * @return
     */
    public static Double getFiveMultiplesRoundedValueByDigits(Double value, int roundingStrategy) {
        if (value == 0.0) {
            return value;
        }
        if (roundingStrategy == 0) {
            Double top2DigitsNumber = preprocessRoundedValueByDigits(value);
            if (value < 0) {
                if (top2DigitsNumber % 5 > 2) {
                    return ceilingToFiveMultiplesByDigitsIgnoreSign(value) * -1.0;
                } else {
                    return flooringToFiveMultiplesByDigitsIgnoreSign(value) * -1.0;
                }
            } else {
                if (top2DigitsNumber % 5 < 3) {
                    return flooringToFiveMultiplesByDigitsIgnoreSign(value);
                } else {
                    return ceilingToFiveMultiplesByDigitsIgnoreSign(value);
                }
            }
        } else if (roundingStrategy == 1) {
            if (value < 0) {
                return flooringToFiveMultiplesByDigitsIgnoreSign(value) * -1.0;
            } else {
                return ceilingToFiveMultiplesByDigitsIgnoreSign(value);
            }
        } else if (roundingStrategy == -1) {
            if (value < 0) {
                return ceilingToFiveMultiplesByDigitsIgnoreSign(value) * -1.0;
            } else {
                return flooringToFiveMultiplesByDigitsIgnoreSign(value);
            }
        } else {
            return value;
        }
    }

    /**
     * 计算将输入值的绝对值挪动几次小数点，可以大于1
     *
     * @param value
     * @return
     */
    public static int getMinimumTimesToGreaterThanOne(final Double value) {
        int n = 0;
        if (value == 0.0 || value >= 1.0) {
            return n;
        }
        Double prevValue = Math.abs(value);
        while (prevValue < 1.0) {
            prevValue *= 10.0;
            n++;
        }
        return n;
    }

    /**
     * 根据数据的位数与取整策略，得到一个2的倍数的结果。<p>
     *
     * @param value
     * @param roundingStrategy 取整策略(rs)：1表示向上取整；0表示自动取整；-1表示向下取整<br/>
     * @return
     */
    public static Double getTwoMultiplesRoundedValueByDigits(Double value, int roundingStrategy) {
        // TODO
        return 0D;
    }

    /**
     * 忽略符号，依数据位数，向上取5的整数。若小于0，则会转成正数后再处理。
     *
     * @param value 待取整的浮点数
     * @return 向上最接近5的倍数的数
     */
    private static Double ceilingToFiveMultiplesByDigitsIgnoreSign(final Double value) {
        Double toRoundValue = value;
        if (value < 0.0) {
            toRoundValue = value * -1.0;
        }
        if (toRoundValue < 1.0) {
            int n = MathUtil.getMinimumTimesToGreaterThanOne(toRoundValue);
            Double newToRoundValue = toRoundValue * Math.pow(10, n);
            long intValue = newToRoundValue.longValue();
            long ceiledIntValue;
            if (intValue % 5L == 0) {
                ceiledIntValue = intValue;
            } else {
                ceiledIntValue = (intValue / 5 + 1) * 5;
            }
            // 如果比原值还小，则再加1倍
            if (ceiledIntValue / 10.0 < toRoundValue) {
                ceiledIntValue = (intValue / 5 + 1) * 5;
            }
            return new Double(ceiledIntValue / Math.pow(10, n));
        } else if (toRoundValue >= 1.0 && toRoundValue < 5.0) {
            // 数值在当前区间的，表示需要的倍数在0~1开区间之间
            Double times = Math.ceil(toRoundValue / 0.5);
            return new Double(times * 0.5);
        } else if (toRoundValue >= 5.0 && toRoundValue < 100.0) {
            long intValue = toRoundValue.longValue();
            long ceiledIntValue;
            if (intValue % 5L == 0) {
                ceiledIntValue = intValue;
            } else {
                ceiledIntValue = (intValue / 5L + 1L) * 5;
            }
            // 如果比原值还小，则再加1倍
            if (ceiledIntValue < toRoundValue) {
                ceiledIntValue = (intValue / 5L + 1L) * 5;
            }
            return new Double(ceiledIntValue);
        } else {
            long intValue = toRoundValue.longValue();
            String strIntValue = String.valueOf(intValue);
            int m = strIntValue.length();
            // 保留前2个高位数进行处理
            int n = m - 2;
            long twoDigitsIntValue = intValue / ((long) Math.pow(10, n));
            long ceiledIntValue;
            if (twoDigitsIntValue % 5L == 0) {
                ceiledIntValue = twoDigitsIntValue;
            } else {
                ceiledIntValue = (twoDigitsIntValue / 5L + 1L) * 5;
            }
            // 如果比原值还小，则再加1倍
            if (ceiledIntValue * Math.pow(10, n) < toRoundValue) {
                ceiledIntValue = (twoDigitsIntValue / 5L + 1L) * 5;
            }
            return new Double(ceiledIntValue * Math.pow(10, n));
        }
    }

    /**
     * 忽略符号，依数据位数，向下取5的整数。若小于0，则会转成正数后再处理。
     *
     * @param value 待取整的浮点数
     * @return 向上最接近5的倍数的数
     */
    private static Double flooringToFiveMultiplesByDigitsIgnoreSign(final Double value) {
        Double toRoundValue = value;
        if (value < 0.0) {
            toRoundValue = value * -1.0;
        }
        if (toRoundValue < 1.0) {
            int n = MathUtil.getMinimumTimesToGreaterThanOne(toRoundValue);
            Double newToRoundValue = toRoundValue * Math.pow(10, n);
            long intValue = newToRoundValue.longValue();
            long flooredIntValue;
            if (intValue % 5L == 0) {
                flooredIntValue = intValue;
            } else {
                flooredIntValue = (intValue / 5L) * 5;
            }
            return new Double(flooredIntValue / Math.pow(10, n));
        } else if (toRoundValue >= 1.0 && toRoundValue < 5.0) {
            // 数值在当前区间的，表示需要的倍数在0~1开区间之间
            Double times = Math.floor(toRoundValue / 0.5);
            return new Double(times * 0.5);
        } else if (toRoundValue >= 5.0 && toRoundValue < 100.0) {
            long intValue = toRoundValue.longValue();
            long flooredIntValue;
            if (intValue % 5L == 0) {
                flooredIntValue = intValue;
            } else {
                flooredIntValue = (intValue / 5L) * 5;
            }
            return new Double(flooredIntValue);
        } else {
            long intValue = toRoundValue.longValue();
            String strIntValue = String.valueOf(intValue);
            int m = strIntValue.length();
            // 保留前2个高位数进行处理
            int n = m - 2;
            long twoDigitsIntValue = intValue / ((long) Math.pow(10, n));
            long flooredIntValue;
            if (twoDigitsIntValue % 5L == 0) {
                flooredIntValue = twoDigitsIntValue;
            } else {
                flooredIntValue = (twoDigitsIntValue / 5L) * 5;
            }
            return new Double(flooredIntValue * Math.pow(10, n));
        }
    }

    /**
     * 依数据位数取整的数据预处理过程
     *
     * @param value
     * @return
     */
    private static Double preprocessRoundedValueByDigits(final Double value) {
        Double toPreprocessValue = value;
        if (value < 0.0) {
            toPreprocessValue = value * -1.0;
        }
        if (toPreprocessValue < 1.0) {
            int times = MathUtil.getMinimumTimesToGreaterThanOne(toPreprocessValue);
            Double newToRoundValue = toPreprocessValue * Math.pow(10, times);
            long intValue = newToRoundValue.longValue();
            return new Double(intValue);
        } else if (toPreprocessValue >= 1.0 && toPreprocessValue < 100.0) {
            return toPreprocessValue;
        } else {
            long intValue = toPreprocessValue.longValue();
            String strIntValue = String.valueOf(intValue);
            int m = strIntValue.length();
            // 保留前2个高位数进行处理
            int n = m - 2;
            long twoDigitsIntValue = intValue / ((long) Math.pow(10, n));
            return new Double(twoDigitsIntValue);
        }
    }

    /**
     * 判断一个浮点数是否可用整数表示
     *
     * @param d
     * @return
     */
    public static boolean canBeInteger(Double d) {
        if (d == null) {
            return false;
        }
        return d.longValue() == d.doubleValue();
    }

    /**
     * 返回两个数值中绝对值最大的一个
     *
     * @param d1 第一个数值。如果为null则默认视为0
     * @param d2 第二个数值。如果为null则默认视为0
     * @return
     */
    public static Double getMaxValueByAbs(final Double d1, final Double d2) {
        Double td1 = null == d1 ? 0D : d1;
        Double td2 = null == d2 ? 0D : d2;
        int compareResult = Double.compare(Math.abs(td1), Math.abs(td2));
        return compareResult >= 0 ? td1 : td2;
    }
}
