package com.digiwin.mobile.mobileuibot.common.string;

import java.util.ArrayList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * <p>功能描述：正则表达式工具类</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 * 参考文章：<br>
 * 1. https://blog.walterlv.com/post/match-web-url-using-regex.html<br>
 * 2. https://www.jianshu.com/p/583998f435d0
 *
 * @FileName: RegexUtil
 * @Author: zaregoto
 * @Date: 2022/12/6 13:40
 */
public final class RegexUtil {

    /**
     * 正则：IP地址
     */
    public static final String REGEX_IP = "^((?:(?:25[0-5]|2[0-4]\\d|[01]?\\d?\\d)\\.){3}(?:25[0-5]|2[0-4]\\d|[01]?\\d?\\d))$";
    public static final Pattern PATTERN_REGEX_IP = Pattern.compile(REGEX_IP);

    /**
     * 正则：域名
     */
    public static final String REGEX_DOMAIN = "^[a-zA-Z0-9][-a-zA-Z0-9]{0,62}(\\.[a-zA-Z0-9][-a-zA-Z0-9]{0,62})+\\.?$";
    public static final Pattern PATTERN_REGEX_DOMAIN = Pattern.compile(REGEX_DOMAIN);

    /**
     * 正则：汉字
     */
    public static final String REGEX_CHINESE_WORD = "^[\\u4e00-\\u9fa5]+$";
    public static final Pattern PATTERN_REGEX_CHINESE_WORD = Pattern.compile(REGEX_CHINESE_WORD);

    /**
     * 正则：数字、26个英文字母组成的字符串
     */
    public static final String REGEX_NUMBER_ALPHABET_WORD = "^[A-Za-z0-9]+$";
    public static final Pattern PATTERN_REGEX_NUMBER_ALPHABET_WORD = Pattern.compile(REGEX_NUMBER_ALPHABET_WORD);

    /**
     * 正则：正数、负数、小数、带科学计数法的浮点数
     */
    public static final String REGEX_POSITIVE_NEGATIVE_DECIMAL_NUMBER = "^(\\-|\\+)?\\d+(\\.\\d+)?((E|e)(\\-)?\\d+)?$";
    public static final Pattern PATTERN_REGEX_POSITIVE_NEGATIVE_DECIMAL_NUMBER = Pattern.compile(REGEX_POSITIVE_NEGATIVE_DECIMAL_NUMBER);

    /**
     * 正则：提取智能助理中提问模板的着重字
     */
    public static final String REGEX_AI_ASSISTANT_QUESTION_TEMPLATE = "\\「.*?\\」";
    public static final Pattern PATTERN_REGEX_AI_ASSISTANT_TEMPLATE = Pattern.compile(REGEX_AI_ASSISTANT_QUESTION_TEMPLATE);


    /**
     * 判断是否匹配正则
     *
     * @param regex 正则表达式字符串
     * @param input 要匹配的字符串
     * @return {@code true}: 匹配<br>{@code false}: 不匹配
     */
    public static boolean isMatch(String regex, CharSequence input) {
        return input != null && input.length() > 0 && Pattern.matches(regex, input);
    }

    /**
     * 判断是否匹配正则（性能佳）
     *
     * @param pattern 正则表达式Pattern
     * @param input   要匹配的字符串
     * @return {@code true}: 匹配<br>{@code false}: 不匹配
     */
    public static boolean isMatch(Pattern pattern, CharSequence input) {
        return input != null && input.length() > 0 && pattern.matcher(input).matches();
    }

    /**
     * 获取正则匹配的部分
     *
     * @param regex 正则表达式
     * @param input 要匹配的字符串
     * @return 正则匹配的部分
     */
    public static List<String> getMatches(String regex, CharSequence input) {
        if (input == null) {
            return null;
        }
        List<String> matches = new ArrayList<>();
        Pattern pattern = Pattern.compile(regex);
        Matcher matcher = pattern.matcher(input);
        while (matcher.find()) {
            matches.add(matcher.group());
        }
        return matches;
    }
}
