package com.digiwin.mobile.mobileuibot.push.jpush.util;

import cn.jiguang.common.resp.APIConnectionException;
import cn.jiguang.common.resp.APIRequestException;
import cn.jpush.api.JPushClient;
import cn.jpush.api.push.PushResult;
import cn.jpush.api.push.model.*;
import cn.jpush.api.push.model.audience.Audience;
import cn.jpush.api.push.model.audience.AudienceTarget;
import cn.jpush.api.push.model.notification.AndroidNotification;
import cn.jpush.api.push.model.notification.IosNotification;
import cn.jpush.api.push.model.notification.Notification;
import com.digiwin.mobile.mobileuibot.push.jpush.constant.PushConstant;
import com.google.gson.JsonObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.StringUtils;

import java.util.HashMap;
import java.util.Map;

/**
 * @author wuyangz
 * @date 2022/10/10
 */
public class PushUtil {
    protected static final Logger LOG = LoggerFactory.getLogger(PushUtil.class);


    /**
     * 快捷地构建推送对象：所有平台，所有设备，内容为 content 的通知。
     *
     * @return
     */
    public static PushPayload buildPushAlertAll(String content) {
        return PushPayload.alertAll(content);
    }

    /**
     * 快捷地构建推送对象 通知
     *
     * @param deviceType   平台
     * @param alias        别名
     * @param notification 正文
     * @return
     */
    public static PushPayload buildPushAlert(Integer deviceType, String alias, com.digiwin.mobile.mobileuibot.push.jpush.model.Notification notification, String appComponent) {
        return getBuilder(deviceType, alias, notification.getApnsProduction()).setNotification(getAlert(deviceType, notification, appComponent))
                .setOptions(getAndroidOption(notification)).build();
    }

    /**
     * 华为、小米、OPPO、VIVO的消息厂商通道配置处理<br/>
     * 参考资料：<br/>
     * 1.极光-厂商消息分类使用指南：https://docs.jiguang.cn/jpush/client/Android/android_channel_id
     *
     * @return
     */
    private static Options getAndroidOption(com.digiwin.mobile.mobileuibot.push.jpush.model.Notification notification) {
        Map<String, Map<String, String>> channelV2 = new HashMap<>(1);
        channelV2.put("huawei", getAndroidHuaweiOption(notification));
        channelV2.put("xiaomi", getAndroidXiaomiOption(notification));
        channelV2.put("oppo", getAndroidOppoOption(notification));
        channelV2.put("vivo", getAndroidVivoOption(notification));
        return Options.newBuilder()
                // 依消息分类使用指南设置为1
                .setClassification(1)
                .setThirdPartyChannel(channelV2).build();
    }

    /**
     * 返回华为的厂商通道推送配置
     *
     * @return
     */
    private static Map<String, String> getAndroidHuaweiOption(com.digiwin.mobile.mobileuibot.push.jpush.model.Notification notification) {
        // 设置消息分类
        Map<String, String> huaweiOption = new HashMap<>(2);
        huaweiOption.put("importance", "NORMAL");
        huaweiOption.put("category", "WORK");

        return huaweiOption;
    }

    /**
     * 返回小米的厂商通道推送配置
     *
     * @return
     */
    private static Map<String, String> getAndroidXiaomiOption(com.digiwin.mobile.mobileuibot.push.jpush.model.Notification notification) {
        // 设置消息分类
        Map<String, String> xiaomiOption = new HashMap<>(2);
        xiaomiOption.put("channel_id", "109491");
        xiaomiOption.put("skip_quota", "true");

        return xiaomiOption;
    }

    /**
     * 返回OPPO的厂商通道推送配置
     *
     * @return
     */
    private static Map<String, String> getAndroidOppoOption(com.digiwin.mobile.mobileuibot.push.jpush.model.Notification notification) {
        // 设置消息分类
        Map<String, String> oppoOption = new HashMap<>(2);
        // TODO OPPO平台的id申请中
        // OPPO的channel_id是自己填写的，所以所有app使用相同的id
        oppoOption.put("channel_id", "digiwin_mobile_notice");
        oppoOption.put("skip_quota", "true");

        return oppoOption;
    }

    /**
     * 返回VIVO的厂商通道推送配置
     *
     * @return
     */
    private static Map<String, String> getAndroidVivoOption(com.digiwin.mobile.mobileuibot.push.jpush.model.Notification notification) {
        // 设置消息分类
        Map<String, String> vivoOption = new HashMap<>(2);
        // VIVO不需要channel_id，只需要给分类
        vivoOption.put("category", "TODO");
        vivoOption.put("distribution", "secondary_push");

        return vivoOption;
    }

    /**
     * 快捷地构建推送对象 通知
     *
     * @param deviceType 平台
     * @param tag        标签
     * @param alias      别名
     * @param content    正文
     * @return
     */
    public static PushPayload buildPushAlert(Integer deviceType, String tag, String alias, String content) {
        PushPayload.Builder builder = PushPayload.newBuilder()
                .setPlatform(getPlatform(deviceType))
                .setNotification(Notification.alert(content));
//        if(PushConstant.DEVICE_TYPE_IOS.equals(deviceType) && EnvironmentUtil.pushProductionMode()){
//            builder.setOptions(Options.newBuilder().setApnsProduction(true).build());
//        }

        if (tag != null && alias != null) {
            builder.setAudience(Audience.newBuilder()
                    .addAudienceTarget(AudienceTarget.tag(tag))
                    .addAudienceTarget(AudienceTarget.alias(alias))
                    .build());
        } else if (tag != null) {
            builder.setAudience(Audience.tag(tag));
        } else if (alias != null) {
            builder.setAudience(Audience.alias(alias));
        }
        return builder.build();
    }

    /**
     * 快捷地构建推送对象：所有平台，所有设备，内容为 content 的自定义消息。
     *
     * @return
     */
    public static PushPayload buildPushMessage(String content) {
        return PushPayload.messageAll(content);
    }

    /**
     * 快捷地构建推送对象 自定义消息
     *
     * @param deviceType   平台
     * @param alias        别名
     * @param notification 消息
     * @return
     */
    public static PushPayload buildPushMessage(Integer deviceType, String alias, com.digiwin.mobile.mobileuibot.push.jpush.model.Notification notification) {
        return getBuilder(deviceType, alias, notification.getApnsProduction()).setMessage(getMessage(notification)).build();
    }

    /**
     * 根据消息 组装 极光的自定义类型消息
     *
     * @param notification
     * @return
     */
    public static Message getMessage(com.digiwin.mobile.mobileuibot.push.jpush.model.Notification notification) {
        Message.Builder builder = Message.newBuilder();
        String title = notification.getTitle();

        if (notification.getNotificationContent() != null) {
            builder.setTitle(title);
            builder.setMsgContent(notification.getNotificationContent());
        } else {
            builder.setMsgContent(title);
        }
        builder.addExtra("tID", notification.getTenantId());
        builder.addExtra("uID", notification.getUserId());
        builder.addExtra("sType", notification.getNotificationType());
        builder.addExtra("nID", notification.getNotificationID());
        if (!StringUtils.isEmpty(notification.getNotificationType())) {
            builder.addExtra("notificationType", notification.getNotificationType());
        }
        if (!StringUtils.isEmpty(notification.getTitle())) {
            builder.addExtra("notificationTitle", notification.getTitle());
        }
        if (!StringUtils.isEmpty(notification.getTenantId())) {
            builder.addExtra("tenantId", notification.getTenantId());
        }
        return builder.build();
    }

    /**
     * 根据消息 组装 极光的自定义类型消息
     *
     * @param deviceType
     * @param notification
     * @return
     */
    public static Notification getAlert(Integer deviceType, com.digiwin.mobile.mobileuibot.push.jpush.model.Notification notification, String appComponent) {
        if (PushConstant.DEVICE_TYPE_IOS.equals(deviceType)) {
            return getIosAlert(notification);
        }
        return getAndroidAlert(notification, appComponent);
    }

    /**
     * 根据消息 组装 极光的自定义类型消息
     *
     * @param notification
     * @return
     */
    public static Notification getAndroidAlert(com.digiwin.mobile.mobileuibot.push.jpush.model.Notification notification, String appComponent) {
        AndroidNotification.Builder builder = AndroidNotification.newBuilder();

        String title = notification.getTitle();

        if (notification.getNotificationContent() != null) {
            builder.setTitle(title);
            builder.setAlert(notification.getNotificationContent());
        } else {
            builder.setAlert(title);
        }
        JsonObject intent = new JsonObject();
        intent.addProperty("url", "intent:#Intent;component=" + appComponent + "/com.dajia.view.login.ui.SplashActivity;end");
        builder.addCustom("badge_add_num", 1);
        builder.addCustom("badge_class", "com.dajia.view.login.ui.SplashActivity");
        builder.addCustom("uri_activity", "com.dajia.view.other.component.push.OpenClickActivity");
        builder.setIntent(intent);
        builder.addExtra("tID", notification.getTenantId());
        builder.addExtra("uID", notification.getUserId());
        builder.addExtra("sType", notification.getNotificationType());
        builder.addExtra("nID", notification.getNotificationID());
        if (!StringUtils.isEmpty(notification.getNotificationType())) {
            builder.addExtra("notificationType", notification.getNotificationType());
        }
        if (!StringUtils.isEmpty(notification.getTitle())) {
            builder.addExtra("notificationTitle", notification.getTitle());
        }
        if (!StringUtils.isEmpty(notification.getTenantId())) {
            builder.addExtra("tenantId", notification.getTenantId());
        }

        return Notification.newBuilder().addPlatformNotification(builder.build()).build();
    }

    /**
     * 根据消息 组装 极光的自定义类型消息
     *
     * @param notification
     * @return
     */
    public static Notification getIosAlert(com.digiwin.mobile.mobileuibot.push.jpush.model.Notification notification) {
        IosNotification.Builder builder = IosNotification.newBuilder().autoBadge().setSound("default");
        String title = notification.getTitle();

        if (notification.getNotificationContent() != null) {
            builder.setAlert(title + "\n" + notification.getNotificationContent());
        } else {
            builder.setAlert(title);
        }
        builder.addExtra("tID", notification.getTenantId());
        builder.addExtra("uID", notification.getUserId());
        builder.addExtra("sType", notification.getNotificationType());
        builder.addExtra("nID", notification.getNotificationID());
        if (!StringUtils.isEmpty(notification.getNotificationType())) {
            builder.addExtra("notificationType", notification.getNotificationType());
        }
        if (!StringUtils.isEmpty(notification.getTitle())) {
            builder.addExtra("notificationTitle", notification.getTitle());
        }
        if (!StringUtils.isEmpty(notification.getTenantId())) {
            builder.addExtra("tenantId", notification.getTenantId());
        }

        return Notification.newBuilder().addPlatformNotification(builder.build()).build();
    }

    /**
     * 构建推送对象：所有平台，推送目标是别名为 "alias1"，通知内容为 ALERT。
     *
     * @return
     */
    public static PushPayload buildPushObject_all_alias_alert(String alias, String content) {
        return PushPayload.newBuilder()
                .setPlatform(Platform.all())
                .setAudience(Audience.alias(alias))
                .setNotification(Notification.alert(content))
                .build();
    }

    /**
     * 构建推送对象：平台是 Android，目标是 tag 为 "tag1" 的设备，内容是 Android 通知 ALERT，并且标题为 TITLE。
     *
     * @return
     */
    public static PushPayload buildPushObject_android_tag_alertWithTitle(String platform, String tag, String title, String content) {
        return PushPayload.newBuilder()
                .setPlatform(Platform.android())
                .setAudience(Audience.tag(tag))
                .setNotification(Notification.android(content, title, null))
                .build();
    }

    /**
     * 构建推送对象：平台是 iOS，推送目标是 "tag1", "tag_all" 的交集，推送内容同时包括通知与消息 - 通知信息是 ALERT，角标数字为 5，通知声音为 "happy"，并且附加字段 from = "JPush"；消息内容是 MSG_CONTENT。通知是 APNs 推送通道的，消息是 JPush 应用内消息通道的。APNs 的推送环境是“生产”（如果不显式设置的话，Library 会默认指定为开发）
     *
     * @return
     */
    public static PushPayload buildPushObject_ios_tagAnd_alertWithExtrasAndMessage() {
        return PushPayload.newBuilder()
                .setPlatform(Platform.ios())
                .setAudience(Audience.tag_and("tag1", "tag_all"))
                .setNotification(Notification.newBuilder()
                        .addPlatformNotification(IosNotification.newBuilder()
                                .setAlert("ALERT")
                                .setBadge(5)
                                .setSound("happy")
                                .addExtra("from", "JPush")
                                .build())
                        .build())
                .setMessage(Message.content("MSG_CONTENT"))
                .setOptions(Options.newBuilder()
                        .setApnsProduction(true)
                        .build())
                .build();
    }

    /**
     * 构建推送对象：平台是 Andorid 与 iOS，推送目标是 （"tag1" 与 "tag2" 的并集）交（"alias1" 与 "alias2" 的并集），推送内容是 - 内容为 MSG_CONTENT 的消息，并且附加字段 from = JPush。
     *
     * @return
     */
    public static PushPayload buildPushObject_ios_audienceMore_messageWithExtras() {
        return PushPayload.newBuilder()
                .setPlatform(Platform.android_ios())
                .setAudience(Audience.newBuilder()
                        .addAudienceTarget(AudienceTarget.tag("tag1", "tag2"))
                        .addAudienceTarget(AudienceTarget.alias("alias1", "alias2"))
                        .build())
                .setMessage(Message.newBuilder()
                        .setMsgContent("MSG_CONTENT")
                        .addExtra("from", "JPush")
                        .build())
                .build();
    }


    /**
     * 构建推送对象：推送内容包含SMS信息
     */
    public static void testSendWithSMS(String masterSecret, String appKey) {
        JPushClient jpushClient = new JPushClient(masterSecret, appKey);
        try {
            SMS sms = SMS.content("Test SMS", 10);
            PushResult result = jpushClient.sendAndroidMessageWithAlias("Test SMS", "test sms", sms, "alias1");
            LOG.debug("Got result - " + result);
        } catch (APIConnectionException e) {
            LOG.error("Connection error. Should retry later. ", e);
        } catch (APIRequestException e) {
            LOG.error("Error response from JPush server. Should review and fix it. ", e);
            LOG.debug("HTTP Status: " + e.getStatus());
            LOG.debug("Error Code: " + e.getErrorCode());
            LOG.debug("Error Message: " + e.getErrorMessage());
        }
    }

    /**
     * 快捷地构建Builder对象
     *
     * @param deviceType     平台
     * @param alias          别名
     * @param apnsProduction
     * @return
     * @Update 2020/12/10 mowj feature-整个方法重写。针对极光iOS推送到apn，提供开发环境与生产环境的区分。
     * 区分条件是：envirionmentConfig.properties中的pushProductionMode参数。true表示生产环境，false表示开发环境
     */
    public static PushPayload.Builder getBuilder(Integer deviceType, String alias, Boolean apnsProduction) {
        PushPayload.Builder builder = PushPayload.newBuilder();
        if (apnsProduction == null) {
            apnsProduction = true;
        }
        if (PushConstant.DEVICE_TYPE_IOS.equals(deviceType)) {
            builder.setOptions(Options.newBuilder().setApnsProduction(apnsProduction).build());
        } else {
            // 不是iOS的推送，不需设置ApnProduction参数， add by mowj 2020/12/10
        }
        builder.setPlatform(getPlatform(deviceType)).setAudience(Audience.alias(alias));
        return builder;
    }

    /**
     * 根据驱动获取要发送的平台
     *
     * @param deviceType
     * @return
     */
    private static Platform getPlatform(Integer deviceType) {
        if (deviceType == null) {
            return Platform.all();
        } else if (PushConstant.DEVICE_TYPE_ANDROID.equals(deviceType)) {
            return Platform.android();
        } else if (PushConstant.DEVICE_TYPE_IOS.equals(deviceType)) {
            return Platform.ios();
        } else if (PushConstant.DEVICE_TYPE_WP.equals(deviceType)) {
            return Platform.winphone();
        } else if (PushConstant.DEVICE_TYPE_ANDROID_IOS.equals(deviceType)) {
            return Platform.android_ios();
        } else if (PushConstant.DEVICE_TYPE_ANDROID_WP.equals(deviceType)) {
            return Platform.android_winphone();
        } else if (PushConstant.DEVICE_TYPE_IOS_WP.equals(deviceType)) {
            return Platform.ios_winphone();
        } else {
            return Platform.all();
        }
    }

}