package com.digiwin.pressure.cs.server;

import com.digiwin.pressure.cs.core.BaseApiHandler;
import com.digiwin.pressure.cs.core.BinaryProtocol;
import com.digiwin.pressure.cs.core.PressureMessageProcessor;
import com.digiwin.pressure.cs.core.SocketTemplate;
import com.digiwin.pressure.cs.core.domain.Message;
import com.digiwin.pressure.cs.core.domain.RegisterReq;
import com.digiwin.pressure.cs.core.domain.Result;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.smartboot.socket.transport.AioQuickServer;
import org.smartboot.socket.transport.AioSession;

import java.io.IOException;
import java.util.Collections;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import static com.digiwin.pressure.cs.core.domain.Constants.REIGSTER_API_ID;

@Slf4j
public class SocketServer {
    private final Integer port;
    private final BaseRegister register;
    private final Map<Long, AioSession> sessionMap = new ConcurrentHashMap<>();
    private String blackIp;
    private SocketTemplate socketTemplate;

    public SocketServer(int port, BaseRegister register) {
        this.port = port;
        this.register = register;
    }

    public <T> Result<T> requestClient(long slaverId, int apiId, Object requestData, int timeoutSeconds) throws Exception {
        AioSession session = sessionMap.get(slaverId);
        return socketTemplate.request(session, apiId, requestData, timeoutSeconds);
    }

    @SneakyThrows
    public void start() throws IOException {
        BaseApiHandler<RegisterReq, Void> registerHandler = new BaseApiHandler<RegisterReq, Void>() {
            @Override
            public int getApiId() {
                return REIGSTER_API_ID;
            }

            @Override
            public Result<Void> handle(AioSession session, RegisterReq req) {
                try {
                    req.setRemoteIp(session.getRemoteAddress().getHostString());
                } catch (IOException e) {
                    log.error("Error getting remote ip", e);
                }
                Result<Long> result = register.register(req);
                if (result.isFail()) {
                    return Result.buildFailResult(result.getErrorCode(), result.getErrorMsg());
                }
                Long slaverId = result.getData();
                sessionMap.put(slaverId, session);
                return Result.buildSuccessResult(null);
            }
        };
        this.socketTemplate = new SocketTemplate(Collections.singletonList(registerHandler));

        AioQuickServer<Message> server = new AioQuickServer<>(this.port, new BinaryProtocol(), new PressureMessageProcessor(socketTemplate) {
            @Override
            protected void handleSessionClosedEvent(AioSession session) {
                Long slaverId = sessionMap.entrySet().stream()
                        .filter(entry -> entry.getValue().getSessionID().equals(session.getSessionID()))
                        .map(Map.Entry::getKey).findFirst().orElse(null);
                if (slaverId != null) {
                    sessionMap.remove(slaverId);
                    register.disconnect(slaverId);
                }
            }

            @Override
            protected String getBlackIp() {
                return blackIp;
            }

            @Override
            protected void handleSessionErrorEvent(AioSession session) {
                session.close();
            }
        });
        server.start();
       log.info("Server started on port {}", this.port);
    }

    public void setBlackIp(String blackIp) {
        this.blackIp = blackIp;
    }
}

