package com.jugg.agile.biz.digiwin.context.ask.language;

import com.jugg.agile.framework.core.context.JaMapContextLink;
import com.jugg.agile.framework.core.context.biz.JaCoreContextEntity;
import com.jugg.agile.framework.core.util.JaStringUtil;
import com.jugg.agile.spring.boot.util.JaI18nUtil;
import org.springframework.context.i18n.LocaleContextHolder;

import java.util.Locale;
import java.util.Map;

/**
 * AcceptLanguageContextLink 优先处理head
 * <p>
 * 后续 {@link LangContextLink} 处理param
 *
 * @author chenjian
 * @since 2024年11月01日 17:34:51
 */
public class AcceptLanguageContextLink implements JaMapContextLink<JaCoreContextEntity> {

    public static final String headerLocaleUp = "Locale";
    public static final String headerLocale = "locale";

    @Override
    public void inherit(Map<String, String> source, JaCoreContextEntity context) {
        String language = getLanguage(source);
        if (JaStringUtil.isNotEmpty(language)) {
            Locale locale = JaI18nUtil.parseLocales(language);
            JaI18nUtil.LocaleThreadLocal.set(locale);
            LocaleContextHolder.setLocale(locale);
        }

    }

    @Override
    public void transmit(JaCoreContextEntity source, Map<String, String> target) {
        Locale locale = JaI18nUtil.LocaleThreadLocal.get();
        if (null != locale) {
            // zh-CN
            transmit(target, JaI18nUtil.Header, locale.toLanguageTag());

            // 容错业务 zh_CN
            transmit(target, headerLocale, locale.toString());
            transmit(target, LangContextLink.headerLang, locale.toString());
        }

    }

    @Override
    public void remove() {
        JaI18nUtil.LocaleThreadLocal.remove();
        LocaleContextHolder.resetLocaleContext();
    }

    private String getLanguage(Map<String, String> source) {
        String acceptLanguage = source.get(JaI18nUtil.Header);
        if (JaStringUtil.isEmpty(acceptLanguage)) {
            acceptLanguage = source.get(JaI18nUtil.HeaderUp);
        }
        if (JaStringUtil.isNotEmpty(acceptLanguage)) {
            return acceptLanguage;
        }

        String locale = source.get(headerLocale);
        if (JaStringUtil.isEmpty(locale)) {
            locale = source.get(headerLocaleUp);
        }
        if (JaStringUtil.isNotEmpty(locale)) {
            return locale.replace('_', '-');
        }
        return null;
    }
}
