package com.jugg.agile.middleware.db.jdbc.config;

import lombok.Getter;
import lombok.Setter;

@Setter
@Getter
public class JdbcPropertyConfig {
    /**
     * 客户端缓存
     */
    private boolean cachePrepStmts = true;
    private int prepStmtCacheSize = 250;
    private int prepStmtCacheSqlLimit = 2048;
    /**
     * 服务端预处理缓存, 建议单独设置
     */
    private boolean useServerPrepStmts = false;
    /**
     * `useLocalSessionState`是一个配置选项，通常在分布式或集群环境中使用。当设置为`true`时，表示使用本地会话状态，即将会话状态数据存储在当前服务器节点上，而不是通过网络在多个服务器节点之间共享。
     * 两者之间的主要区别如下：
     * 1. 当`useLocalSessionState`为`true`时，由于会话状态数据存储在本地服务器节点上，因此在处理客户端请求时，服务器节点不需要在不同节点之间传输会话状态数据。这样可以减少网络通信开销，从而提高性能。
     * 2. 当`useLocalSessionState`为`false`时，表示会话状态数据需要在多个服务器节点之间共享。这可能导致更大的网络通信开销和额外的处理用时，因为节点之间需要同步会话状态数据。在这种情况下，性能可能受到影响。
     * 然而，值得注意的是，将`useLocalSessionState`设置为`true`会带来一些潜在的问题：
     * 1. 当某个服务器节点崩溃或发生故障时，存储在该节点上的会话状态数据可能会丢失。而当`useLocalSessionState`为`false`时，由于会话状态数据在节点之间共享，因此在某个节点故障时，其他节点仍然可以访问这些数据。
     * 2. 当`useLocalSessionState`为`true`时，负载均衡策略应仅限于会话保持，这意味着在会话期间，所有来自特定客户端的请求都应当被路由到同一个服务器节点。这可能限制了负载均衡算法的选择。
     * 总之，`useLocalSessionState`的设置取决于您的应用程序需求和部署环境。如果性能是重要因素，且可以接受潜在的会话状态数据丢失风险，那么设置为`true`可能是一个合适的选择。否则，设置为`false`将允许在多个服务器节点之间共享会话状态数据，带来更好的可用性和弹性，但可能损失一些性能。
     */
    private boolean useLocalSessionState = true;
    /**
     * `rewriteBatchedStatements`是一个MySQL JDBC驱动的配置参数，用于控制是否对批量操作进行优化。它的取值可以是`true`或`false`。
     * 当`rewriteBatchedStatements`设置为`true`时，JDBC驱动将尽可能地将多个独立的插入、更新或删除语句重写为单个批量操作。这种重写方式可以显著提高批量操作的执行性能，因为它减少了数据库服务器与客户端之间的往返次数。数据库服务器只需要处理一个合并后的批量操作，而不是逐个处理每个独立的操作。
     * 当`rewriteBatchedStatements`设置为`false`时（这是默认值），JDBC驱动不会对批量操作进行优化，每个插入、更新或删除语句都会独立执行。这种情况下，即使使用`addBatch`和`executeBatch`方法进行批量操作，性能也不会得到显著提升。
     * 需要注意的是，使用`rewriteBatchedStatements`参数时，有以下几点须知：
     * 1. 仅适用于MySQL 5.1.13及更高版本的JDBC驱动。
     * 2. 该选项仅对传统的插入、更新和删除语句有效。对于预处理语句（`PreparedStatement`）和其他SQL语句，如`SELECT`、`CREATE`等，此选项无效。
     * 3. 启用此选项可能改变批量操作的行为，比如插入重复主键时，如果JDBC驱动将多个插入语句合并成一个批量插入，数据库可能会拒绝整个批量操作，而不仅仅是其中的某个插入语句。
     * 因此，在使用`rewriteBatchedStatements`时，需要根据应用程序的需求和特性来决定是否启用它。如果有大量需要优化的批量操作，可以通过启用此选项来获得性能提升。但在启用此选项之前，请确保了解它可能对应用程序行为产生的影响。
     */
    private boolean rewriteBatchedStatements = true;
    /**
     * `cacheResultSetMetadata`是一个MySQL JDBC驱动的配置参数，用于控制是否缓存结果集元数据。它的取值可以是`true`或`false`。
     * 结果集元数据（`ResultSetMetaData`）包括了一系列关于结果集中列的信息，如列名、数据类型、大小等。在执行查询操作（如`SELECT`语句）时，返回的结果集可能包含大量的列信息。对于某些应用程序，可能需要频繁访问这些结果集元数据，以检索和处理列的相关信息。
     * 当`cacheResultSetMetadata`设置为`true`时，JDBC驱动将在内存中缓存结果集元数据。这意味着当应用程序在执行相同的查询操作时，驱动可以直接从缓存中获取结果集元数据，而不需要每次都从数据库服务器检索元数据。这样可以减少网络通信开销，提高查询性能。
     * 当`cacheResultSetMetadata`设置为`false`时（这是默认值），JDBC驱动将不会缓存结果集元数据。这意味着每次执行查询操作时，都需要从数据库服务器检索结果集元数据，可能导致额外的通信开销。
     * 需要注意的是，启用`cacheResultSetMetadata`可能会带来一些潜在问题：
     * 1. 额外的内存消耗：缓存大量的结果集元数据可能会占用较多的客户端内存。在有限的内存资源下，这可能会影响应用程序的性能。
     * 2. 缓存一致性：如果数据库表结构发生变化，如修改列名或数据类型，缓存的结果集元数据可能变得不再准确。在这种情况下，需要清除缓存或禁用元数据缓存以确保正确性。
     * 在实际应用中，可以根据实际需要和场景来决定是否启用`cacheResultSetMetadata`。如果应用程序中有大量频繁使用的查询操作，且对查询性能有较高要求，可以考虑将此选项设置为`true`。在其他情况下，可以保留默认值`false`，以确保获取结果集元数据的实时性。
     */
    private boolean cacheResultSetMetadata = false;
    /**
     * `cacheServerConfiguration`是一个MySQL JDBC驱动的配置参数，用于控制是否缓存来自MySQL服务器的特定配置信息。它的取值可以是`true`或`false`。
     * MySQL服务器的配置信息可能包含了诸多有关服务器设置的详细信息，如字符集、系统变量、全局状态等。当JDBC驱动连接到MySQL服务器时，它可能会检索这些配置信息以便在客户端正确处理查询和结果集。
     * 当`cacheServerConfiguration`设置为`true`时，JDBC驱动会缓存从MySQL服务器检索的配置信息。这意味着对于后续的连接和请求，驱动可以直接从缓存中获取所需的服务器配置信息，而不需要每次都从服务器检索。这样可以减轻网络通信开销，提高连接和查询性能。
     * 当`cacheServerConfiguration`设置为`false`时（这是默认值），JDBC驱动将不会缓存MySQL服务器的配置信息。在这种情况下，每次执行操作或建立连接时都可能需要从服务器检索配置信息，可能会导致额外的通信开销。
     * 需要注意的是，在某些情况下，缓存服务器配置可能会导致问题：
     * 1. 配置信息变更：如果MySQL服务器配置发生更改（如修改系统变量或字符集设置），缓存的配置信息可能变得不再准确。这种情况可能导致客户端基于过时的配置处理查询和结果集。
     * 2. 缓存失效策略：对于缓存服务器配置信息的驱动，需要注意清理缓存策略，以防止缓存失效或过期。
     * 在实际应用中，可以根据应用场景和需求来决定是否启用`cacheServerConfiguration`。如果应用程序需要频繁建立连接和执行查询操作，且服务器配置信息相对稳定，可以考虑启用缓存以提高性能。反之，如果服务器配置信息可能经常发生变化或需要实时检索配置信息，可以保留默认值`false`，以确保客户端始终基于最新的配置信息。
     */
    private boolean cacheServerConfiguration = false;
    /**
     * elideSetAutoCommits`是一个MySQL JDBC驱动的配置参数，用于控制是否省略对`setAutoCommit()`不必要的调用。它的取值可以是`true`或`false`。
     * `setAutoCommit()`方法是用于控制数据库连接的自动提交模式。当`autoCommit`设置为`true`时，每个SQL语句执行后都会自动提交更改（DML操作）；设置为`false`时，需要使用`commit()`方法来显式提交更改。
     * 当`elideSetAutoCommits`设置为`true`时，JDBC驱动会跟踪每个连接的自动提交模式，并仅在`autoCommit`值与当前连接不同时调用`setAutoCommit()`方法。这可以减少JDBC驱动给MySQL服务器发送不必要的`setAutoCommit()`请求的次数，提高数据库操作性能。
     * 当`elideSetAutoCommits`设置为`false`时（这是默认值），JDBC驱动将不会尝试优化`setAutoCommit()`方法的调用，每次调用`setAutoCommit()`都会将请求发送给MySQL服务器。这可能导致额外的通信开销，降低性能。
     * 在实际应用中，根据应用场景和需求来决定是否启用`elideSetAutoCommits`。如果发现应用程序中有大量不必要的`setAutoCommit()`方法调用，可以考虑启用此选项以提高性能。在其他情况下，可以保留默认值`false`。需要注意的是，设置此选项对于实际操纵数据库事务的逻辑并不产生任何改变，仅仅是在自动提交模式调用的性能优化。
     */
    private boolean elideSetAutoCommits = false;
    /**
     * `maintainTimeStats`是一个MySQL JDBC驱动的配置参数，用于控制驱动程序是否维护和跟踪与执行查询和获取连接相关的时间统计信息。`maintainTimeStats`的取值可以是`true`或`false`。
     * - 当`maintainTimeStats`设置为`true`时（这是默认值），JDBC驱动程序将跟踪和维护与执行查询、获取/关闭连接以及其他数据库操作相关的时间统计信息。这些信息可以用于诊断、性能分析和调试应用程序。另外，当启用此选项时，JDBC驱动还会提供PROFILE SQL（性能分析SQL）功能，可以使用`getProfilerEventHandler()`方法获取更多关于SQL执行和性能的详细信息。
     * - 当`maintainTimeStats`设置为`false`时，JDBC驱动程序将不会进行任何时间统计。这将减少驱动程序在执行请求时的计算开销和内存消耗。在这种情况下，禁用时间统计可能为提高应用程序性能提供一定帮助。
     * 需要注意的是，在实际应用中，启用或禁用`maintainTimeStats`的影响可能不会很明显。在大多数情况下，时间统计可能只占用很小的计算资源，因此保留默认值`true`通常是合适的。然而，在高度关注性能的场景中，可以考虑将`maintainTimeStats`设置为`false`，以节省一些计算和内存资源。在不需要详细的时间统计或性能分析时，可以禁用此选项。
     */
    private boolean maintainTimeStats = false;
}
