package com.jugg.agile.middleware.db.jdbc.config;

import lombok.Getter;
import lombok.Setter;

import java.nio.charset.StandardCharsets;
import java.time.ZoneId;

/**
 * https://dev.mysql.com/doc/connector-j/en/connector-j-reference-configuration-properties.html
 *
 * @author chenjian
 * @since 2024年03月28日 17:08:26
 */
@Setter
@Getter
public class JdbcUrlConfig {
    /**
     * 决定是否启用Unicode字符集支持。
     * 当值为true时，JDBC驱动会理解并处理Unicode字符。
     */
    private boolean useUnicode = true;
    /**
     * 允许在一个单独的执行（execute、executeQuery或executeUpdate）中执行多个语句（使用分号;分隔）。
     * 当值为true时，可以一次执行多个SQL语句，这可以提高批量操作的性能。
     */
    private boolean allowMultiQueries = true;
    /**
     * 决定当数据库连接意外丢失时，JDBC驱动是否自动尝试重新连接。
     * 当值为true时，驱动程序将在连接丢失时自动尝试重新连接到数据库。
     */
    private boolean autoReconnect = true;
    /**
     * 决定是否使用SSL（安全套接字层）进行加密通信。当值为true时，JDBC驱动将使用SSL连接到MySQL服务器。
     * 这有助于提高数据在客户端和服务器之间传输的安全性。
     */
    private boolean useSSL = false;
    /**
     * 决定如何在Java中表示MySQL的TINYINT(1)数据类型。
     * 当值为true时，JDBC驱动将TINYINT(1)映射为Java中的BIT数据类型（进一步处理为boolean类型）。
     * 当值为false时，将TINYINT(1)映射为Java中的TINYINT数据类型（处理为byte或short类型）。
     */
    private boolean tinyInt1isBit = false;
    /**
     * 决定当使用故障切换（从主服务器切换到备用服务器）时，备用服务器是否自动处于只读模式。
     * 当值为false时，在发生故障切换时，备用服务器不会自动设置为只读模式。
     */
    private boolean failOverReadOnly = false;
    /**
     * 设置用于与MySQL服务器通信的时区。值应为有效的Java时区ID。
     */
    private String serverTimezone = ZoneId.systemDefault().getId();
    /**
     * 决定如何将MySQL中的0000-00-00和0000-00-00 00:00:00值映射到Java中。可选值有：
     * EXCEPTION：将这些值视为无效，如果遇到此类值，将抛出异常。
     * convertToNull：将这些值映射为NULL。
     * ROUND：将日期值0000-00-00四舍五入到Java的最近日期，通常是0001-01-01。
     */
    private String zeroDateTimeBehavior = "convertToNull";
    /**
     * 设置客户端与MySQL服务器之间通信时使用的字符集(encoding)，应为有效的Java字符集名称（如UTF-8、ISO-8859-1等）。
     */
    private String characterEncoding = StandardCharsets.UTF_8.name();
}
