package com.jugg.agile.middleware.kafka.test;

import com.alibaba.fastjson.JSON;
import org.apache.kafka.clients.producer.KafkaProducer;
import org.apache.kafka.clients.producer.Producer;
import org.apache.kafka.clients.producer.ProducerConfig;
import org.apache.kafka.clients.producer.ProducerRecord;
import org.apache.kafka.common.serialization.StringSerializer;

import java.util.Properties;

public class LogKafkaProducer {

    private Producer<String, String> producer;
    private String topicName;

    public LogKafkaProducer(String topicName) {
        this.topicName = topicName;
        this.producer = createProducer();
    }

    private Producer<String, String> createProducer() {
        Properties props = new Properties();

        // ========== 必需的基础配置 ==========
        // Kafka集群地址
        props.put(ProducerConfig.BOOTSTRAP_SERVERS_CONFIG,
                "10.176.187.18:9092,10.176.187.17:9092,10.176.187.16:9092");

        // 序列化器配置
        props.put(ProducerConfig.KEY_SERIALIZER_CLASS_CONFIG, StringSerializer.class.getName());
        props.put(ProducerConfig.VALUE_SERIALIZER_CLASS_CONFIG, StringSerializer.class.getName());

        // ========== 低开销配置（业务优先） ==========
        // 确认机制：0表示不等待确认，无网络开销
        props.put(ProducerConfig.ACKS_CONFIG, "0");

        // 重试次数：0次重试，避免额外CPU和网络开销
        props.put(ProducerConfig.RETRIES_CONFIG, 0);

        // 请求超时：快速超时，避免线程阻塞
        props.put(ProducerConfig.REQUEST_TIMEOUT_MS_CONFIG, 3000);

        // 传输超时：快速失败
        props.put(ProducerConfig.DELIVERY_TIMEOUT_MS_CONFIG, 5000);

        // ========== 内存优化配置 ==========
        // 批处理大小：32KB，平衡内存使用和批处理效率（约3-4条日志）
        props.put(ProducerConfig.BATCH_SIZE_CONFIG, 32768);

        // 等待时间：50ms，让批次有充分时间填满，减少发送频率
        props.put(ProducerConfig.LINGER_MS_CONFIG, 50);

        // 缓冲区内存：16MB，较小的内存占用
        props.put(ProducerConfig.BUFFER_MEMORY_CONFIG, 16777216);

        // 压缩类型：none，避免CPU压缩开销
        props.put(ProducerConfig.COMPRESSION_TYPE_CONFIG, "none");

        // ========== CPU优化配置 ==========
        // 关闭幂等性：减少CPU开销
        props.put(ProducerConfig.ENABLE_IDEMPOTENCE_CONFIG, false);

        // 飞行请求数：较小值，减少内存和网络开销
        props.put(ProducerConfig.MAX_IN_FLIGHT_REQUESTS_PER_CONNECTION, 2);

        // ========== 网络低开销配置 ==========
        // 发送缓冲区大小：64KB，较小的缓冲区减少内存占用
        props.put(ProducerConfig.SEND_BUFFER_CONFIG, 65536);

        // 接收缓冲区大小：32KB
        props.put(ProducerConfig.RECEIVE_BUFFER_CONFIG, 32768);

        // 连接空闲超时：2分钟，适中的连接管理
        props.put(ProducerConfig.CONNECTIONS_MAX_IDLE_MS_CONFIG, 120000);

        // ========== 阻塞和元数据优化 ==========
        // 元数据刷新间隔：5分钟，减少元数据请求频率
        props.put(ProducerConfig.METADATA_MAX_AGE_CONFIG, 300000);

        // 阻塞超时：100ms小超时，避免长时间阻塞业务线程
        props.put(ProducerConfig.MAX_BLOCK_MS_CONFIG, 100);

        // ========== 监控配置 ==========
        // 客户端ID：用于监控和日志识别
        props.put(ProducerConfig.CLIENT_ID_CONFIG, "log-producer-" + System.currentTimeMillis());

        // JMX监控：启用JMX指标
//        props.put(ProducerConfig.AUTO_INCLUDE_JMX_REPORTER_CONFIG, true);

        // ========== 分区策略配置 ==========
        // 分区器：使用默认的粘性分区器，提高批处理效率
        props.put(ProducerConfig.PARTITIONER_CLASS_CONFIG,
                "org.apache.kafka.clients.producer.internals.DefaultPartitioner");

        return new KafkaProducer<>(props);
    }

    /**
     * 低开销模式：最小化对业务的影响
     * @param logJson 日志JSON字符串
     * @param traceId 链路ID
     */
    public void sendLogSilentMode(String logJson, String traceId) {
        try {
            ProducerRecord<String, String> record = new ProducerRecord<>(topicName, traceId, logJson);
            // 纯异步发送，不要回调，最小开销
            producer.send(record);
        } catch (Exception e) {
            // 静默处理异常，不影响业务
            // 可选：记录到本地错误计数器
        }
    }

    /**
     * 带简单错误统计的发送（可选）
     */
    private volatile long errorCount = 0;
    private volatile long lastErrorLogTime = 0;

    public void sendLogWithErrorCount(String logJson, String traceId) {
        try {
            ProducerRecord<String, String> record = new ProducerRecord<>(topicName, traceId, logJson);
            producer.send(record, (metadata, exception) -> {
                if (exception != null) {
                    errorCount++;
                    long now = System.currentTimeMillis();
                    // 每10秒最多打印一次错误日志，避免刷屏
                    if (now - lastErrorLogTime > 10000) {
                        System.err.println("Log send errors in last period: " + errorCount);
                        lastErrorLogTime = now;
                        errorCount = 0;
                    }
                }
            });
        } catch (Exception e) {
            // 静默处理
        }
    }

    /**
     * 获取错误统计（用于监控）
     */
    public long getErrorCount() {
        return errorCount;
    }

    /**
     * 发送日志消息（保留回调版本，用于调试）
     * @param logJson 日志JSON字符串
     * @param traceId 链路ID作为key，保证同一链路的日志在同一分区
     */
    public void sendLog(String logJson, String traceId) {
        // 使用traceId作为key，确保同一链路的日志发送到同一分区
        ProducerRecord<String, String> record = new ProducerRecord<>(
                topicName,
                traceId,  // key: 使用traceId保证同链路消息有序
                logJson   // value: 日志JSON
        );

        // 异步发送，轻量级回调
        producer.send(record, (metadata, exception) -> {
            if (exception != null) {
                // 仅记录错误，不做重试
                System.err.println("Log send failed: " + exception.getMessage());
            }
            // 成功时什么都不做，减少开销
        });
    }

    /**
     * 优雅关闭：快速释放资源，不等太久
     */
    public void flush() {
        try {
            producer.flush();
        } catch (Exception e) {
            // 静默处理，避免关闭时异常
        }
    }

    /**
     * 快速关闭Producer，设置较短的超时时间
     */
    public void close() {
        if (producer != null) {
            try {
                // 最多等待5秒，然后强制关闭
                producer.close(java.time.Duration.ofSeconds(5));
            } catch (Exception e) {
                // 静默处理关闭异常
            }
        }
    }

    // 使用示例
    public static void main(String[] args) {
        LogKafkaProducer logProducer = new LogKafkaProducer("log-topic");

        // 模拟日志对象
        LogMessage logMessage = new LogMessage();
        logMessage.setTime(System.currentTimeMillis());
        logMessage.setThread("main");
        logMessage.setTraceId("trace-123456");
        logMessage.setSpanId("span-001");
        logMessage.setLevel("INFO");
        logMessage.setLoggerName("com.example.LogService");
        logMessage.setNodeType("application");
        logMessage.setType("business");
        logMessage.setNodeId("node-001");
        logMessage.setCurrentNodeSpanId("current-span-001");
        logMessage.setTimeConsume(150L);
        logMessage.setParamSize(1024);
        logMessage.setCustomContent("用户登录");
        logMessage.setMessage("用户登录成功");
        logMessage.setThrowable(null);
        logMessage.setAppId("my-app");
        logMessage.setSource("hostname-001");
        logMessage.setAppender("kafka");
        logMessage.setModifyDate(System.currentTimeMillis());
        logMessage.setCreateDate(System.currentTimeMillis());

        // 转换为JSON并发送
        String logJson = JSON.toJSONString(logMessage);
        logProducer.sendLog(logJson, logMessage.getTraceId());

        // 确保发送完成
        logProducer.flush();

        // 关闭资源
        logProducer.close();



        long errors = logProducer.getErrorCount();
        if (errors > 100) {
            // 发送告警，但不影响业务
//            alertService.sendAlert("日志发送错误率过高: " + errors);
        }
        // 在业务方法中调用
            try {
                // 主业务逻辑
//                doBusinessWork();
            } finally {
                // 日志发送不能影响业务
//                logProducer.sendLogSilentMode(createLogJson(), getTraceId());
        }
    }
}

// 日志消息实体类
class LogMessage {
    private Long time;
    private String thread;
    private String traceId;
    private String spanId;
    private String level;
    private String loggerName;
    private String nodeType;
    private String type;
    private String nodeId;
    private String currentNodeSpanId;
    private Long timeConsume;
    private Integer paramSize;
    private String customContent;
    private String message;
    private String throwable;
    private String appId;
    private String source;
    private String appender;
    private Long modifyDate;
    private Long createDate;

    // Getters and Setters
    public Long getTime() { return time; }
    public void setTime(Long time) { this.time = time; }

    public String getThread() { return thread; }
    public void setThread(String thread) { this.thread = thread; }

    public String getTraceId() { return traceId; }
    public void setTraceId(String traceId) { this.traceId = traceId; }

    public String getSpanId() { return spanId; }
    public void setSpanId(String spanId) { this.spanId = spanId; }

    public String getLevel() { return level; }
    public void setLevel(String level) { this.level = level; }

    public String getLoggerName() { return loggerName; }
    public void setLoggerName(String loggerName) { this.loggerName = loggerName; }

    public String getNodeType() { return nodeType; }
    public void setNodeType(String nodeType) { this.nodeType = nodeType; }

    public String getType() { return type; }
    public void setType(String type) { this.type = type; }

    public String getNodeId() { return nodeId; }
    public void setNodeId(String nodeId) { this.nodeId = nodeId; }

    public String getCurrentNodeSpanId() { return currentNodeSpanId; }
    public void setCurrentNodeSpanId(String currentNodeSpanId) { this.currentNodeSpanId = currentNodeSpanId; }

    public Long getTimeConsume() { return timeConsume; }
    public void setTimeConsume(Long timeConsume) { this.timeConsume = timeConsume; }

    public Integer getParamSize() { return paramSize; }
    public void setParamSize(Integer paramSize) { this.paramSize = paramSize; }

    public String getCustomContent() { return customContent; }
    public void setCustomContent(String customContent) { this.customContent = customContent; }

    public String getMessage() { return message; }
    public void setMessage(String message) { this.message = message; }

    public String getThrowable() { return throwable; }
    public void setThrowable(String throwable) { this.throwable = throwable; }

    public String getAppId() { return appId; }
    public void setAppId(String appId) { this.appId = appId; }

    public String getSource() { return source; }
    public void setSource(String source) { this.source = source; }

    public String getAppender() { return appender; }
    public void setAppender(String appender) { this.appender = appender; }

    public Long getModifyDate() { return modifyDate; }
    public void setModifyDate(Long modifyDate) { this.modifyDate = modifyDate; }

    public Long getCreateDate() { return createDate; }
    public void setCreateDate(Long createDate) { this.createDate = createDate; }

}