package com.jugg.agile.middleware.kafka.test;

import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.apache.kafka.clients.consumer.ConsumerRecords;
import org.apache.kafka.clients.consumer.KafkaConsumer;
import org.apache.kafka.common.errors.WakeupException;

import java.time.Duration;
import java.util.Collections;
import java.util.Properties;
import java.util.concurrent.atomic.AtomicBoolean;

public class TestKafkaConsumerLocal {
    private final AtomicBoolean running = new AtomicBoolean(true);
    private KafkaConsumer<String, String> consumer;

    public static void main(String[] args) {
        TestKafkaConsumerLocal continuousConsumer = new TestKafkaConsumerLocal();
        continuousConsumer.start();

        // 添加关闭钩子，优雅关闭消费者
        Runtime.getRuntime().addShutdownHook(new Thread(continuousConsumer::shutdown));
    }

    public void start() {
        // 配置消费者
        Properties props = new Properties();
        props.put("bootstrap.servers", "localhost:9092");
        props.put("group.id", "test-consumer-group");
        props.put("key.deserializer", "org.apache.kafka.common.serialization.StringDeserializer");
        props.put("value.deserializer", "org.apache.kafka.common.serialization.StringDeserializer");
        props.put("enable.auto.commit", "true"); // 自动提交偏移量
        props.put("auto.commit.interval.ms", "1000"); // 自动提交间隔
        props.put("auto.offset.reset", "earliest"); // 如果没有偏移量，从最早的消息开始

        // 创建消费者实例
        consumer = new KafkaConsumer<>(props);

        // 订阅主题
        consumer.subscribe(Collections.singletonList("test-topic"));

        System.out.println("开始持续监听消息，按Ctrl+C退出...");

        try {
            while (running.get()) {
                // 轮询获取消息，设置超时时间为100毫秒
                ConsumerRecords<String, String> records = consumer.poll(Duration.ofMillis(1000));

                for (ConsumerRecord<String, String> record : records) {
                    System.out.printf("收到消息: 主题=%s, 分区=%d, 偏移量=%d, 键=%s, 值=%s%n",
                            record.topic(),
                            record.partition(),
                            record.offset(),
                            record.key(),
                            record.value());

                    // 在这里添加您的业务逻辑处理
                    processMessage(record.value());
                }
            }
        } catch (WakeupException e) {
            // 忽略异常，这是正常关闭的一部分
            if (!running.get()) {
                System.out.println("消费者正在关闭...");
            }
        } finally {
            // 关闭消费者
            consumer.close();
            System.out.println("消费者已关闭");
        }
    }

    private void processMessage(String message) {
        // 这里添加您的消息处理逻辑
        // 例如：解析JSON、存储到数据库、调用其他服务等
        System.out.println("处理消息: " + message);
    }

    public void shutdown() {
        // 设置运行标志为false
        running.set(false);
        // 唤醒消费者，使其从poll()调用中返回
        if (consumer != null) {
            consumer.wakeup();
        }
    }
}
