package com.jugg.agile.middleware.nacos.config;

import com.alibaba.nacos.api.NacosFactory;
import com.alibaba.nacos.api.PropertyKeyConst;
import com.alibaba.nacos.api.config.ConfigService;
import com.jugg.agile.framework.core.config.JaSpringProperty;
import com.jugg.agile.framework.core.config.processor.JaYamlProcessor;
import com.jugg.agile.framework.core.dapper.log.JaLog;
import com.jugg.agile.framework.core.util.JaShutdownHookUtil;
import com.jugg.agile.framework.core.util.JaStringUtil;
import com.jugg.agile.framework.core.util.JaThrowableUtil;
import com.jugg.agile.framework.core.util.JaValidateUtil;
import com.jugg.agile.framework.core.util.io.file.JaPropertiesUtil;
import com.jugg.agile.middleware.nacos.config.meta.JaNacosConfigEntity;
import com.jugg.agile.middleware.nacos.config.meta.JaNacosConfigListener;
import lombok.SneakyThrows;

import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;

/**
 * 读取nacos配置
 *
 * @author chenjian
 * @since 2020年07月13日 12:05:52
 */
public class JaNacosConfig {

    private static Properties properties;

    /**
     * 读取nacos配置
     *
     * @author chenjian
     * @since 2020年07月29日 12:06:32
     */
    @SneakyThrows
    public static String read(JaNacosConfigEntity entity) {
        String dataId = entity.getDataId();
        JaLog.info("read nacos dataId : {}", dataId);
        JaValidateUtil.notNull(dataId, () -> "read nacos dataId is null");

        Properties properties = entity.getProperties();
        if (null == properties) {
            properties = new Properties();
            properties.put(PropertyKeyConst.SERVER_ADDR, entity.getServerAddr());
            properties.put(PropertyKeyConst.NAMESPACE, entity.getNamespace());
            properties.put(PropertyKeyConst.USERNAME, entity.getUsername());
            properties.put(PropertyKeyConst.PASSWORD, entity.getPassword());
        }


        ConfigService configService = NacosFactory.createConfigService(properties);
        String content = configService.getConfig(dataId, entity.getGroup(), entity.getTimeoutMs());
        if (null != entity.getListener()) {
            configService.addListener(dataId, entity.getGroup(), entity.getListener());
        }
        JaLog.info("nacos config : \n{}", content);
        if (entity.isReadStop()) {
            configService.shutDown();
        } else {
            JaShutdownHookUtil.add(dataId, () -> JaThrowableUtil.tryCatchRun(configService::shutDown));
        }
        return content;

    }

    public static Map<String, Object> readYaml(JaNacosConfigEntity entity) {
        String content = read(entity);
        return JaStringUtil.isEmpty(content) ? new HashMap<>(0) : JaYamlProcessor.getFlattenedMap(content);
    }

    @SneakyThrows
    public static Map<String, Object> readYaml(String dataId, JaNacosConfigListener listener) {
        try {
            return JaNacosConfig.readYaml(JaNacosConfigEntity.builder()
                    .dataId(dataId)
                    .listener(listener)
                    .properties(JaNacosConfig.getProperties())
                    .build());
        } catch (Throwable e) {
            JaLog.get().error("read nacos [{}] config failed", dataId, e);
            return new HashMap<>(0);
        }
    }

    @SneakyThrows
    public static Map<String, Object> readYaml(String dataId) {
        try {
            return JaNacosConfig.readYaml(JaNacosConfigEntity.builder()
                    .dataId(dataId)
                    .properties(JaNacosConfig.getProperties())
                    .readStop(true)
                    .build());
        } catch (Throwable e) {
            JaLog.get().error("read nacos [{}] config failed", dataId, e);
            return new HashMap<>(0);
        }
    }

    public static Properties getProperties() {
        if (null == properties) {
            // 获取nacos环境，决定使用哪个nacos注册中心和配置中心
            // 优先获取环境变量 nacos_env指定的环境，如apac-prod,prod等
            // 如果没有配置nacos_env，则使用env环境变量使用的环境
            String nacosEnv = System.getProperty("nacos_env", JaSpringProperty.getEnv());
            if (JaStringUtil.isNotEmpty(nacosEnv)) {
                String nacosPath = String.format("nacos/%s.properties", nacosEnv.toLowerCase());
                properties = JaPropertiesUtil.read(nacosPath);
            }

        }
        return properties;
    }

    public static void setProperties(Properties properties) {
        JaNacosConfig.properties = properties;
    }

    public static void main(String[] args) throws IOException {
        JaSpringProperty.setEnv("paas");
        Map<String, Object> read = JaNacosConfig.readYaml(JaNacosConfigEntity.builder()
                .serverAddr("https://nacos-paas.digiwincloud.com.cn")
                .username("athena-nonfunc")
                .password("athena")
                .namespace("f1271d47-fe9c-4938-abfa-2dbf2cf3c144")
                .dataId("jugg-dapper.yaml")
                .build());
        System.out.println(read);
    }
}
