package com.jugg.agile.middleware.nacos.config;

import com.alibaba.nacos.api.NacosFactory;
import com.alibaba.nacos.api.PropertyKeyConst;
import com.alibaba.nacos.api.config.ConfigService;
import com.jugg.agile.framework.core.config.JaEnvProperty;
import com.jugg.agile.framework.core.config.JaProperty;
import com.jugg.agile.framework.core.config.processor.JaYamlProcessor;
import com.jugg.agile.framework.core.dapper.log.JaDapperLog;
import com.jugg.agile.framework.core.dapper.log.JaLog;
import com.jugg.agile.framework.core.util.JaShutdownHookUtil;
import com.jugg.agile.framework.core.util.JaStringUtil;
import com.jugg.agile.framework.core.util.JaThrowableUtil;
import com.jugg.agile.framework.core.util.JaValidateUtil;
import com.jugg.agile.framework.core.util.datastructure.JaCollectionUtil;
import com.jugg.agile.framework.core.util.reflect.clazz.JaClassUtil;
import com.jugg.agile.middleware.nacos.config.listenter.JaNacosConfigListener;
import com.jugg.agile.middleware.nacos.config.meta.JaNacosConfigEntity;
import com.jugg.agile.middleware.nacos.config.meta.JaNacosConfigServerEntity;
import lombok.SneakyThrows;

import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;

/**
 * 读取nacos配置
 *
 * @author chenjian
 * @since 2020年07月13日 12:05:52
 */
public class JaNacosConfigService {

    static {
        // com.alibaba.nacos.client.logging.AbstractNacosLogging.getLocation
        if (JaClassUtil.hashClass("ch.qos.logback.classic.LoggerContext") && JaDapperLog.getUseLogback()) {
            System.setProperty("nacos.logging.config", "classpath:jugg-nacos-logback.xml");
        }
    }

    /**
     * 读取nacos配置
     *
     * @author chenjian
     * @since 2020年07月29日 12:06:32
     */
    @SneakyThrows
    public static String read(JaNacosConfigEntity entity, Properties properties) {
        if (!JaNacosConfigPropertiesProcessor.isUser()) {
            return null;
        }
        String dataId = entity.getDataId();
        JaValidateUtil.notNull(dataId, () -> "read nacos dataId is null");
        String group = entity.getGroup();
        JaValidateUtil.notNull(group, () -> "read nacos group is null");
        String namespace = entity.getNamespace();
        JaValidateUtil.notNull(namespace, () -> "read nacos namespace is null");
        JaValidateUtil.notNull(properties, () -> "read nacos properties is null");
        properties.put(PropertyKeyConst.NAMESPACE, namespace);

        ConfigService configService = NacosFactory.createConfigService(properties);
        String content = configService.getConfig(dataId, entity.getGroup(), entity.getTimeoutMs());
        if (null != entity.getListener()) {
            configService.addListener(dataId, entity.getGroup(), entity.getListener());
            JaShutdownHookUtil.add(dataId, () -> JaThrowableUtil.tryCatchRun(configService::shutDown));
        } else {
            configService.shutDown();
        }
        JaLog.info("{} {} nacos config : \n{}", dataId, group, content);
        return content;
    }

    public static String read(JaNacosConfigEntity entity, JaNacosConfigServerEntity serverEntity) {
        return read(entity, JaNacosConfigPropertiesProcessor.getServerAddrProperties(serverEntity));
    }

    public static Map<String, Object> readYaml(JaNacosConfigEntity entity, Properties properties) {
        String content = read(entity, properties);
        return JaStringUtil.isEmpty(content) ? new HashMap<>(0) : JaYamlProcessor.getFlattenedMap(content);
    }

    public static Map<String, Object> readYaml(JaNacosConfigEntity entity, JaNacosConfigServerEntity serverEntity) {
        return readYaml(entity, JaNacosConfigPropertiesProcessor.getServerAddrProperties(serverEntity));
    }

    public static Map<String, Object> readYaml(JaNacosConfigEntity entity) {
        Properties serverAddrProperties;
        try {
            serverAddrProperties = JaNacosConfigPropertiesProcessor.getServerAddrProperties();
        } catch (NullPointerException e) {
            JaLog.warn("{} nacos readYaml error, serverAddrProperties is null", entity.getDataId());
            return null;
        }
        return readYaml(entity, serverAddrProperties);
    }

    public static Map<String, Object> readYaml(String dataId, String namespace, JaNacosConfigListener listener) {
        return JaNacosConfigService.readYaml(JaNacosConfigEntity.builder()
                .dataId(dataId)
                .namespace(namespace)
                .group(JaEnvProperty.getVersion())
                .listener(listener)
                .build());
    }

    public static Map<String, Object> readYaml(String dataId, String namespace) {
        return readYaml(dataId, namespace, null);
    }

    public static void loadYaml(String dataId, String namespace) {
        Map<String, Object> nacosMap = readYaml(dataId, namespace, null);
        if (JaCollectionUtil.isNotEmpty(nacosMap)) {
            JaProperty.getPropertyMap().putAll(nacosMap);
        }
    }


    public static void main(String[] args) throws IOException {
        JaNacosConfigListener jaNacosConfigListener = new JaNacosConfigListener() {

            @Override
            public void receiveConfigInfo(String configInfo) {
                System.out.println("configInfo : " + configInfo);
            }
        };
        Map<String, Object> read = JaNacosConfigService.readYaml(JaNacosConfigEntity.builder()
                        .namespace("8231cc09-273b-43db-8561-6a82f21f15e0")
                        .dataId("rabbitmq.yml")
                        .group("2.2.21")
                        .listener(jaNacosConfigListener)
                        .build()
                , JaNacosConfigServerEntity.builder()
                        .serverAddr("http://121.37.176.231:8848")
                        .username("")
                        .password("")
                        .build()
        );
        System.out.println(read);
        System.in.read();
    }
}
