package com.jugg.agile.middleware.rabbitmq.spring;

import com.jugg.agile.framework.core.context.JaMapContextChain;
import com.jugg.agile.framework.core.dapper.aspect.JaDapperAspectPointcut;
import com.jugg.agile.framework.core.dapper.aspect.JaNodeSpanResolver;
import com.jugg.agile.framework.core.dapper.log.JaLog;
import com.jugg.agile.framework.core.dapper.log.JaMDC;
import com.jugg.agile.framework.core.dapper.meta.NodeKind;
import com.jugg.agile.framework.core.dapper.meta.NodeSpan;
import com.jugg.agile.framework.core.util.bytecode.aop.JaAopUtil;
import com.jugg.agile.framework.core.util.datastructure.JaCollectionUtil;
import com.rabbitmq.client.Channel;
import org.aopalliance.intercept.MethodInvocation;
import org.springframework.amqp.core.Message;
import org.springframework.amqp.core.MessagePostProcessor;
import org.springframework.amqp.rabbit.connection.CorrelationData;
import org.springframework.amqp.rabbit.core.RabbitTemplate;
import org.springframework.aop.framework.ReflectiveMethodInvocation;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.stereotype.Component;

import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

@Component
@ConditionalOnClass(name = {JaNodeSpanRabbitTemplateProducer.ProxyClassName})
public class JaNodeSpanRabbitTemplateProducer implements JaNodeSpanResolver, JaDapperAspectPointcut {

    public static final String ProxyClassName = "org.springframework.amqp.rabbit.core.RabbitTemplate";

    private static final Set<String> sendMethodNameSet = new HashSet<String>() {{
        /*
         * 设置一个 SpEL 表达式,用于在发送消息时选择合适的 ConnectionFactory。
         * 通过评估 SpEL 表达式的结果,可以在运行时动态地决定使用哪个 ConnectionFactory 来发送消息。
         * 这在使用 RabbitMQ 集群且需要根据消息内容、目的地等因素选择特定节点发送消息时用。
         */
//        add("setSendConnectionFactorySelectorExpression");
        /*
         * 发送消息到 RabbitMQ,同时等待并接收回复消息。
         * 该方法发送消息后会阻塞等待,直到收到回复消息或超时。
         * 适用于请求-响应模式的消息交互,发送方需要根据回复消息进行后续处理。
         */
        add("sendAndReceive");
        /*
         * 将 Java 对象转换为 AMQP 消息后发送,并等待接收回复消息,最后将回复消息转换为指定的 Java 类型。
         * 该方法结合了消息的发送、接收和类型转换,简化了请求-响应模式下的消息处理。
         * 发送方可以直接使用域对象进行消息发送,并获得转换后的回复对象,无需手动进行类型转换。
         */
        add("convertSendAndReceive");
        /*
         * 执行实际的消息发送操作,将消息发送到 RabbitMQ。
         * 该方法是 RabbitTemplate 中消息发送的核心方法,其他发送方法最终都会调用 doSend。
         * doSend 方法负责创建 Channel,设置消息属性,并将消息发送到指定的交换机和路由键。
         */
        add("doSend");
        /*
         * 将 Java 对象转换为 AMQP 消息后发送,并设置关联 ID,用于匹配回复消息
         * 该方法适用于请求-响应模式,发送方通过关联 ID 将请求消息和回复消息关联起来。
         * 接收方可以根据关联 ID 将回复消息发送给正确的请求方。
         */
        add("correlationConvertAndSend");
        /*
         * 发送 AMQP 消息到 RabbitMQ。
         * 该方法接收 Message 对象作为参数,直接发送预构建的 AMQP 消息。
         * 适用于需要完全控制消息属性和内容的场景。
         */
        add("send");
        /*
         * 将 Java 对象转换为 AMQP 消息后发送。
         * 该方法接收 Java 对象作为参数,自动将其转换为 AMQP 消息后发送。
         * 适用于发送方只关注消息内容,而不需要手动构建 AMQP 消息的场景。
         */
        add("convertAndSend");
        /*
         * 将 Java 对象转换为 AMQP 消息后发送,并等待接收指定类型的回复消息。
         * 该方法与 convertSendAndReceive 类似,但允许指定期望的回复消息类型。
         * 适用于请求-响应模式下,发送方需要根据回复消息的具体类型进行不同处理的场景。
         */
        add("convertSendAndReceiveAsType");
    }};

    public static final NodeKind RabbitTemplateProduceNodeKind = new NodeKind("rabbitmq-produce")
            .buildIdHandler(JaAopUtil::getSimpleName)
            .buildResponseHandler(o -> o instanceof Message ? null : o)
            .buildReqArgsHandler(args -> {
                if (JaCollectionUtil.isEmpty(args)) {
                    return args;
                }
                ArrayList<Object> objects = new ArrayList<>(args.length);
                for (Object arg : args) {
                    if (arg instanceof Channel
                            || arg instanceof CorrelationData
                            || arg instanceof Type
                            || arg instanceof ParameterizedTypeReference
                    ) {
                        continue;
                    }
                    if (arg instanceof Message) {
                        objects.add(((Message) arg).getBody());
                    } else {
                        objects.add(arg);
                    }
                }
                return objects.toArray();
            })
            .buildRequestIdThreadLocal();

    @Override
    public NodeSpan getNodeSpan(MethodInvocation invocation) {
        if (invocation instanceof ReflectiveMethodInvocation) {
            Object proxy = ((ReflectiveMethodInvocation) invocation).getProxy();
            if (proxy.toString().startsWith(ProxyClassName)
                    && sendMethodNameSet.contains(invocation.getMethod().getName())
            ) {
                return get(RabbitTemplateProduceNodeKind);
            }
        }
        return null;
    }

    @Override
    public String getExpression() {
        return "execution(* org.springframework.amqp.rabbit.core.RabbitTemplate.*(..))";
    }

//    @Override
//    public Message postProcessMessage(Message message) throws AmqpException {
//        try {
//            Map<String, Object> headers = message.getMessageProperties().getHeaders();
//            JaMapContextChain.getInstance().transmit().forEach(headers::put);
//            headers.put(JaMDC.RequestId, rabbitTemplateNodeKind.getRequestIdThreadLocal().get());
//            return message;
//        } finally {
//            rabbitTemplateNodeKind.getRequestIdThreadLocal().remove();
//        }
//    }


    public static void transmitContext(List<RabbitTemplate> beanList) {
        if (JaCollectionUtil.isEmpty(beanList)) {
            return;
        }
        try {
            MessagePostProcessor messagePostProcessor = message -> {
                try {
                    Map<String, Object> headers = message.getMessageProperties().getHeaders();
                    JaMapContextChain.getInstance().transmit().forEach(headers::put);
                    headers.put(JaMDC.RequestId, JaNodeSpanRabbitTemplateProducer.RabbitTemplateProduceNodeKind.getRequestIdThreadLocal().get());
                    return message;
                } finally {
                    JaNodeSpanRabbitTemplateProducer.RabbitTemplateProduceNodeKind.getRequestIdThreadLocal().remove();
                }
            };
            for (RabbitTemplate rabbitTemplate : beanList) {
                rabbitTemplate.addBeforePublishPostProcessors(messagePostProcessor);
            }
        } catch (Throwable e) {
            JaLog.error("rabbitTemplate.addBeforePublishPostProcessors error", e);
        }
    }
}
