package com.jugg.agile.middleware.rabbitmq.spring;

import com.jugg.agile.framework.core.context.JaMapContextChain;
import com.jugg.agile.framework.core.context.biz.JaCoreContext;
import com.jugg.agile.framework.core.dapper.JaDapper;
import com.jugg.agile.framework.core.dapper.log.JaLog;
import com.jugg.agile.framework.core.dapper.meta.NodeKind;
import com.jugg.agile.framework.core.dapper.meta.NodeSpan;
import com.jugg.agile.framework.core.util.bytecode.aop.JaAspectUtil;
import com.jugg.agile.framework.core.util.concurrent.JaThreadLocal;
import com.jugg.agile.framework.core.util.datastructure.JaCollectionUtil;
import com.jugg.agile.framework.core.util.datastructure.JaMapUtil;
import com.jugg.agile.framework.core.util.io.serialize.json.JaJson;
import com.jugg.agile.framework.core.util.reflect.JaReflectUtil;
import com.jugg.agile.framework.core.util.unsafe.JaUnsafe;
import com.jugg.agile.spring.util.JaSpringAopUtil;
import com.jugg.agile.spring.util.JaSpringBeanUtil;
import com.rabbitmq.client.Channel;
import org.aopalliance.intercept.MethodInterceptor;
import org.aopalliance.intercept.MethodInvocation;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Pointcut;
import org.springframework.amqp.core.Message;
import org.springframework.amqp.core.MessageProperties;
import org.springframework.amqp.rabbit.listener.AbstractMessageListenerContainer;
import org.springframework.amqp.rabbit.listener.MessageListenerContainer;
import org.springframework.amqp.rabbit.listener.RabbitListenerEndpointRegistry;
import org.springframework.amqp.rabbit.listener.SimpleMessageListenerContainer;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.event.ContextRefreshedEvent;
import org.springframework.core.Ordered;
import org.springframework.core.annotation.Order;
import org.springframework.lang.Nullable;

import java.lang.reflect.Field;
import java.util.Map;

@Order(10)
@Configuration
@ConditionalOnClass(name = {"org.springframework.amqp.rabbit.annotation.EnableRabbit"})
public class JaNodeSpanRabbitListenerConsumer
//        implements ApplicationListener<ContextRefreshedEvent>
{

    public static final JaThreadLocal<Message> MessageThreadLocal = new JaThreadLocal<>();
    public static final JaThreadLocal<Channel> ChannelThreadLocal = new JaThreadLocal<>();

    //    @Override
    public void onApplicationEvent(@Nullable ContextRefreshedEvent event) {
        RabbitListenerEndpointRegistry bean = JaSpringBeanUtil.getBean(RabbitListenerEndpointRegistry.class);
        if (null == bean) {
            return;
        }
        Map<String, MessageListenerContainer> listenerContainers = JaReflectUtil.getFieldValue(bean, "listenerContainers");
        if (JaCollectionUtil.isEmpty(listenerContainers)) {
            return;
        }
        JaMessagingMessageListenerAdapterInterceptor methodInterceptor = new JaMessagingMessageListenerAdapterInterceptor();
        listenerContainers.forEach((k, v) -> {
            if (v instanceof SimpleMessageListenerContainer) {
                Object messageListener = ((SimpleMessageListenerContainer) v).getMessageListener();
                Object messageListenerProxy = JaSpringAopUtil.getProxy(messageListener, methodInterceptor);
                Field messageListenerField = JaReflectUtil.getField(AbstractMessageListenerContainer.class, "messageListener");
                JaUnsafe.setObjectFieldValue(v, messageListenerProxy, messageListenerField);
            }
        });
    }

    @Bean
    public JaRabbitListenerAspect jaRabbitListenerAspect() {
        return new JaRabbitListenerAspect();
    }

    /**
     * @author chenjian
     * @see MessagingMessageListenerAdapter#onMessage(org.springframework.amqp.core.Message, com.rabbitmq.client.Channel)
     * @since 2024年06月03日 20:36:53
     */
    static class JaMessagingMessageListenerAdapterInterceptor implements MethodInterceptor {

        private static final String onMessageMethodName = "onMessage";

        @Override
        public Object invoke(MethodInvocation invocation) throws Throwable {
            if (invocation.getMethod().getName().equals(onMessageMethodName)) {
                try {
                    MessageThreadLocal.set((Message) invocation.getArguments()[0]);
                    ChannelThreadLocal.set((Channel) invocation.getArguments()[1]);
                    return invocation.proceed();
                } finally {
                    MessageThreadLocal.remove();
                    ChannelThreadLocal.remove();
                }
            } else {
                return invocation.proceed();
            }
        }
    }

    @Aspect
    @Order(Ordered.HIGHEST_PRECEDENCE + 100)
    static class JaRabbitListenerAspect {
        private static final NodeKind rabbitMQConsume = new NodeKind("rabbitmq-consume", true);
        private static final String logFormatEntry = "queue:{},context:{}";

        public JaRabbitListenerAspect() {
            JaMapContextChain instance = JaMapContextChain.getInstance();
            JaLog.info("JaRabbitListenerAspect init:{}", instance);
        }


        @Pointcut("@annotation(org.springframework.amqp.rabbit.annotation.RabbitListener)")
        public void pointcut() {
        }

        @Around("pointcut()")
        public Object aroundAdvice(ProceedingJoinPoint pjp) throws Throwable {
            Message message = MessageThreadLocal.get();
            if (null == message) {
                JaLog.warn("rabbitmq consumer not fount dapper node:{}", JaAspectUtil.getFullName(pjp));
                return pjp.proceed();
            }
            try {
                MessageProperties messageProperties = message.getMessageProperties();
                JaMapContextChain.getInstance().inherit(JaMapUtil.convertToString(messageProperties.getHeaders()));
                NodeSpan nodeSpan = NodeSpan.builder()
                        .id(JaAspectUtil.getSimpleName(pjp))
                        .nodeKind(rabbitMQConsume)
                        .build();
                JaLog.info(logFormatEntry, messageProperties.getConsumerQueue(), JaJson.toString(JaCoreContext.getInstance().get()));
                return JaDapper.dapperAspect(nodeSpan, new Object[]{message.getBody()}, pjp);
            } finally {
                JaMapContextChain.getInstance().remove();
            }
        }
    }
}
