package com.jugg.agile.middleware.redis.adapter.spring.command;

import com.jugg.agile.framework.core.util.datastructure.JaCollectionUtil;
import com.jugg.agile.middleware.redis.JaRedisCommand;
import com.jugg.agile.middleware.redis.adapter.spring.JaRedisTemplateProcessor;
import com.jugg.agile.middleware.redis.meta.JaRedisConstants;
import lombok.SneakyThrows;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.core.script.DefaultRedisScript;
import org.springframework.data.redis.core.script.RedisScript;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.TimeUnit;

@SuppressWarnings({"unchecked", "rawtypes"})
public class JaRedisTemplateCommand implements JaRedisCommand {

    private final RedisTemplate redisTemplate;

    public JaRedisTemplateCommand() {
        redisTemplate = JaRedisTemplateProcessor.getRedisTemplate();
    }

    public JaRedisTemplateCommand(RedisTemplate redisTemplate) {
        this.redisTemplate = redisTemplate;
    }

    @Override
    public Boolean set(String key, String value, Long... seconds) {
        if (JaCollectionUtil.isEmpty(seconds)) {
            redisTemplate.opsForValue().set(key, value);
        } else {
            redisTemplate.opsForValue().set(key, value, seconds[0], TimeUnit.SECONDS);
        }
        return true;
    }

    private static final RedisScript<Long> scriptSetNx = new DefaultRedisScript<>(JaRedisConstants.LuaSetNxScript, Long.class);

    @Override
    public Boolean setNx(String key, String value, Long... seconds) {
        if (JaCollectionUtil.isEmpty(seconds)) {
            return redisTemplate.opsForValue().setIfAbsent(key, value);
        } else {
            return JaRedisConstants.LuaSuccess == redisTemplate.execute(scriptSetNx, Collections.singletonList(key), value, seconds[0].toString());
        }
    }

    @Override
    public Long del(String key) {
        return Boolean.TRUE.equals(redisTemplate.delete(key)) ? 1L : 0L;
    }

    @Override
    public Boolean exists(String key) {
        return Boolean.TRUE.equals(redisTemplate.hasKey(key));
    }

    @Override
    @SneakyThrows
    public String get(String key) {
        Object o = redisTemplate.opsForValue().get(key);
        return null == o ? null : o.toString();
    }

    @Override
    public Long expire(String key, Long seconds) {
        return Boolean.TRUE.equals(redisTemplate.expire(key, seconds, TimeUnit.SECONDS)) ? 1L : 0;
    }

    @Override
    public Long incr(String key, Long... increment) {
        if (JaCollectionUtil.isEmpty(increment)) {
            return redisTemplate.opsForValue().increment(key, 1);
        }
        return redisTemplate.opsForValue().increment(key, increment[0]);
    }

    @Override
    public Long decr(String key, Long... decrement) {
        if (JaCollectionUtil.isEmpty(decrement)) {
            return redisTemplate.opsForValue().increment(key, -1);
        }
        return redisTemplate.opsForValue().increment(key, -decrement[0]);
    }

    @Override
    public Object eval(String script, List<String> keys, Object... args) {
        return redisTemplate.execute(new DefaultRedisScript<>(script, Object.class), keys, args);
    }
}
