package com.jugg.agile.middleware.redis.spring;

import com.jugg.agile.framework.core.dapper.JaDapper;
import com.jugg.agile.framework.core.dapper.meta.NodeKind;
import com.jugg.agile.framework.core.dapper.meta.NodeSpan;
import com.jugg.agile.framework.core.util.bytecode.aop.JaAopUtil;
import com.jugg.agile.framework.core.util.unsafe.JaUnsafe;
import com.jugg.agile.spring.util.JaSpringAopUtil;
import lombok.SneakyThrows;
import org.aopalliance.intercept.MethodInterceptor;
import org.aopalliance.intercept.MethodInvocation;
import org.springframework.aop.framework.AopProxyUtils;
import org.springframework.beans.factory.config.BeanPostProcessor;
import org.springframework.context.annotation.Configuration;
import org.springframework.data.redis.core.RedisTemplate;

/**
 * 临时方案, RedisTemplate节点简易处理
 *
 * @author chenjian
 * @since 2024年04月24日 13:55:42
 */
@Configuration
public class JaNodeSpanRedisTemplate implements BeanPostProcessor {

    @Override
    @SuppressWarnings("rawtypes")
    public Object postProcessAfterInitialization(Object bean, String beanName) {
        if (bean.getClass().getName().startsWith("org.springframework.data.redis.core.RedisTemplate")
                || bean.getClass().getName().startsWith("org.springframework.data.redis.core.StringRedisTemplate")
        ) {

            JaRedisTemplateMethodInterceptor methodInterceptor = new JaRedisTemplateMethodInterceptor();
            if (bean instanceof RedisTemplate) {
                proxy(bean, ((RedisTemplate) getSingletonTarget(bean)).opsForValue(), methodInterceptor, "valueOps");
                proxy(bean, ((RedisTemplate) getSingletonTarget(bean)).opsForList(), methodInterceptor, "listOps");
                proxy(bean, ((RedisTemplate) getSingletonTarget(bean)).opsForZSet(), methodInterceptor, "zSetOps");
                proxy(bean, ((RedisTemplate) getSingletonTarget(bean)).opsForGeo(), methodInterceptor, "geoOps");
                proxy(bean, ((RedisTemplate) getSingletonTarget(bean)).opsForHyperLogLog(), methodInterceptor, "hllOps");
                proxy(bean, ((RedisTemplate) getSingletonTarget(bean)).opsForSet(), methodInterceptor, "setOps");
            }


        }
        return bean;
    }


    @SneakyThrows
    private void proxy(Object redisTemplate, Object opsObject, JaRedisTemplateMethodInterceptor methodInterceptor, String opsFieldName) {

        JaUnsafe.setObjectFieldValue(getSingletonTarget(redisTemplate), opsFieldName
                , JaSpringAopUtil.getProxy(opsObject
                        , methodInterceptor));
    }

    static class JaRedisTemplateMethodInterceptor implements MethodInterceptor {

        @Override
        public Object invoke(MethodInvocation invocation) throws Throwable {
            NodeSpan nodeSpan = NodeSpan.builder()
                    .id(JaAopUtil.getSimpleName(invocation))
                    .nodeKind(NodeKind.Constant.Redis)
                    .build();
            Object[] reqArgs = invocation.getArguments();
            return JaDapper.dapperAop(nodeSpan, reqArgs, invocation);
        }
    }

    private Object getSingletonTarget(Object redisTemplate) {
        Object singletonTarget = AopProxyUtils.getSingletonTarget(redisTemplate);
        return null == singletonTarget ? redisTemplate : singletonTarget;
    }
}