package com.jugg.agile.middleware.redis;

import com.jugg.agile.framework.core.config.JaProperty;
import com.jugg.agile.framework.core.config.JaPropertyListener;
import com.jugg.agile.framework.core.dapper.log.JaLog;
import com.jugg.agile.framework.core.util.JaStringUtil;
import com.jugg.agile.framework.core.util.JaValidateUtil;
import com.jugg.agile.framework.core.util.concurrent.JaThreadLocal;
import com.jugg.agile.middleware.redis.meta.JaRedisLockEntity;
import lombok.SneakyThrows;
import org.springframework.transaction.support.TransactionSynchronizationAdapter;
import org.springframework.transaction.support.TransactionSynchronizationManager;

import java.util.concurrent.Callable;
import java.util.concurrent.TimeUnit;

/**
 * redis 分布式锁
 *
 * @author chenjian
 * @since 2020年09月07日 10:00:27
 */
public class JaRedisLock {

    private static long maxSpinTime;
    private static final JaThreadLocal<String> reentrantRequestId = new JaThreadLocal<>();


    static {
        JaPropertyListener.addAndRunCommonListener(() -> {
            maxSpinTime = JaProperty.getLong("ja.redis.lock.maxSpinTime", 10000L);
        });
    }

    /**
     * 使用demo 看 DrpLockAspect
     *
     * @author chenjian
     * @since 2020年10月13日 12:04:29
     */
    @SneakyThrows
    public static <T> T lock(JaRedisLockEntity entity, Callable<T> callable) {
        // 判断是否可重入
        if (JaStringUtil.isNotEmpty(reentrantRequestId.get())) {
            JaLog.get().info("lock reentry");
            return callable.call();
        }
        String lockKey = entity.getLockKey();
        String apiName = entity.getApiName();
        if (JaStringUtil.isEmpty(lockKey)) {
            JaLog.get().info("lockKey is null,{}", apiName);
            return callable.call();
        }
        JaLog.get().info("lockKey:{}", lockKey);
        // TODO 应对误删, 其实防不住, 待考虑优化
        String requestId = lockKey + System.currentTimeMillis();
        long timeout = entity.getTimeout();
        long spinTime = entity.getSpinTime();
        boolean lock = JaRedis.setNx(lockKey, requestId, timeout);
        // 自旋
        if (!lock && spinTime > 0) {
            TimeUnit.MILLISECONDS.sleep(Math.min(spinTime, maxSpinTime));
            lock = JaRedis.setNx(lockKey, requestId, timeout);
        }
        JaValidateUtil.isTrue(!lock, entity::getMessage);
        reentrantRequestId.set(requestId);
        try {
            return callable.call();
        } finally {
            unlock(lockKey, requestId, entity.isTx());
        }
    }

    /**
     * 释放锁
     *
     * @author chenjian
     * @since 2020年08月11日 11:43:45
     */
    private static void unlock(String lockKey, String requestId, boolean tx) {
        if (TransactionSynchronizationManager.isSynchronizationActive() && tx) {
            try {
                TransactionSynchronizationManager.registerSynchronization(new TransactionSynchronizationAdapter() {
                    @Override
                    public void afterCompletion(int status) {
                        unlock(lockKey, requestId);
                    }
                });
            } catch (Throwable e) {
                JaLog.get().error("Transaction afterCompletion unLock lock failed", e);
                unlock(lockKey, requestId);
            }
        } else {
            unlock(lockKey, requestId);
        }
    }

    /**
     * TODO 暂用del
     * 是否要使用unlink
     * org.springframework.data.redis.core.RedisTemplate#unlink(java.lang.Object)
     */
    private static void unlock(String lockKey, String requestId) {
        try {
            if (JaRedis.del(lockKey) < 1) {
                JaLog.get().error("redis unlock fail:{}", lockKey);
            }
        } catch (Throwable e) {
            JaLog.get().error("redis unlock error:{}", lockKey);
        } finally {
            reentrantRequestId.remove();
        }
    }
}
