package com.jugg.agile.middleware.redis;

import com.jugg.agile.framework.core.config.JaProperty;
import com.jugg.agile.framework.core.dapper.log.JaDapperLog;
import com.jugg.agile.framework.core.dapper.log.JaLog;
import com.jugg.agile.framework.core.dapper.meta.NodeKind;
import com.jugg.agile.framework.core.dapper.meta.NodeSpan;
import com.jugg.agile.framework.core.dapper.meta.RspSpan;
import com.jugg.agile.framework.meta.function.JaFunctionP;
import com.jugg.agile.framework.core.util.JaStringUtil;
import com.jugg.agile.framework.core.util.JaThrowableUtil;
import com.jugg.agile.framework.core.util.bytecode.proxy.JaProxyCglib;
import com.jugg.agile.framework.core.util.concurrent.JaConcurrentSubmit;
import com.jugg.agile.framework.core.util.concurrent.JaConcurrentTasks;
import com.jugg.agile.framework.core.util.concurrent.JaThreadLocal;
import com.jugg.agile.framework.core.util.datastructure.JaCollectionUtil;
import com.jugg.agile.framework.core.util.io.serialize.json.JaJson;
import com.jugg.agile.framework.core.util.io.serialize.json.adapter.jackson.migration.JacksonAutoType;
import com.jugg.agile.middleware.redis.adapter.jedis.command.JaJedisClusterCommand;
import com.jugg.agile.middleware.redis.adapter.jedis.command.JaJedisSimpleCommand;
import com.jugg.agile.middleware.redis.adapter.spring.command.JaRedisTemplateCommand;
import com.jugg.agile.middleware.redis.config.JaRedisConfig;
import lombok.SneakyThrows;

import java.lang.reflect.Type;
import java.util.List;
import java.util.concurrent.Callable;

public class JaRedis {

    private JaRedis() {
    }

    private static final String set = "set";
    private static final JaRedisCommand redisCommand;

    public static final JaThreadLocal<Boolean> SkipSpanThreadLocal = new JaThreadLocal<>();

    static {
        try {
            Integer adapter = JaProperty.getInteger("ja.redis.type", 4);
            if (4 == adapter) {
                redisCommand = new JaRedisTemplateCommand();
            } else {
                Class<? extends JaRedisCommand> sourceClz = JaJedisSimpleCommand.class;
                switch (adapter) {
                    case 0:
                        sourceClz = JaJedisClusterCommand.class;
                        break;
                    case 3:
                        sourceClz = JaJedisSimpleCommand.class;
                        break;
                    default:
                        JaThrowableUtil.throwRunLog("ja.redis.adapter.base set error:{}", adapter);
                }
                redisCommand = JaProxyCglib.newInstance(sourceClz
                        , (proxyInstance, method, args, methodProxy, invokeHandler) -> {
                            String methodName = method.getName();
                            boolean skipSpan = null != SkipSpanThreadLocal.get() && SkipSpanThreadLocal.get();
                            RspSpan rspSpan = null;
                            if (!skipSpan) {
                                rspSpan = RspSpan.builder()
                                        .nodeSpan(NodeSpan.builder()
                                                .nodeKind(NodeKind.Constant.Redis)
                                                .nodeId(methodName + " " + args[0])// args[0] -> redis key
                                                .build())
                                        .build();
                            }

                            try {
                                Object result = invokeHandler.invoke();
                                if (!skipSpan) {
                                    rspSpan.setResult(methodName.startsWith(set) ? args : result);
                                }

                                return result;
                            } catch (Throwable e) {
                                if (!skipSpan) {
                                    rspSpan.setThrowable(e);
                                } else {
                                    JaLog.get().error(methodName + " " + args[0], e);
                                }
                                throw e;
                            } finally {
                                if (!skipSpan) {
                                    JaDapperLog.response(rspSpan);
                                }
                            }
                        });
            }

        } catch (Throwable e) {
            JaLog.get().error("create redis error : {}", JaJson.toString(JaRedisConfig.load()), e);
            throw e;
        }

    }

    /**
     * 大对象list, 分割存储
     */
    public static <T> void setBigList(String key, List<T> list, int size, Long... seconds) {
        List<List<T>> lists = JaCollectionUtil.splitBySize(list, size);
        JaConcurrentTasks build = JaConcurrentTasks.build();
        for (int i = 0; i < size; i++) {
            int finalI = i;
            build.add(() -> setJson(key + "-" + finalI, lists.get(finalI), seconds));
        }
        build.add(() -> setT(key + "-size", size, seconds)).get();
    }

    /**
     * 获取分割之后大对象list
     */
    public static <T> List<T> getBigList(String key, Type type) {
        Integer size = getT(key + "-size");
        if (null == size) {
            return null;
        }
        JaConcurrentSubmit<T> build = JaConcurrentSubmit.build();
        for (int i = 0; i < size; i++) {
            int finalI = i;
            build.add(() -> getByJson(key + "-" + finalI, type));
        }
        return build.get();
    }

    public static Boolean setString(String key, String value, Long... seconds) {
        return redisCommand.set(key, value, seconds);
    }

    public static Boolean setNx(String key, String value, Long... seconds) {
        return redisCommand.setNx(key, value, seconds);
    }

    /**
     * 自适应类型, 存储的是json, json中含有type
     */
    public static <T> Boolean setT(String key, T value, Long... seconds) {
        return redisCommand.set(key, JacksonAutoType.Instance.toString(value), seconds);
    }

    /**
     * 存储的是json, 原生json, 效率高
     */
    public static <T> Boolean setJson(String key, T value, Long... seconds) {
        return redisCommand.set(key, JaJson.toString(value), seconds);
    }

    public static String getString(String key) {
        return redisCommand.get(key);
    }

    /**
     * 存储的是原生json, 需要指定type获取, 效率高
     */
    public static <T> T getByJson(String key, Type type) {
        String json = redisCommand.get(key);
        return JaStringUtil.isEmpty(json) ? null : JaJson.toObject(json, type);
    }

    /**
     * 存储的是含有type的json, 自适应type, 效率低
     */
    public static <T> T getT(String key) {
        String json = redisCommand.get(key);
        return JaStringUtil.isEmpty(json) ? null : JacksonAutoType.Instance.toObject(json);
    }

    public static Long del(String key) {
        return redisCommand.del(key);
    }

    public static Boolean exists(String key) {
        return redisCommand.exists(key);
    }

    public static Long expire(String key, Long seconds) {
        return redisCommand.expire(key, seconds);
    }

    public static Long incr(String key, Long... increment) {
        return redisCommand.incr(key, increment);
    }

    public static Long decr(String key, Long... increment) {
        return redisCommand.decr(key, increment);
    }

    public static Object eval(String script, List<String> keys, List<String> args) {
        return redisCommand.eval(script, keys, args);
    }

    /**
     * LIFO, get 0 永远都是最新的
     */
    public static Long lpush(String key, String... args) {
        return redisCommand.lpush(key, args);
    }

    /**
     * FIFO, get 0 不会变
     */
    public static Long rpush(String key, String... args) {
        return redisCommand.rpush(key, args);
    }

    /**
     * @param start 从0开始
     * @param end   -1取所有
     */
    public static List<String> lrange(String key, long start, long end) {
        return redisCommand.lrange(key, start, end);
    }

    /**
     * @param index 从0开始, 精准取list某个值
     */
    public static String lindex(String key, long index) {
        return redisCommand.lindex(key, index);
    }

    /**
     * @param count 大于 0：从头部开始移除最多 count 个匹配的元素。
     *              小于 0：从尾部开始移除最多 count 的绝对值个匹配的元素。
     *              0：移除所有匹配的元素。
     * @param value 移除的元素
     */
    public static Long lrem(String key, long count, String value) {
        return redisCommand.lrem(key, count, value);
    }

    /**
     * 先从redis中获取, 没有则执行业务函数, 并set redis {@see JaRedis}
     * {@link JaRedis#setT(String, Object, Long...)}
     * {@link JaRedis#getT(String)}
     *
     * @param key      redis key
     * @param callable 执行的业务函数
     * @param seconds  失效时间
     * @param <T>      返回对象
     * @return 业务值
     */
    @SneakyThrows
    public static <T> T cacheT(String key, Callable<T> callable, Long... seconds) {
        return cache(callable
                , () -> getT(key)
                , value -> setT(key, value, seconds)
        );
    }

    /**
     * 先从redis中获取, 没有则执行业务函数, 并set redis
     * {@link JaRedis#setJson(String, Object, Long...)}
     * {@link JaRedis#getByJson(String, Type)}
     *
     * @param key      redis key
     * @param callable 执行的业务函数
     * @param seconds  失效时间
     * @param <T>      返回对象
     * @return 业务值
     */
    @SneakyThrows
    public static <T> T cacheJson(String key, Callable<T> callable, Type type, Long... seconds) {
        return cache(callable
                , () -> getByJson(key, type)
                , value -> setJson(key, value, seconds)
        );
    }


    /**
     * @param callable 执行的业务函数
     * @param get      缓存 get
     * @param set      缓存 set
     * @param <T>      返回对象
     * @return 业务值
     */
    @SneakyThrows
    public static <T> T cache(Callable<T> callable, Callable<T> get, JaFunctionP<T> set) {
        T t;
        try {
            t = get.call();
            if (null != t) {
                return t;
            }
        } catch (Throwable e) {
            JaLog.get().error("redis get error", e);
        }
        t = callable.call();
        if (null != t) {
            set.apply(t);
        }
        return t;
    }


}
