package com.jugg.agile.middleware.redis;

import com.jugg.agile.framework.core.config.JaEnvProperty;
import com.jugg.agile.framework.core.util.io.serialize.json.JaJson;
import com.jugg.agile.framework.core.util.io.serialize.json.adapter.jackson.migration.JacksonAutoType;

import java.lang.reflect.Type;
import java.util.List;

/**
 * redis key 自动添加服务名作为前缀
 *
 * @author chenjian
 * @since 2025年10月28日 17:45:28
 */
public class JaRedisNamespace {

    public static Boolean setString(String key, String value, Long... seconds) {
        return JaRedis.setString(wrapKey(key), value, seconds);
    }

    public static Boolean setNx(String key, String value, Long... seconds) {
        return JaRedis.setNx(wrapKey(key), value, seconds);
    }

    /**
     * 自适应类型, 存储的是json, json中含有type
     */
    public static <T> Boolean setT(String key, T value, Long... seconds) {
        return JaRedis.setT(wrapKey(key), JacksonAutoType.Instance.toString(value), seconds);
    }

    /**
     * 存储的是json, 原生json, 效率高
     */
    public static <T> Boolean setJson(String key, T value, Long... seconds) {
        return JaRedis.setJson(wrapKey(key), JaJson.toString(value), seconds);
    }

    public static String getString(String key) {
        return JaRedis.getString(wrapKey(key));
    }

    /**
     * 存储的是原生json, 需要指定type获取, 效率高
     */
    public static <T> T getByJson(String key, Type type) {
        return JaRedis.getByJson(wrapKey(key), type);
    }

    /**
     * 存储的是含有type的json, 自适应type, 效率低
     */
    public static <T> T getT(String key) {
        return JaRedis.getT(wrapKey(key));
    }

    public static Long del(String key) {
        return JaRedis.del(wrapKey(key));
    }

    public static Boolean exists(String key) {
        return JaRedis.exists(wrapKey(key));
    }

    public static Long expire(String key, Long seconds) {
        return JaRedis.expire(wrapKey(key), seconds);
    }

    public static Long incr(String key, Long... increment) {
        return JaRedis.incr(wrapKey(key), increment);
    }

    public static Long decr(String key, Long... increment) {
        return JaRedis.decr(wrapKey(key), increment);
    }

    public static Object eval(String script, List<String> keys, List<String> args) {
        return JaRedis.eval(script, keys, args);
    }

    /**
     * LIFO, get 0 永远都是最新的
     */
    public static Long lpush(String key, String... args) {
        return JaRedis.lpush(wrapKey(key), args);
    }

    /**
     * FIFO, get 0 不会变
     */
    public static Long rpush(String key, String... args) {
        return JaRedis.rpush(wrapKey(key), args);
    }

    /**
     * @param start 从0开始
     * @param end   -1取所有
     */
    public static List<String> lrange(String key, long start, long end) {
        return JaRedis.lrange(wrapKey(key), start, end);
    }

    /**
     * @param index 从0开始, 精准取list某个值
     */
    public static String lindex(String key, long index) {
        return JaRedis.lindex(wrapKey(key), index);
    }

    /**
     * @param count 大于 0：从头部开始移除最多 count 个匹配的元素。
     *              小于 0：从尾部开始移除最多 count 的绝对值个匹配的元素。
     *              0：移除所有匹配的元素。
     * @param value 移除的元素
     */
    public static Long lrem(String key, long count, String value) {
        return JaRedis.lrem(wrapKey(key), count, value);
    }

    public static String wrapKey(String key) {
        return JaEnvProperty.getApplicationName() + ":" + key;

    }
}
