package com.jugg.agile.middleware.redis.adapter.redisson.pool;

import com.jugg.agile.framework.core.dapper.log.JaLog;
import com.jugg.agile.framework.core.config.JaProperty;
import com.jugg.agile.framework.core.config.JaSystemProperty;
import com.jugg.agile.framework.core.util.JaStringUtil;
import org.redisson.Redisson;
import org.redisson.api.RedissonClient;
import org.redisson.codec.JsonJacksonCodec;
import org.redisson.config.Config;
import org.redisson.config.TransportMode;

/**
 * 创建redisson pool
 *
 * @author chenjian
 * @since 2020年08月14日 16:34:25
 */
public class RedissonClusterPool {
    private static volatile RedissonClient redissonClient;

    public static void init() {
        if (null == redissonClient) {
            synchronized (RedissonClusterPool.class) {
                if (null == redissonClient) {
                    Config config = readConfig();
                    try {
                        redissonClient = Redisson.create(readConfig());
                    } catch (Throwable e) {
                        JaLog.get().error("redisson init error", e);
                        config.setTransportMode(TransportMode.NIO);
                        redissonClient = Redisson.create(config);
                    }
                }
            }
        }
    }

    public static RedissonClient get() {
        return redissonClient;
    }

    /**
     * 加载redisson配置
     *
     * @author chenjian
     * @since 2020年09月02日 18:00:02
     */
    private static Config readConfig() {
        // eventLoopGroup netty 调度线程配置
        // performanceMode 高性能模式 该参数仅限于Redisson PRO版本
        Config config = new Config();
        // 需要有netty-transport-native-epoll包 TransportMode
        String epoll = get("epoll");
        if (!JaSystemProperty.isWindows()) {
            config.setTransportMode(TransportMode.EPOLL);
        } else {
            config.setTransportMode(TransportMode.NIO);
        }

        // 监控锁的看门狗超时，单位：毫秒, 默认值：30000
        // 该参数只适用于分布式锁的加锁请求中未明确使用leaseTimeout参数的情况
        String lockWatchdogTimeout = get("lockWatchdogTimeout");
        if (JaStringUtil.isNotEmpty(lockWatchdogTimeout)) {
            config.setLockWatchdogTimeout(Long.parseLong(lockWatchdogTimeout));
        }
        // 保持订阅发布顺序, 默认值：true
        config.setKeepPubSubOrder(false);

        String cluster = JaProperty.get("redis.cluster");
        assert cluster != null;
        String[] clusterArray = cluster.split(",");
        for (int i = 0; i < clusterArray.length; i++) {
            clusterArray[i] = "redis://" + clusterArray[i];
        }

        // 集群扫描间隔时间
        String scanInterval = get("scanInterval");
        // 从节点发布和订阅连接的最小空闲连接数, 默认值：1
        String subscriptionConnectionMinimumIdleSize = get("subscriptionConnectionMinimumIdleSize");
        // 从节点发布和订阅连接池大小, 默认值：50
        String subscriptionConnectionPoolSize = get("subscriptionConnectionPoolSize");
        // 单个连接最大订阅数量, 默认值：5, 每个连接的最大订阅数量
        String subscriptionsPerConnection = get("subscriptionsPerConnection");
        // 从节点最小空闲连接数, 默认值：32, 非 发布和订阅
        String slaveConnectionMinimumIdleSize = get("slaveConnectionMinimumIdleSize");
        // 从节点连接池大小, 默认值：64, 非 发布和订阅
        String slaveConnectionPoolSize = get("slaveConnectionPoolSize");
        // 主节点最小空闲连接数, 默认值：32
        String masterConnectionMinimumIdleSize = get("masterConnectionMinimumIdleSize");
        // 主节点连接池大小, 默认值：64, 多主节点的环境里，每个 主节点的连接池最大容量。连接池的连接数量自动弹性伸缩
        String masterConnectionPoolSize = get("masterConnectionPoolSize");
        // 连接空闲超时，单位：毫秒, 默认值：10000
        String idleConnectionTimeout = get("idleConnectionTimeout");
        // 连接超时，单位：毫秒, 同任何节点建立连接时的等待超时。时间单位是毫秒
        String connectTimeout = get("connectTimeout");
        // 命令等待超时，单位：毫秒
        String timeout = get("timeout");
        // 命令失败重试次数, 默认值：3
        String retryAttempts = get("retryAttempts");
        // 命令重试发送时间间隔，单位：毫秒, 默认值：1500
        String retryInterval = get("retryInterval");
        // 重新连接时间间隔，单位：毫秒, 默认值：3000?????
        String reconnectionTimeout = get("reconnectionTimeout");
        // 执行失败最大次数, 默认值：3
        // 在某个节点执行相同或不同命令时，连续 失败 failedAttempts（执行失败最大次数） 时，该节点将被从可用节点列表里清除，直到 reconnectionTimeout（重新连接时间间隔） 超时以后再次尝试
        String failedAttempts = get("failedAttempts");
        config.setCodec(JsonJacksonCodec.INSTANCE);
        config.useClusterServers()
                .setPassword(JaProperty.get("redis.password"))
                .setScanInterval(JaStringUtil.isEmpty(scanInterval) ? 1000 * 60 * 30 : Integer.parseInt(scanInterval))
                .setSubscriptionConnectionMinimumIdleSize(JaStringUtil.isEmpty(subscriptionConnectionMinimumIdleSize) ? 1 : Integer.parseInt(subscriptionConnectionMinimumIdleSize))
                .setSubscriptionConnectionPoolSize(JaStringUtil.isEmpty(subscriptionConnectionPoolSize) ? 20 : Integer.parseInt(subscriptionConnectionPoolSize))
                .setSubscriptionsPerConnection(JaStringUtil.isEmpty(subscriptionsPerConnection) ? 5 : Integer.parseInt(subscriptionsPerConnection))
                .setSlaveConnectionMinimumIdleSize(JaStringUtil.isEmpty(slaveConnectionMinimumIdleSize) ? 12 : Integer.parseInt(slaveConnectionMinimumIdleSize))
                .setSlaveConnectionPoolSize(JaStringUtil.isEmpty(slaveConnectionPoolSize) ? 64 : Integer.parseInt(slaveConnectionPoolSize))
                .setMasterConnectionMinimumIdleSize(JaStringUtil.isEmpty(masterConnectionMinimumIdleSize) ? 12 : Integer.parseInt(masterConnectionMinimumIdleSize))
                .setMasterConnectionPoolSize(JaStringUtil.isEmpty(masterConnectionPoolSize) ? 64 : Integer.parseInt(masterConnectionPoolSize))
                .setIdleConnectionTimeout(JaStringUtil.isEmpty(idleConnectionTimeout) ? 10000 : Integer.parseInt(idleConnectionTimeout))
                .setConnectTimeout(JaStringUtil.isEmpty(connectTimeout) ? 4000 : Integer.parseInt(connectTimeout))
                .setTimeout(JaStringUtil.isEmpty(timeout) ? 2000 : Integer.parseInt(timeout))
                .setRetryAttempts(JaStringUtil.isEmpty(retryAttempts) ? 0 : Integer.parseInt(retryAttempts))
                .setRetryInterval(JaStringUtil.isEmpty(retryInterval) ? 1000 : Integer.parseInt(retryInterval))
//                .setreconnectionTimeout(JaStringUtil.isEmpty(retryInterval) ? 1 : Integer.parseInt(retryInterval))
//                .failedAttemp(JaStringUtil.isEmpty(retryInterval) ? 1 : Integer.parseInt(retryInterval))
                .addNodeAddress(clusterArray);
        return config;
    }

    private static String get(String name) {
        return JaProperty.get("redis.redisson" + name);
    }
}
