package com.jugg.agile.middleware.redis;

import com.jugg.agile.framework.core.config.JaProperty;
import com.jugg.agile.framework.core.dapper.log.JaLog;
import com.jugg.agile.framework.core.dapper.meta.NodeKind;
import com.jugg.agile.framework.core.dapper.meta.NodeSpan;
import com.jugg.agile.framework.core.dapper.meta.RespSpan;
import com.jugg.agile.framework.core.meta.JaFunctionP;
import com.jugg.agile.framework.core.util.JaStringUtil;
import com.jugg.agile.framework.core.util.bytecode.proxy.JaProxyCglib;
import com.jugg.agile.framework.core.util.concurrent.JaConcurrentSubmit;
import com.jugg.agile.framework.core.util.concurrent.JaConcurrentTasks;
import com.jugg.agile.framework.core.util.concurrent.JaThreadLocal;
import com.jugg.agile.framework.core.util.datastructure.JaCollectionUtil;
import com.jugg.agile.framework.core.util.io.serialize.json.JaJson;
import com.jugg.agile.framework.core.util.io.serialize.json.adapter.jackson.migration.JacksonAutoType;
import com.jugg.agile.middleware.redis.adapter.jedis.adapter.JaJedisSimpleAdapter;
import com.jugg.agile.middleware.redis.config.JaRedisConfig;
import lombok.SneakyThrows;

import java.lang.reflect.Type;
import java.util.List;
import java.util.concurrent.Callable;

public class JaRedis {

    private JaRedis() {
    }

    private static final String set = "set";
    private static final JaRedisCommand redisCommand;

    public static final JaThreadLocal<Boolean> SkipSpanThreadLocal = new JaThreadLocal<>();

    static {
        try {
            redisCommand = JaProxyCglib.newInstance(JaJedisSimpleAdapter.class
                    , (proxyInstance, method, args, methodProxy, invokeHandler) -> {
                        String methodName = method.getName();
                        boolean skipSpan = null != SkipSpanThreadLocal.get() && SkipSpanThreadLocal.get();
                        RespSpan respSpan = null;
                        if (!skipSpan) {
                            respSpan = RespSpan.builder()
                                    .nodeSpan(NodeSpan.builder()
                                            .nodeKind(NodeKind.Constant.Redis)
                                            .id(methodName + " " + args[0])// args[0] -> redis key
                                            .build())
                                    .build();
                        }

                        try {
                            Object result = invokeHandler.invoke();
                            if (!skipSpan) {
                                respSpan.setResult(methodName.startsWith(set) ? args : result);
                            }

                            return result;
                        } catch (Throwable e) {
                            if (!skipSpan) {
                                respSpan.setThrowable(e);
                            } else {
                                JaLog.get().error(methodName + " " + args[0], e);
                            }
                            throw e;
                        } finally {
                            if (!skipSpan) {
                                JaLog.response(respSpan);
                            }
                        }
                    });
        } catch (Throwable e) {
            JaLog.get().error("create redis error : {} {}"
                    , JaProperty.get(JaRedisConfig.Key_Cluster)
                    , JaProperty.get(JaRedisConfig.Key_Password)
                    , e);
            throw e;
        }

    }

    /**
     * 大对象list, 分割存储
     */
    public static <T> void setBigList(String key, List<T> list, int size, Long... seconds) {
        List<List<T>> lists = JaCollectionUtil.splitBySize(list, size);
        JaConcurrentTasks build = JaConcurrentTasks.build();
        for (int i = 0; i < size; i++) {
            int finalI = i;
            build.add(() -> setJson(key + "-" + finalI, lists.get(finalI), seconds));
        }
        build.add(() -> setT(key + "-size", size, seconds)).get();
    }

    /**
     * 获取分割之后大对象list
     */
    public static <T> List<T> getBigList(String key, Type type) {
        Integer size = getT(key + "-size");
        if (null == size) {
            return null;
        }
        JaConcurrentSubmit<T> build = JaConcurrentSubmit.build();
        for (int i = 0; i < size; i++) {
            int finalI = i;
            build.add(() -> getByJson(key + "-" + finalI, type));
        }
        return build.get();
    }

    public static Boolean setString(String key, String value, Long... seconds) {
        return redisCommand.set(key, value, seconds);
    }

    public static Boolean setNx(String key, String value, Long... seconds) {
        return redisCommand.setNx(key, value, seconds);
    }

    /**
     * 自适应类型, 存储的是json, json中含有type
     */
    public static <T> Boolean setT(String key, T value, Long... seconds) {
        return redisCommand.set(key, JacksonAutoType.Instance.toString(value), seconds);
    }

    /**
     * 存储的是json, 原生json, 效率高
     */
    public static <T> Boolean setJson(String key, T value, Long... seconds) {
        return redisCommand.set(key, JaJson.toString(value), seconds);
    }

    public static String getString(String key) {
        return redisCommand.get(key);
    }

    /**
     * 存储的是原生json, 需要指定type获取, 效率高
     */
    public static <T> T getByJson(String key, Type type) {
        String json = redisCommand.get(key);
        return JaStringUtil.isEmpty(json) ? null : JaJson.toObject(json, type);
    }

    /**
     * 存储的是含有type的json, 自适应type, 效率低
     */
    public static <T> T getT(String key) {
        String json = redisCommand.get(key);
        return JaStringUtil.isEmpty(json) ? null : JacksonAutoType.Instance.toObject(json);
    }

    public static Long del(String key) {
        return redisCommand.del(key);
    }

    public static Boolean exists(String key) {
        return redisCommand.exists(key);
    }

    public static Long expire(String key, Long seconds) {
        return redisCommand.expire(key, seconds);
    }

    public static Long incr(String key, Long... increment) {
        return redisCommand.incr(key, increment);
    }

    public static Long decr(String key, Long... increment) {
        return redisCommand.decr(key, increment);
    }

    /**
     * 先从redis中获取, 没有则执行业务函数, 并set redis {@see JaRedis}
     * {@link JaRedis#setT(String, Object, Long...)}
     * {@link JaRedis#getT(String)}
     *
     * @param key      redis key
     * @param callable 执行的业务函数
     * @param seconds  失效时间
     * @param <T>      返回对象
     * @return 业务值
     */
    @SneakyThrows
    public static <T> T cacheT(String key, Callable<T> callable, Long... seconds) {
        return cache(callable
                , () -> getT(key)
                , value -> setT(key, value, seconds)
        );
    }

    /**
     * 先从redis中获取, 没有则执行业务函数, 并set redis
     * {@link JaRedis#setJson(String, Object, Long...)}
     * {@link JaRedis#getByJson(String, Type)}
     *
     * @param key      redis key
     * @param callable 执行的业务函数
     * @param seconds  失效时间
     * @param <T>      返回对象
     * @return 业务值
     */
    @SneakyThrows
    public static <T> T cacheJson(String key, Callable<T> callable, Type type, Long... seconds) {
        return cache(callable
                , () -> getByJson(key, type)
                , value -> setJson(key, value, seconds)
        );
    }


    /**
     * @param callable 执行的业务函数
     * @param get      缓存 get
     * @param set      缓存 set
     * @param <T>      返回对象
     * @return 业务值
     */
    @SneakyThrows
    public static <T> T cache(Callable<T> callable, Callable<T> get, JaFunctionP<T> set) {
        T t;
        try {
            t = get.call();
            if (null != t) {
                return t;
            }
        } catch (Throwable e) {
            JaLog.get().error("redis get error", e);
        }
        t = callable.call();
        if (null != t) {
            set.apply(t);
        }
        return t;
    }
}
