package  com.jugg.agile.middleware.seata.utils;

import org.springframework.context.EnvironmentAware;
import org.springframework.core.env.Environment;
import org.springframework.util.StringUtils;

import java.util.Objects;

/**
 * 获取环境变量
 *
 * @author wuzq
 * @version 1.0.0
 * @date 2024/06/06
 */
public class JaEnvironmentUtil implements EnvironmentAware {
    private static final String APPLICATION_NAME_KEY = "app.name";
    private static final String DEFAULT_APPLICATION_NAME = "spring.application.name";

    private static Environment environment;

    /**
     * 获取当前服务的环境对象实例
     *
     * @return 环境Bean
     */
    public static Environment obtainEnvironment() {
        if (Objects.isNull(environment)) {
            throw new RuntimeException("获取环境对象失败");
        }
        return environment;
    }

    @Override
    public void setEnvironment(Environment environment) {
        this.environment = environment;
    }

    /**
     * 获取String类型属性
     *
     * @param key 属性key
     * @return String属性值
     */
    public static String getProperty(String key) {
        return getProperty(key, String.class);
    }

    /**
     * 获取String类型属性
     *
     * @param key          属性key
     * @param defaultValue 默认值
     * @return String属性值
     */
    public static String getProperty(String key, String defaultValue) {
        String value = getProperty(key);
        return (Objects.nonNull(value) ? value : defaultValue);
    }


    /**
     * 获取环境属性值
     *
     * @param key             属性key
     * @param targetValueType 属性类型
     * @param <T>             泛型
     * @return 属性值
     */
    public static <T> T getProperty(String key, Class<T> targetValueType) {
        return obtainEnvironment().getProperty(key, targetValueType);
    }

    /**
     * 获取环境属性值
     *
     * @param key             属性key
     * @param targetValueType 属性类型
     * @param defaultValue    默认值
     * @param <T>             泛型
     * @return 属性值
     */
    public static <T> T getProperty(String key, Class<T> targetValueType, T defaultValue) {
        T value = getProperty(key, targetValueType);
        return Objects.nonNull(value) ? value : defaultValue;
    }

    /**
     * 获取当前服务的微服务名称
     * 优先取app.name
     * 取不到再取spring.application.name
     *
     * @return 服务名
     */
    public static String getAppName() {
        String appId = obtainEnvironment().getProperty(APPLICATION_NAME_KEY);
        return StringUtils.hasText(appId) ? appId : obtainEnvironment().getProperty(DEFAULT_APPLICATION_NAME);
    }

}
