package com.jugg.agile.middleware.seata.processor;

import com.jugg.agile.middleware.seata.properties.JaDistributedTransactionProperties;
import org.springframework.boot.SpringApplication;
import org.springframework.boot.context.properties.bind.Bindable;
import org.springframework.boot.context.properties.bind.Binder;
import org.springframework.boot.env.EnvironmentPostProcessor;
import org.springframework.core.env.ConfigurableEnvironment;
import org.springframework.core.env.MapPropertySource;
import org.springframework.core.env.PropertiesPropertySource;
import org.springframework.util.StringUtils;

import java.util.HashMap;
import java.util.Map;
import java.util.Properties;

/**
 * seata properties 预设置
 * 不影响原生seata的配置使用，只需配置 {@link JaDistributedTransactionProperties} 中的几个属性即可
 *
 * @author wuzq
 * @version 1.0.0
 * @date 2024/06/06
 */
public class JaSeataPostPropertyProcessor implements EnvironmentPostProcessor {

    @Override
    public void postProcessEnvironment(ConfigurableEnvironment environment, SpringApplication application) {
        if (isCloudContainer()) {
            return;
        }
        checkParams(environment);

        JaDistributedTransactionProperties distributedTransactionProperties = new JaDistributedTransactionProperties();
        Binder.get(environment).bind("ja.distributed-transaction", Bindable.ofInstance(distributedTransactionProperties));
        Properties seataProps = new Properties();
        seataProps.setProperty("seata.enabled", "false");
        if (distributedTransactionProperties.getEnabled()) {
            setSeataProperty("seata.application-id", environment, distributedTransactionProperties.getApplicationId(environment), seataProps, true);
            setSeataProperty("seata.enable-auto-data-source-proxy", environment, distributedTransactionProperties.getEnableAutoDataSourceProxy(), seataProps, false);
            setSeataProperty("seata.tx-service-group", environment, distributedTransactionProperties.getTxServiceGroup(), seataProps, false);
            setSeataProperty("seata.service.vgroupMapping", environment, distributedTransactionProperties.getServiceVgroupMapping(), seataProps, false);
            distributedTransactionProperties.getServiceVgroupMapping().forEach((k, v) -> setSeataProperty("seata.service.vgroupMapping." + k, environment, v, seataProps, false));
            setSeataProperty("seata.service.grouplist", environment, distributedTransactionProperties.getServiceGrouplist(), seataProps, false);
            distributedTransactionProperties.getServiceGrouplist().forEach((k, v) -> setSeataProperty("seata.service.grouplist." + k, environment, v, seataProps, false));
            setSeataProperty("seata.client.tm.defaultGlobalTransactionTimeout", environment, distributedTransactionProperties.getClientTmDefaultGlobalTransactionTimeout(), seataProps, false);
            setSeataProperty("seata.client.tm.rollbackRetryCount", environment, distributedTransactionProperties.getClientTmRollbackRetryCount(), seataProps, false);
            setSeataProperty("seata.client.tm.commitRetryCount", environment, distributedTransactionProperties.getClientTmCommitRetryCount(), seataProps, false);

            if ("nacos".equals(distributedTransactionProperties.getRegistryType())) {
                seataProps.setProperty("seata.registry.type", "nacos");
                setSeataProperty("seata.registry.nacos.server-addr", environment, distributedTransactionProperties.getNacosRegistryServerAddr(), seataProps, true);
                setSeataProperty("seata.registry.nacos.cluster", environment, distributedTransactionProperties.getNacosRegistryCluster(), seataProps, true);
                setSeataProperty("seata.registry.nacos.group", environment, distributedTransactionProperties.getNacosRegistryGroup(), seataProps, false);
                setSeataProperty("seata.registry.nacos.namespace", environment, distributedTransactionProperties.getNacosRegistryNameSpace(), seataProps, false);
                setSeataProperty("seata.registry.nacos.password", environment, distributedTransactionProperties.getNacosRegistryPassword(), seataProps, false);
                setSeataProperty("seata.registry.nacos.username", environment, distributedTransactionProperties.getNacosRegistryUsername(), seataProps, false);
                setSeataProperty("seata.registry.nacos.application", environment, distributedTransactionProperties.getNacosRegistryApplication(), seataProps, false);
            }
            if ("nacos".equals(distributedTransactionProperties.getConfigType())) {
                seataProps.setProperty("seata.config.type", "nacos");
                setSeataProperty("seata.config.nacos.server-addr", environment, distributedTransactionProperties.getNacosConfigServerAddr(), seataProps, true);
                setSeataProperty("seata.config.nacos.group", environment, distributedTransactionProperties.getNacosConfigGroup(), seataProps, false);
                setSeataProperty("seata.config.nacos.data-id", environment, distributedTransactionProperties.getNacosConfigDataId(), seataProps, false);
                setSeataProperty("seata.config.nacos.namespace", environment, distributedTransactionProperties.getNacosConfigNamespace(), seataProps, false);
                setSeataProperty("seata.config.nacos.password", environment, distributedTransactionProperties.getNacosConfigPassword(), seataProps, false);
                setSeataProperty("seata.config.nacos.username", environment, distributedTransactionProperties.getNacosConfigUsername(), seataProps, false);
            }
        }
        environment.getPropertySources().addLast(new PropertiesPropertySource("jaSeata", seataProps));
    }

    private void checkParams(ConfigurableEnvironment environment) {
        Map<String, Object> map = new HashMap<>();
        if (!isBoolean(environment.getProperty("ja.distributed-transaction.enabled"))){
            map.put("ja.distributed-transaction.enabled", false);
        }
        if (!isBoolean(environment.getProperty("ja.distributed-transaction.enable-auto-data-source-proxy"))){
            map.put("ja.distributed-transaction.enable-auto-data-source-proxy", true);
        }
        if (!isInteger(environment.getProperty("ja.distributed-transaction.client-tm-default-global-transaction-timeout"))){
            map.put("ja.distributed-transaction.client-tm-default-global-transaction-timeout", 120000);
        }

        MapPropertySource propertySource = new MapPropertySource("jaSeata", map);
        environment.getPropertySources().addFirst(propertySource);
    }

    public void setSeataProperty(String propertyName, ConfigurableEnvironment environment, Object propertyValue, Properties seataProps, boolean must) {
        Object value = environment.getProperty(propertyName);
        if (StringUtils.isEmpty(value)) {
            value = propertyValue;
            if (StringUtils.isEmpty(value) && must) {
                throw new RuntimeException("[" + propertyName + "] is must seata config property");
            }
            seataProps.put(propertyName, value);
        }
    }

    public boolean isCloudContainer() {
        StackTraceElement[] stackTrace = Thread.currentThread().getStackTrace();
        for (StackTraceElement stackTraceElement : stackTrace) {
            if (stackTraceElement.getClassName().endsWith("BootstrapApplicationListener")) {
                return true;
            }
        }
        return false;
    }

    public boolean isBoolean(String str) {
        if (str == null) {
            return false;
        }
        String lowerCaseStr = str.toLowerCase();
        return "true".equals(lowerCaseStr) || "false".equals(lowerCaseStr);
    }

    public static boolean isInteger(String str) {
        try {
            Integer.parseInt(str);
            return true;
        } catch (NumberFormatException e) {
            return false;
        }
    }
}
