package com.jugg.agile.middleware.seata;

import io.seata.common.util.CollectionUtils;
import io.seata.config.ConfigurationFactory;
import io.seata.core.exception.TmTransactionException;
import io.seata.core.exception.TransactionException;
import io.seata.core.exception.TransactionExceptionCode;
import io.seata.core.model.GlobalStatus;
import io.seata.tm.api.GlobalTransaction;
import io.seata.tm.api.GlobalTransactionContext;
import io.seata.tm.api.GlobalTransactionRole;
import io.seata.tm.api.TransactionalExecutor;
import io.seata.tm.api.transaction.TransactionInfo;
import lombok.extern.slf4j.Slf4j;

import java.util.Objects;

/**
 * 分布式事务模版类
 *
 * @author wuzq
 * @version 1.0.0
 * @date 2024/06/06
 */
@Slf4j
public class JaGlobalTransactionalTemplate {
    public static Object handleGlobalTransaction(TransactionInfo txInfo, JaFunctionTR business) throws Throwable {
        if (Objects.isNull(txInfo)) {
            txInfo = new TransactionInfo();
        }
        if (txInfo.getTimeOut() <= 0) {
            txInfo.setTimeOut(ConfigurationFactory.getInstance().getInt("client.tm.default-global-transaction-timeout", 120000));
        }
        if (Objects.isNull(txInfo.getName()) || txInfo.getName().isEmpty()) {
            txInfo.setName(ConfigurationFactory.getInstance().getConfig("application-id"));
        }

        //采用默认的事务传播途径REQUIRED
        GlobalTransaction tx = GlobalTransactionContext.getCurrentOrCreate();

        // 开启全局事务
        beginTransaction(txInfo, tx);

        Object result;
        try {
            //执行业务代码
            result = business.apply();
        } catch (Throwable e) {
            completeTransactionAfterThrowing(txInfo, tx, e);
            throw e;
        }
        commitTransaction(tx, txInfo);
        return result;
    }

    public static Object handleGlobalTransaction(TransactionInfo txInfo, JaFunctionTR business, JaFunctionTR<Boolean> judge) throws Throwable {
        if (!Objects.isNull(judge) && judge.apply()) {
            return handleGlobalTransaction(txInfo, business);
        }
        return business.apply();
    }

    private static void beginTransaction(TransactionInfo txInfo, GlobalTransaction tx) throws TransactionalExecutor.ExecutionException {
        if (tx.getGlobalTransactionRole() != GlobalTransactionRole.Launcher) {
            log.debug("Ignore begin: just involved in global transaction [{}]", tx.getXid());
            return;
        }

        try {
            tx.begin(txInfo.getTimeOut(), txInfo.getName());
        } catch (TransactionException txe) {
            throw new TransactionalExecutor.ExecutionException(tx, txe, TransactionalExecutor.Code.BeginFailure);
        }
    }

    private static void commitTransaction(GlobalTransaction tx, TransactionInfo txInfo) throws TransactionalExecutor.ExecutionException {
        if (tx.getGlobalTransactionRole() != GlobalTransactionRole.Launcher) {
            log.debug("Ignore commit: just involved in global transaction [{}]", tx.getXid());
            return;
        }

        if (isTimeout(tx.getCreateTime(), txInfo.getTimeOut())) {
            // business execution timeout
            Exception exx = new TmTransactionException(TransactionExceptionCode.TransactionTimeout,
                    String.format("client detected transaction timeout before commit, so change to rollback, xid = %s", tx.getXid()));
            rollbackTransaction(tx, exx);
            return;
        }

        try {
            tx.commit();
            GlobalStatus afterCommitStatus = tx.getLocalStatus();
            TransactionalExecutor.Code code = TransactionalExecutor.Code.Unknown;
            switch (afterCommitStatus) {
                case TimeoutRollbacking:
                    code = TransactionalExecutor.Code.Rollbacking;
                    break;
                case TimeoutRollbacked:
                    code = TransactionalExecutor.Code.RollbackDone;
                    break;
                case Finished:
                    code = TransactionalExecutor.Code.CommitFailure;
                    break;
                default:
            }
            Exception statusException = null;
            if (GlobalStatus.isTwoPhaseHeuristic(afterCommitStatus)) {
                statusException = new TmTransactionException(TransactionExceptionCode.CommitHeuristic,
                        String.format("Global transaction[%s] not found, may be rollbacked.", tx.getXid()));
            } else if (GlobalStatus.isOnePhaseTimeout(afterCommitStatus)) {
                statusException = new TmTransactionException(TransactionExceptionCode.TransactionTimeout,
                        String.format("Global transaction[%s] is timeout and will be rollback[TC].", tx.getXid()));
            }
            if (null != statusException) {
                throw new TransactionalExecutor.ExecutionException(tx, statusException, code);
            }
        } catch (TransactionException txe) {
            throw new TransactionalExecutor.ExecutionException(tx, txe, TransactionalExecutor.Code.CommitFailure);
        }
    }

    private static void completeTransactionAfterThrowing(TransactionInfo txInfo, GlobalTransaction tx, Throwable originalException) throws TransactionalExecutor.ExecutionException {

        if (txInfo != null && CollectionUtils.isNotEmpty(txInfo.getRollbackRules())) {
            if (txInfo.rollbackOn(originalException)) {
                //如果配置了回滚错误类型，只按照配置错误回滚
                rollbackTransaction(tx, originalException);
            } else {
                //不在配置回滚范围直接提交
                commitTransaction(tx, txInfo);
            }
        } else {
            //没配置直接回滚
            rollbackTransaction(tx, originalException);
        }
    }

    private static void rollbackTransaction(GlobalTransaction tx, Throwable originalException) throws TransactionalExecutor.ExecutionException {
        if (tx.getGlobalTransactionRole() != GlobalTransactionRole.Launcher) {
            log.debug("Ignore rollback: just involved in global transaction [{}]", tx.getXid());
            return;
        }

        if (tx == null) {
            return;
        }
        try {
            tx.rollback();
        } catch (TransactionException txe) {
            throw new TransactionalExecutor.ExecutionException(tx, txe, TransactionalExecutor.Code.RollbackFailure, originalException);
        }

        TransactionalExecutor.Code code;
        switch (tx.getLocalStatus()) {
            case RollbackFailed:
            case TimeoutRollbackFailed:
            case RollbackRetryTimeout:
                code = TransactionalExecutor.Code.RollbackFailure;
                break;
            case Rollbacking:
            case RollbackRetrying:
            case TimeoutRollbacking:
            case TimeoutRollbackRetrying:
                code = TransactionalExecutor.Code.Rollbacking;
                break;
            case TimeoutRollbacked:
            case Rollbacked:
                //rollback transactions but do not exist are usually considered completed
            case Finished:
                code = TransactionalExecutor.Code.RollbackDone;
                break;
            default:
                code = TransactionalExecutor.Code.Unknown;
                log.warn("{} rollback in the state {}", tx.getXid(), tx.getLocalStatus());
        }
        throw new TransactionalExecutor.ExecutionException(tx, originalException, code, originalException);
    }

    private static boolean isTimeout(long beginTime, long timeoutMills) {
        return (System.currentTimeMillis() - beginTime) > timeoutMills;
    }

    @FunctionalInterface
    public interface JaFunctionTR<R> {

        R apply() throws Throwable;
    }
}
