package com.jugg.agile.middleware.seata.autoconfigure;

import com.jugg.agile.middleware.seata.aspect.JaGlobalTransactionalAspect;
import com.jugg.agile.middleware.seata.context.JaSeataRestTemplateInterceptor;
import com.jugg.agile.middleware.seata.utils.JaEnvironmentUtil;
import io.seata.common.loader.EnhancedServiceLoader;
import io.seata.spring.annotation.GlobalTransactionScanner;
import io.seata.spring.annotation.ScannerChecker;
import io.seata.spring.boot.autoconfigure.SeataAutoConfiguration;
import io.seata.spring.boot.autoconfigure.properties.SeataProperties;
import io.seata.tm.api.DefaultFailureHandlerImpl;
import io.seata.tm.api.FailureHandler;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.aop.aspectj.AspectJExpressionPointcut;
import org.springframework.aop.support.DefaultPointcutAdvisor;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.config.ConfigurableListableBeanFactory;
import org.springframework.boot.autoconfigure.AutoConfigureAfter;
import org.springframework.boot.autoconfigure.AutoConfigureBefore;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnExpression;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.DependsOn;
import org.springframework.core.env.Environment;
import org.springframework.web.client.RestTemplate;

import java.util.List;

import static io.seata.common.Constants.BEAN_NAME_FAILURE_HANDLER;
import static io.seata.common.Constants.BEAN_NAME_SPRING_APPLICATION_CONTEXT_PROVIDER;

/**
 * 全局事务自动配置类
 *
 * @author wuzq
 * @version 1.0.0
 * @date 2024/06/06
 */
@Configuration
@ConditionalOnExpression("${ja.distributed-transaction.enabled:true}")
@AutoConfigureBefore({SeataAutoConfiguration.class})
@AutoConfigureAfter({JaSeataCoreAutoConfiguration.class})
public class JaGlobalTransactionAutoConfiguration {
    private static final Logger LOGGER = LoggerFactory.getLogger(SeataAutoConfiguration.class);

    @Bean(BEAN_NAME_FAILURE_HANDLER)
    @ConditionalOnMissingBean(FailureHandler.class)
    public FailureHandler failureHandler() {
        return new DefaultFailureHandlerImpl();
    }

    @Bean
    @DependsOn({BEAN_NAME_SPRING_APPLICATION_CONTEXT_PROVIDER, BEAN_NAME_FAILURE_HANDLER})
    @ConditionalOnMissingBean(GlobalTransactionScanner.class)
    public static GlobalTransactionScanner globalTransactionScanner(SeataProperties seataProperties, FailureHandler failureHandler,
                                                                    ConfigurableListableBeanFactory beanFactory,
                                                                    @Autowired(required = false) List<ScannerChecker> scannerCheckers) {
        if (LOGGER.isInfoEnabled()) {
            LOGGER.info("Automatically configure Seata");
        }

        // set bean factory
        GlobalTransactionScanner.setBeanFactory(beanFactory);

        // add checkers
        // '/META-INF/services/io.seata.spring.annotation.ScannerChecker'
        GlobalTransactionScanner.addScannerCheckers(EnhancedServiceLoader.loadAll(ScannerChecker.class));
        // spring beans
        GlobalTransactionScanner.addScannerCheckers(scannerCheckers);

        // add scannable packages
        GlobalTransactionScanner.addScannablePackages(seataProperties.getScanPackages());
        // add excludeBeanNames
        GlobalTransactionScanner.addScannerExcludeBeanNames(seataProperties.getExcludesForScanning());
        //set accessKey and secretKey
        GlobalTransactionScanner.setAccessKey(seataProperties.getAccessKey());
        GlobalTransactionScanner.setSecretKey(seataProperties.getSecretKey());
        // create global transaction scanner
        return new GlobalTransactionScanner(seataProperties.getApplicationId(), seataProperties.getTxServiceGroup(), failureHandler);
    }

    @Bean
    public DefaultPointcutAdvisor jaGlobalTransactionalAspect() {
        AspectJExpressionPointcut pointcut = new AspectJExpressionPointcut();
        pointcut.setExpression("@annotation(com.jugg.agile.middleware.seata.annotation.JaGlobalTransactional)");
        JaGlobalTransactionalAspect interceptor = new JaGlobalTransactionalAspect();
        DefaultPointcutAdvisor advisor = new DefaultPointcutAdvisor();
        advisor.setPointcut(pointcut);
        advisor.setAdvice(interceptor);
        return advisor;
    }

    @Bean
    @ConditionalOnBean(RestTemplate.class)
    public JaSeataRestTemplateInterceptor jaSeataRestTemplateInterceptor(RestTemplate restTemplate) {
        JaSeataRestTemplateInterceptor jaSeataRestTemplateInterceptor = new JaSeataRestTemplateInterceptor();
        restTemplate.getInterceptors().add(jaSeataRestTemplateInterceptor);
        return jaSeataRestTemplateInterceptor;
    }

    @Bean
    public JaEnvironmentUtil jaEnvironmentUtil() {
        return new JaEnvironmentUtil();
    }
}
