package com.jugg.agile.spring.boot.webmvc.servlet;

import com.jugg.agile.framework.core.context.biz.JaCoreContext;
import com.jugg.agile.framework.core.dapper.log.JaLog;
import com.jugg.agile.framework.core.dapper.log.JaMDC;
import com.jugg.agile.framework.core.util.JaStringUtil;
import com.jugg.agile.framework.core.util.JaThrowableUtil;
import com.jugg.agile.framework.core.util.io.net.http.meta.JaHttpStatus;
import com.jugg.agile.framework.core.util.io.serialize.json.JaJson;
import com.jugg.agile.framework.meta.dto.JaResultDTO;
import com.jugg.agile.framework.meta.exception.JaException;
import com.jugg.agile.spring.boot.webmvc.context.JaWebContextChain;
import com.jugg.agile.spring.boot.webmvc.util.JaServletUtil;
import lombok.SneakyThrows;

import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.nio.charset.StandardCharsets;

/**
 * controller 前置处理, 以防三方包内置controller未拦截到
 *
 * @author chenjian
 * @since 2024年01月05日 19:35:06
 */
public class JaServletFilter implements Filter {

    private static final String logFormatEntry = "url:{},context:{}";

    @Override
    public void init(FilterConfig filterConfig) {
        JaWebContextChain instance = JaWebContextChain.getInstance();
        JaLog.info("JaServletFilter init:{}", instance);
    }

    @Override
    @SneakyThrows
    public void doFilter(ServletRequest servletRequest, ServletResponse servletResponse, FilterChain chain) {
        HttpServletRequest request = (HttpServletRequest) servletRequest;
        HttpServletResponse response = (HttpServletResponse) servletResponse;
        String requestURI = request.getRequestURI();
        try {
            JaWebContextChain.getInstance().inherit(request);
            request.setCharacterEncoding(StandardCharsets.UTF_8.name());
            if (JaStringUtil.isEmpty(response.getHeader(JaMDC.TraceId))) {
                response.addHeader(JaMDC.TraceId, JaMDC.get());
            }
            if (JaServletConfig.isPass(requestURI)) {
                chain.doFilter(servletRequest, servletResponse);
                return;
            }
            JaLog.info(logFormatEntry, requestURI, JaJson.toString(JaCoreContext.getInstance().get()));
            chain.doFilter(servletRequest, servletResponse);
        } catch (Throwable e) {
            handleException(request, response, e);
        } finally {
            JaWebContextChain.getInstance().remove();
        }
    }

    @Override
    public void destroy() {
        JaLog.info("JaServletFilter destroy");
    }

    protected static void handleException(HttpServletRequest request, HttpServletResponse response, Throwable e) {
        JaLog.get().error("http filter catch ex", e);
        e = JaThrowableUtil.getCause(e);
        String code = null;
        String codeMsg = null;
        // TODO 未做国际化
        if (e instanceof JaException) {
            code = ((JaException) e).getCode();
        }
//        if (JaStringUtil.isEmpty(code)) {
//            codeMsg = JaI18nUtil.getMessage("Unknown Error"
//                    , request.getRequestURI()
//                    , JaMDC.get());
//        }
        codeMsg = String.format("Unknown Error:[%s]-[%s]"
                , request.getRequestURI()
                , JaMDC.get());
        JaResultDTO<Object> result = new JaResultDTO<>();
        result.setCode(code);
        result.setCodeMsg(codeMsg);
        response.setStatus(JaHttpStatus.HTTP_VERSION_NOT_SUPPORTED);
        JaServletUtil.printResponse(request, response, result);
    }

}
