package com.jugg.agile.spring.boot.util;

import com.jugg.agile.framework.core.dapper.log.JaLog;
import com.jugg.agile.framework.core.util.JaStringUtil;
import com.jugg.agile.framework.core.util.concurrent.JaThreadLocal;
import com.jugg.agile.framework.meta.adapter.JaI18nAdapter;
import com.jugg.agile.spring.util.JaSpringBeanUtil;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.context.MessageSource;
import org.springframework.context.MessageSourceAware;
import org.springframework.context.NoSuchMessageException;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.context.support.MessageSourceAccessor;

import java.util.Locale;

/**
 * 国际化工具类
 *
 * @author chenjian﻿
 * @since 2023年03月09日 15:57:28
 */
public class JaSpringI18nUtil {

    private JaSpringI18nUtil() {
    }

    public static final String Header = "accept-language";
    public static final String HeaderUp = "Accept-Language";

    private static MessageSourceAccessor accessor;
    public static final Locale DefaultLocale = Locale.SIMPLIFIED_CHINESE;

    public static void setAccessor(MessageSourceAccessor accessor) {
        JaSpringI18nUtil.accessor = accessor;
    }

    public static MessageSourceAccessor getAccessor() {
        return accessor;
    }

    public static final JaThreadLocal<Locale> LocaleThreadLocal = new JaThreadLocal<>(o -> LocaleContextHolder.setLocale((Locale) o));

    public static String getMessage(String code, Object... args) {
        return getMessage(code, LocaleThreadLocal.get(), args);
    }

    public static String getMessage(String code, Locale locale, Object... args) {
        if (null == accessor) {
            initAccessor();
            if (null == accessor) {
                return code;
            }
        }
        if (null == locale) {
            locale = DefaultLocale;
        }
        try {
            return accessor.getMessage(code, args, locale);
        } catch (NoSuchMessageException e) {
            return "NoSuch:" + code;
        } catch (Throwable e) {
            return "ParseError:" + code;
        }
    }

    private static synchronized void initAccessor() {
        accessor = JaSpringBeanUtil.getBean(MessageSourceAccessor.class);
    }

    /**
     * 获取字符串AcceptLanguage构建Locale, 容错tomcat复用请求时不做解析的bug
     */
    public static Locale parseLocales(String acceptLanguage) {
        try {
            if (JaStringUtil.isEmpty(acceptLanguage)) {
                return DefaultLocale;
            }

            String[] language = acceptLanguage.split(";")[0].split(",")[0].split("-");
            if (language.length == 2) {
                return new Locale(language[0], language[1]);
            } else if (language.length == 1) {
                return new Locale(language[0]);
            }
            return DefaultLocale;
        } catch (Throwable e) {
            JaLog.error("parseLocales error:{}", acceptLanguage, e);
            return DefaultLocale;
        }
    }

    public static String getLanguage() {
        Locale locale = LocaleThreadLocal.get();
        return null == locale ? DefaultLocale.toLanguageTag() : locale.toLanguageTag();
    }

    /**
     * 构建MessageSource, 以便后续出现后端国际化需求
     *
     * @author chenjian
     * @see org.springframework.boot.autoconfigure.context.MessageSourceAutoConfiguration
     * @see org.springframework.context.support.ResourceBundleMessageSource#ResourceBundleMessageSource()
     * @since 2023年03月09日 19:38:19
     */
    @Configuration
    public static class JaI18nConfiguration {

        public JaI18nConfiguration() {
            JaI18nAdapter.setI18nHandler(JaSpringI18nUtil::getMessage);
            JaI18nAdapter.setLocaleHandler(JaSpringI18nUtil::getLanguage);
        }

        @Configuration
        @ConditionalOnBean({MessageSource.class})
        public static class MessageSourceAwareConfiguration implements MessageSourceAware {
            @Override
            @SuppressWarnings("all")
            public void setMessageSource(MessageSource messageSource) {
                JaSpringI18nUtil.accessor = new MessageSourceAccessor(messageSource);
            }
        }
    }
}
