package com.jugg.agile.spring.boot.dapper.node;

import com.jugg.agile.framework.core.context.JaMapContextChain;
import com.jugg.agile.framework.core.dapper.aspect.JaDapperAspectPointcut;
import com.jugg.agile.framework.core.dapper.aspect.JaNodeSpanResolver;
import com.jugg.agile.framework.core.dapper.log.JaLog;
import com.jugg.agile.framework.core.dapper.log.JaMDC;
import com.jugg.agile.framework.core.dapper.meta.NodeKind;
import com.jugg.agile.framework.core.dapper.meta.NodeSpan;
import com.jugg.agile.framework.core.util.datastructure.JaCollectionUtil;
import com.jugg.agile.spring.util.JaSpringAopUtil;
import com.jugg.agile.spring.util.JaSpringBeanUtil;
import org.aopalliance.intercept.MethodInvocation;
import org.springframework.aop.framework.ReflectiveMethodInvocation;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.http.HttpRequest;
import org.springframework.http.RequestEntity;
import org.springframework.http.ResponseEntity;
import org.springframework.http.client.ClientHttpRequestExecution;
import org.springframework.http.client.ClientHttpRequestInterceptor;
import org.springframework.http.client.ClientHttpResponse;
import org.springframework.lang.NonNull;
import org.springframework.stereotype.Component;
import org.springframework.web.client.RestTemplate;

import java.io.IOException;
import java.lang.reflect.Type;
import java.net.URI;
import java.util.ArrayList;
import java.util.List;

/**
 * TODO 未彻底监控, 短期解决方案
 * 未来需要重写RestTemplate.doExecute方法的字节码
 * (因需要创建byte[] class文件和需要javax tool.jar环境 且 k8s的busybox环境命令未知, 待验证)
 *
 * @author chenjian
 * @since 2024年05月29日 17:55:31
 */
@Component
@ConditionalOnClass(name = {JaNodeSpanRestTemplate.ProxyClassName})
public class JaNodeSpanRestTemplate implements JaNodeSpanResolver, JaDapperAspectPointcut, ClientHttpRequestInterceptor {

    public static final String ProxyClassName = "org.springframework.web.client.RestTemplate";

    static {
        initNodeKind();
    }

    @SuppressWarnings("rawtypes")
    private static void initNodeKind() {
        NodeKind.Constant.RestTemplate
                .buildRspHandler(o -> {
                    return o instanceof ResponseEntity ? ((ResponseEntity) o).getBody() : o;
                })
                .buildNodeIdHandler(invocation -> {
                    Object[] arguments = invocation.getArguments();
                    if (JaCollectionUtil.isNotEmpty(arguments)) {
                        Object argument = arguments[0];
                        if (argument instanceof String || argument instanceof URI) {
                            return argument.toString();
                        }
                        if (argument instanceof RequestEntity) {
                            return ((RequestEntity) argument).getUrl().toString();
                        }
                    }
                    return invocation.getMethod().hashCode() + "";
                })
                .buildReqArgsHandler(args -> {
                    if (JaCollectionUtil.isEmpty(args)) {
                        return args;
                    }
                    ArrayList<Object> objects = new ArrayList<>(args.length);
                    for (Object arg : args) {
                        if (arg instanceof String
                                || arg instanceof URI
                                || arg instanceof HttpMethod
                                || arg instanceof Type
                                || arg instanceof ParameterizedTypeReference
                        ) {
                            continue;
                        }
                        if (arg instanceof HttpEntity) {
                            objects.add(((HttpEntity) arg).getBody());
                        } else {
                            objects.add(arg);
                        }
                    }
                    return objects.toArray();
                })
                .buildSpanIdThreadLocal();
    }


    @Override
    public NodeSpan getNodeSpan(MethodInvocation invocation) {
        if (invocation instanceof ReflectiveMethodInvocation) {
            Object proxy = ((ReflectiveMethodInvocation) invocation).getProxy();
            if (proxy.toString().startsWith(ProxyClassName)) {
                return get(NodeKind.Constant.RestTemplate);
            }
        }
        return null;
    }

    @Override
    public String getExpression() {
//        return "execution(* org.springframework.web.client.RestTemplate.doExecute(..))";
        // org.springframework.web.client.RestOperations.postForEntity(java.lang.String, java.lang.Object, java.lang.Class<T>, java.lang.Object...)
        return "execution(* org.springframework.web.client.RestOperations.*(..))";
    }


    @NonNull
    @Override
    public ClientHttpResponse intercept(HttpRequest request, @NonNull byte[] body, ClientHttpRequestExecution execution) throws IOException {
        HttpHeaders headers = request.getHeaders();
        JaMapContextChain.getInstance().transmit().forEach(headers::add);
        headers.add(JaMDC.KeyParentSpanId, NodeKind.Constant.RestTemplate.transmitSpanId());
        return execution.execute(request, body);
    }

    @Override
    public void initNodeSpanResolver() {
        // 延迟依赖, 直接注入会导致aop失效
        List<RestTemplate> beanList = JaSpringBeanUtil.getBeanList(RestTemplate.class);
        if (JaCollectionUtil.isNotEmpty(beanList)) {
            beanList.forEach(restTemplate -> JaSpringAopUtil.getSingletonTarget(restTemplate).getInterceptors().add(0, this));
        } else {
            JaLog.warn("JaNodeSpanRestTemplate add interceptor error : restTemplate is null");
        }
    }
}
