package com.jugg.agile.spring.boot.core.config;

import com.jugg.agile.framework.core.config.JaEnvProperty;
import com.jugg.agile.framework.core.config.JaProperty;
import com.jugg.agile.framework.core.dapper.log.JaLog;
import com.jugg.agile.framework.core.meta.function.JaFunctionP;
import com.jugg.agile.framework.core.util.datastructure.JaCollectionUtil;
import com.jugg.agile.spring.util.JaSpringBeanUtil;
import com.jugg.agile.spring.util.JaSpringContextUtil;
import org.springframework.boot.context.properties.ConfigurationPropertiesBindingPostProcessor;
import org.springframework.core.env.AbstractEnvironment;
import org.springframework.core.env.ConfigurableEnvironment;
import org.springframework.core.env.MutablePropertySources;
import org.springframework.core.env.PropertiesPropertySource;
import org.springframework.core.env.PropertySource;

import java.io.FileOutputStream;
import java.util.ArrayList;
import java.util.Map;
import java.util.Properties;
import java.util.TreeMap;

public class JaSpringPropertyProcessor {

    private static final Properties jaProperties = new Properties();

    private JaSpringPropertyProcessor() {
    }

    // org.springframework.boot.context.config.ConfigFileApplicationListener.Loader.load(org.springframework.boot.env.PropertySourceLoader, java.lang.String, org.springframework.boot.context.config.ConfigFileApplicationListener.Profile, org.springframework.boot.context.config.ConfigFileApplicationListener.DocumentFilter, org.springframework.boot.context.config.ConfigFileApplicationListener.DocumentConsumer)
    private static final String LoadPropertySourcePre = "application";

    /**
     * JaProperty加载到Environment
     */
    public static synchronized void setEnvironment(ConfigurableEnvironment environment, String name) {
        MutablePropertySources propertySources = getMutablePropertySources(environment);
        if (null != propertySources) {
            String relativePropertySourceName = null;
            for (PropertySource<?> propertySource : propertySources) {
                if (null != propertySource && propertySource.getName().contains(LoadPropertySourcePre)) {
                    relativePropertySourceName = propertySource.getName();
                    break;
                }
            }

            JaProperty.getPropertyMap().forEach(jaProperties::put);
            if (null == relativePropertySourceName) {
                propertySources.addLast(new PropertiesPropertySource(name, jaProperties));
            } else {
                propertySources.addBefore(relativePropertySourceName, new PropertiesPropertySource(name, jaProperties));
            }
        }
    }

    /**
     * 最终使用配置输出
     */
    public static void outputFinalConfig(ConfigurableEnvironment environment) {
        if (JaEnvProperty.isCD() || Boolean.TRUE.equals(JaProperty.getBoolean("ja.config.output", false))) {
            try (FileOutputStream outputStream = new FileOutputStream("jugg.properties")) {

                Map<String, Object> map = new TreeMap<>();
                MutablePropertySources propertySources = getMutablePropertySources(environment);
                if (null != propertySources) {
                    ArrayList<Map<String, Object>> allPropertySourcesList = new ArrayList<>();
                    propertySources.forEach(propertySource -> invokeMapPropertySource(propertySource, allPropertySourcesList::add));

                    if (JaCollectionUtil.isNotEmpty(allPropertySourcesList)) {
                        for (int i = allPropertySourcesList.size() - 1; i >= 0; i--) {
                            allPropertySourcesList.get(i).forEach((s, o) -> map.put(s, o.toString()));
                        }
                    }
                }

                Properties properties = new Properties();
                properties.putAll(map);
                properties.store(outputStream, null);
            } catch (Throwable e) {
                JaLog.warn("final.properties store error:{}", e);
            }
        }
    }

    /**
     * 刷新配置
     */
    public static <T> T refresh(Class<T> configBeanClass) {
        try {
            JaProperty.getPropertyMap().forEach(jaProperties::put);
            T targetConfigBean = JaSpringBeanUtil.getBean(configBeanClass);
            ConfigurationPropertiesBindingPostProcessor bindingPostProcessor = JaSpringBeanUtil.getBean(ConfigurationPropertiesBindingPostProcessor.class);
            assert bindingPostProcessor != null;
            assert targetConfigBean != null;
            String beanName = JaSpringContextUtil.getApplicationContext().getBeanNamesForType(configBeanClass)[0];
            bindingPostProcessor.postProcessBeforeInitialization(targetConfigBean, beanName);
            return targetConfigBean;
        } catch (Throwable e) {
            JaLog.error("{} config refresh error:{}", configBeanClass.getName(), e.getMessage());
            return null;
        }
    }

    /**
     * 读取Spring配置到JaProperty
     */
    public static synchronized void loadLocalConfig(ConfigurableEnvironment environment) {
        MutablePropertySources propertySources = getMutablePropertySources(environment);
        if (null != propertySources) {
            ArrayList<Map<String, Object>> applicationConfigPropertySourcesList = new ArrayList<>();
            propertySources.forEach(propertySource -> {
                if (null != propertySource && propertySource.getName().contains(LoadPropertySourcePre)) {
                    invokeMapPropertySource(propertySource, applicationConfigPropertySourcesList::add);
                }
            });

            if (JaCollectionUtil.isNotEmpty(applicationConfigPropertySourcesList)) {
                for (int i = applicationConfigPropertySourcesList.size() - 1; i >= 0; i--) {
                    applicationConfigPropertySourcesList.get(i).forEach((s, o) -> JaProperty.put(s, o.toString()));
                }
            }
        }
    }

    public static MutablePropertySources getMutablePropertySources(ConfigurableEnvironment environment) {
        MutablePropertySources propertySources = null;
        if (environment instanceof AbstractEnvironment) {
            propertySources = environment.getPropertySources();
        } else {
            JaLog.get().warn("not support this environment{}", environment.getClass().getName());
        }
        return propertySources;
    }

    @SuppressWarnings({"unchecked"})
    public static void invokeMapPropertySource(PropertySource<?> propertySource, JaFunctionP<Map<String, Object>> functionP) {
        Object source = propertySource.getSource();
        if (source instanceof Map) {
            Map<String, Object> sourceMap = (Map<String, Object>) source;
            if (JaCollectionUtil.isNotEmpty(sourceMap)) {
                functionP.apply(sourceMap);
            }
        } else {
            JaLog.info("unknown propertySource, name:{}", propertySource.getName());
        }
    }
}
