package io.seata.common.util;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonTypeInfo;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.databind.*;
import com.fasterxml.jackson.databind.module.SimpleModule;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import com.fasterxml.jackson.datatype.jsr310.deser.LocalDateTimeDeserializer;
import com.fasterxml.jackson.datatype.jsr310.ser.LocalDateTimeSerializer;

import java.text.SimpleDateFormat;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.TimeZone;

public class JsonUtil {

    private static final ObjectMapper MAPPER = new ObjectMapper();


    static {
        init();
    }

    private static void init() {
        // 空对象不报错
        MAPPER.disable(SerializationFeature.FAIL_ON_EMPTY_BEANS);
        // 忽略未知字段（不报错）
        MAPPER.disable(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES);
        // 空字符串转为 null
        MAPPER.enable(DeserializationFeature.ACCEPT_EMPTY_STRING_AS_NULL_OBJECT);
        // 允许无引号字段名
        MAPPER.configure(JsonParser.Feature.ALLOW_UNQUOTED_FIELD_NAMES, true);
        // 允许单引号
        MAPPER.configure(JsonParser.Feature.ALLOW_SINGLE_QUOTES, true);

        // ===== 日期/时间格式化 =====
        DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");
        // 支持 Java 8 时间 API（LocalDate 等）
        MAPPER.registerModule(new JavaTimeModule()
                .addSerializer(LocalDateTime.class, new LocalDateTimeSerializer(formatter))
                .addDeserializer(LocalDateTime.class, new LocalDateTimeDeserializer(formatter)));
        // 禁用时间戳格式
        MAPPER.disable(SerializationFeature.WRITE_DATES_AS_TIMESTAMPS);
        MAPPER.setDateFormat(new SimpleDateFormat("yyyy-MM-dd HH:mm:ss")); // 自定义日期格式

        // 支持BigDecimal精确解析
        MAPPER.enable(DeserializationFeature.USE_BIG_DECIMAL_FOR_FLOATS);
        // 设置默认时区
        MAPPER.setTimeZone(TimeZone.getTimeZone("Asia/Shanghai"));

        // 启用多态类型处理
//        MAPPER.enableDefaultTyping(ObjectMapper.DefaultTyping.NON_FINAL, JsonTypeInfo.As.PROPERTY);
        // 关闭多态类型处理
        MAPPER.disableDefaultTyping();
        // 忽略transient字段
        MAPPER.enable(MapperFeature.PROPAGATE_TRANSIENT_MARKER);

        // 配置 Jackson 序列化规则（等价于 Fastjson 的 SerializerFeature）
        MAPPER.enable(SerializationFeature.WRITE_EMPTY_JSON_ARRAYS);  // 空数组序列化为 []
        MAPPER.setSerializationInclusion(JsonInclude.Include.ALWAYS); // 覆盖默认的 NON_NULL 行为
    }

    /**
     * 获取jsonMapper
     * @return
     */
    public static ObjectMapper getJson() {
        return MAPPER;
    }
}
