/*
 * Copyright 2017-2025 noear.org and authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.noear.solon.ai.mcp.client;

import org.noear.solon.ai.util.ProxyDesc;
import org.noear.solon.net.http.HttpSslSupplier;
import org.noear.solon.net.http.HttpTimeout;

import java.net.InetSocketAddress;
import java.net.Proxy;
import java.time.Duration;
import java.util.*;

/**
 * Mcp 客户端属性
 *
 * @author noear
 * @since 3.1
 */
public class McpClientProperties {
    /**
     * 客户端名称
     */
    private String name = "Solon-Ai-Mcp-Client";

    /**
     * 客户端版本号
     */
    private String version = "1.0.0";

    /**
     * 通道（传输方式）
     */
    private String channel;

    /**
     * http 接口完整地址
     *
     * @deprecated 3.5 {@link #url}
     */
    @Deprecated
    private String apiUrl;

    /**
     * http 接口完整地址
     */
    private String url;

    /**
     * http 接口密钥
     */
    private String apiKey;

    /**
     * http 请求头信息
     */
    private final Map<String, String> headers = new LinkedHashMap<>();


    /**
     * http 超时
     */
    private HttpTimeout httpTimeout = HttpTimeout.of(10, 60, 60);

    /**
     * http 代理简单描述
     */
    protected ProxyDesc httpProxy;

    /**
     * http 代理实例
     */
    private Proxy httpProxyInstance;

    /**
     * http ssl 提供者
     *
     */
    private HttpSslSupplier httpSsl;

    /**
     * mcp 请求超时
     */
    private Duration requestTimeout = Duration.ofSeconds(20); // Default timeout

    /**
     * mcp 初始化超时
     */
    private Duration initializationTimeout = Duration.ofSeconds(20);

    /**
     * mcp 心跳间隔（辅助自动重连）
     */
    private Duration heartbeatInterval = Duration.ofSeconds(15);

    /**
     * mcp 缓存秒数
     */
    private int cacheSeconds = 30; // Default timeout

    /**
     * 服务端参数（用于 stdio）
     *
     * @deprecated 3.5
     */
    @Deprecated
    private McpServerParameters serverParameters;


    /// ////////////
    /**
     * stdio 命令
     */
    private String command;
    /**
     * stdio 参数
     */
    private List<String> args = new ArrayList<>();
    /**
     * stdio 命令环境变量
     */
    private Map<String, String> env = new HashMap<>();


    public McpClientProperties() {
        //用于序列化
    }

    /**
     * @param apiUrl 接口地址
     */
    public McpClientProperties(String apiUrl) {
        this.apiUrl = apiUrl;
    }

    /// ///////////////////

    public String getName() {
        return name;
    }

    public void setName(String name) {
        this.name = name;
    }

    public String getVersion() {
        return version;
    }

    public void setVersion(String version) {
        this.version = version;
    }

    public String getChannel() {
        return channel;
    }

    public void setChannel(String channel) {
        this.channel = channel;
    }

    /**
     * @deprecated 3.5 {@link #getUrl()}
     * */
    @Deprecated
    public String getApiUrl() {
        return apiUrl;
    }
    /**
     * @deprecated 3.5 {@link #setUrl(String)}
     * */
    @Deprecated
    public void setApiUrl(String apiUrl) {
        this.apiUrl = apiUrl;
    }

    public String getUrl() {
        if(apiUrl != null){
            return apiUrl;
        }

        return url;
    }

    public void setUrl(String url) {
        this.apiUrl = null;
        this.url = url;
    }


    /**
     * @deprecated 3.5 {@link #getHeaders()}
     * */
    @Deprecated
    public String getApiKey() {
        return apiKey;
    }

    /**
     * @deprecated 3.5 {@link #setHeaders(Map)}
     * */
    @Deprecated
    public void setApiKey(String apiKey) {
        this.apiKey = apiKey;
    }

    public Map<String, String> getHeaders() {
        return headers;
    }

    public void setHeaders(Map<String, String> headers) {
        this.headers.putAll(headers);
    }


    public HttpTimeout getHttpTimeout() {
        return httpTimeout;
    }

    public void setHttpTimeout(HttpTimeout httpTimeout) {
        this.httpTimeout = httpTimeout;
    }

    public Proxy getHttpProxy() {
        if (httpProxyInstance == null) {
            if (httpProxy != null) {
                httpProxyInstance = new Proxy(httpProxy.type, new InetSocketAddress(httpProxy.host, httpProxy.port));
            }
        }

        return httpProxyInstance;
    }

    public void setHttpProxy(Proxy httpProxy) {
        this.httpProxyInstance = httpProxy;
        this.httpProxy = null;
    }

    public void setHttpSsl(HttpSslSupplier httpSslSupplier) {
        this.httpSsl = httpSslSupplier;
    }

    public HttpSslSupplier getHttpSsl() {
        return httpSsl;
    }

    public Duration getRequestTimeout() {
        return requestTimeout;
    }

    public void setRequestTimeout(Duration requestTimeout) {
        this.requestTimeout = requestTimeout;
    }

    public Duration getInitializationTimeout() {
        return initializationTimeout;
    }

    public void setInitializationTimeout(Duration initializationTimeout) {
        this.initializationTimeout = initializationTimeout;
    }

    public Duration getHeartbeatInterval() {
        return heartbeatInterval;
    }

    public void setHeartbeatInterval(Duration heartbeatInterval) {
        this.heartbeatInterval = heartbeatInterval;
    }


    public int getCacheSeconds() {
        return cacheSeconds;
    }

    public void setCacheSeconds(int cacheSeconds) {
        this.cacheSeconds = cacheSeconds;
    }

    /**
     * @deprecated 3.5 {@link #getCommand()}
     */
    @Deprecated
    public McpServerParameters getServerParameters() {
        return serverParameters;
    }

    /**
     * @deprecated 3.5 {@link #setCommand(String)}
     */
    @Deprecated
    public void setServerParameters(McpServerParameters serverParameters) {
        this.serverParameters = serverParameters;
    }


    /// ////////////

    public String getCommand() {
        if (serverParameters != null) {
            return serverParameters.getCommand();
        }

        return command;
    }

    public void setCommand(String command) {
        this.serverParameters = null;
        this.command = command;
    }

    public List<String> getArgs() {
        if (serverParameters != null) {
            return serverParameters.getArgs();
        }

        return args;
    }

    public void setArgs(List<String> args) {
        this.serverParameters = null;

        this.args = args;
    }

    public Map<String, String> getEnv() {
        if (serverParameters != null) {
            return serverParameters.getEnv();
        }

        return env;
    }

    public void setEnv(Map<String, String> env) {
        this.serverParameters = null;

        this.env = env;
    }

    @Override
    public String toString() {
        return "McpClientProperties{" +
                "name='" + name + '\'' +
                ", version='" + version + '\'' +
                ", channel='" + channel + '\'' +
                ", apiUrl='" + apiUrl + '\'' +
                ", apiKey='" + apiKey + '\'' +
                ", headers=" + headers +
                ", httpTimeout=" + httpTimeout +
                ", httpProxy=" + getHttpProxy() +
                ", requestTimeout=" + requestTimeout +
                ", initializationTimeout=" + initializationTimeout +
                ", heartbeatInterval=" + heartbeatInterval +
                ", command='" + command + '\'' +
                ", args=" + args +
                ", env=" + env +
                '}';
    }
}